import { T as TransportFactory, C as ClientConfig, a as TransportProtocolName, b as Client, c as Transport, R as RequestOptions, S as SendMessageResult, A as A2AStreamEventData } from '../core-BAzQJfA2.js';
export { d as A2AClient, e as A2AClientOptions, g as AfterArgs, B as BeforeArgs, f as CallInterceptor, l as ClientCallContext, n as ClientCallContextKey, h as ClientCallInput, i as ClientCallResult, m as ContextUpdate, j as ServiceParameters, k as ServiceParametersUpdate, w as withA2AExtensions } from '../core-BAzQJfA2.js';
import { ae as AgentCard, x as MessageSendParams, $ as TaskPushNotificationConfig, a3 as GetTaskPushNotificationConfigParams, a7 as ListTaskPushNotificationConfigParams, a9 as DeleteTaskPushNotificationConfigParams, X as TaskQueryParams, ay as Task, Z as TaskIdParams, j as JSONRPCResponse } from '../extensions-DvruCIzw.js';

interface HttpHeaders {
    [key: string]: string;
}
/**
 * Generic interface for handling authentication for HTTP requests.
 *
 * - For each HTTP request, this handler is called to provide additional headers to the request through
 *   the headers() function.
 * - After the server returns a response, the shouldRetryWithHeaders() function is called.  Usually this
 *   function responds to a 401 or 403 response or JSON-RPC codes, but can respond to any other signal -
 *   that is an implementation detail of the AuthenticationHandler.
 * - If the shouldRetryWithHeaders() function returns new headers, then the request should retried with the provided
 *   revised headers.  These provisional headers may, or may not, be optimistically stored for subsequent requests -
 *   that is an implementation detail of the AuthenticationHandler.
 * - If the request is successful and the onSuccessfulRetry() is defined, then the onSuccessfulRetry() function is
 *   called with the headers that were used to successfully complete the request.  This callback provides an
 *   opportunity to save the headers for subsequent requests if they were not already saved.
 *
 */
interface AuthenticationHandler {
    /**
     * Provides additional HTTP request headers.
     * @returns HTTP headers which may include Authorization if available.
     */
    headers: () => Promise<HttpHeaders>;
    /**
     * For every HTTP response (even 200s) the shouldRetryWithHeaders() method is called.
     * This method is supposed to check if the request needs to be retried and if, yes,
     * return a set of headers. An A2A server might indicate auth failures in its response
     * by JSON-rpc codes, HTTP codes like 401, 403 or headers like WWW-Authenticate.
     *
     * @param req The RequestInit object used to invoke fetch()
     * @param res The fetch Response object
     * @returns If the HTTP request should be retried then returns the HTTP headers to use,
     * 	or returns undefined if no retry should be made.
     */
    shouldRetryWithHeaders: (req: RequestInit, res: Response) => Promise<HttpHeaders | undefined>;
    /**
     * If the last HTTP request using the headers from shouldRetryWithHeaders() was successful, and
     * this function is implemented, then it will be called with the headers provided from
     * shouldRetryWithHeaders().
     *
     * This callback allows transient headers to be saved for subsequent requests only when they
     * are validated by the server.
     */
    onSuccessfulRetry?: (headers: HttpHeaders) => Promise<void>;
}
/**
 * Higher-order function that wraps fetch with authentication handling logic.
 * Returns a new fetch function that automatically handles authentication retries for 401/403 responses.
 *
 * @param fetchImpl The underlying fetch implementation to wrap
 * @param authHandler Authentication handler for managing auth headers and retries
 * @returns A new fetch function with authentication handling capabilities
 *
 * Usage examples:
 * - const authFetch = createAuthHandlingFetch(fetch, authHandler);
 * - const response = await authFetch(url, options);
 * - const response = await authFetch(url); // Direct function call
 */
declare function createAuthenticatingFetchWithRetry(fetchImpl: typeof fetch, authHandler: AuthenticationHandler): typeof fetch;

interface AgentCardResolverOptions {
    path?: string;
    fetchImpl?: typeof fetch;
}
declare class DefaultAgentCardResolver implements AgentCardResolver {
    readonly options?: AgentCardResolverOptions;
    constructor(options?: AgentCardResolverOptions);
    /**
     * Fetches the agent card based on provided base URL and path.
     * Path is selected in the following order:
     * 1) path parameter
     * 2) path from options
     * 3) .well-known/agent-card.json
     */
    resolve(baseUrl: string, path?: string): Promise<AgentCard>;
    private fetchImpl;
}
interface AgentCardResolver {
    /**
     * Fetches the agent card based on provided base URL and path,
     */
    resolve(baseUrl: string, path?: string): Promise<AgentCard>;
}
declare const AgentCardResolver: {
    default: DefaultAgentCardResolver;
};

interface ClientFactoryOptions {
    /**
     * Transport factories to use.
     * Effectively defines transports supported by this client factory.
     */
    transports: TransportFactory[];
    /**
     * Client config to be used for clients created by this factory.
     */
    clientConfig?: ClientConfig;
    /**
     * Transport preferences to override ones defined by the agent card.
     * If no matches are found among preferred transports, agent card values are used next.
     */
    preferredTransports?: TransportProtocolName[];
    /**
     * Used for createFromAgentCardUrl to download agent card.
     */
    cardResolver?: AgentCardResolver;
}
declare const ClientFactoryOptions: {
    /**
     * SDK default options for {@link ClientFactory}.
     */
    default: Readonly<ClientFactoryOptions>;
    /**
     * Creates new options by merging an original and an override object.
     * Transports are merged based on `TransportFactory.protocolName`,
     * interceptors are concatenated, other fields are overriden.
     *
     * @example
     * ```ts
     * const options = ClientFactoryOptions.createFrom(ClientFactoryOptions.default, {
     *  transports: [new MyCustomTransportFactory()], // adds a custom transport
     *  clientConfig: { interceptors: [new MyInterceptor()] }, // adds a custom interceptor
     * });
     * ```
     */
    createFrom(original: ClientFactoryOptions, overrides: Partial<ClientFactoryOptions>): ClientFactoryOptions;
};
declare class ClientFactory {
    readonly options: ClientFactoryOptions;
    private readonly transportsByName;
    private readonly agentCardResolver;
    constructor(options?: ClientFactoryOptions);
    /**
     * Creates a new client from the provided agent card.
     */
    createFromAgentCard(agentCard: AgentCard): Promise<Client>;
    /**
     * Downloads agent card using AgentCardResolver from options
     * and creates a new client from the downloaded card.
     *
     * @example
     * ```ts
     * const factory = new ClientFactory(); // use default options and default {@link AgentCardResolver}.
     * const client1 = await factory.createFromUrl('https://example.com'); // /.well-known/agent-card.json is used by default
     * const client2 = await factory.createFromUrl('https://example.com', '/my-agent-card.json'); // specify custom path
     * const client3 = await factory.createFromUrl('https://example.com/my-agent-card.json', ''); // specify full URL and set path to empty
     * ```
     */
    createFromUrl(baseUrl: string, path?: string): Promise<Client>;
}

declare class TaskNotFoundError extends Error {
    constructor(message?: string);
}
declare class TaskNotCancelableError extends Error {
    constructor(message?: string);
}
declare class PushNotificationNotSupportedError extends Error {
    constructor(message?: string);
}
declare class UnsupportedOperationError extends Error {
    constructor(message?: string);
}
declare class ContentTypeNotSupportedError extends Error {
    constructor(message?: string);
}
declare class InvalidAgentResponseError extends Error {
    constructor(message?: string);
}
declare class AuthenticatedExtendedCardNotConfiguredError extends Error {
    constructor(message?: string);
}

interface JsonRpcTransportOptions {
    endpoint: string;
    fetchImpl?: typeof fetch;
}
declare class JsonRpcTransport implements Transport {
    private readonly customFetchImpl?;
    private readonly endpoint;
    private requestIdCounter;
    constructor(options: JsonRpcTransportOptions);
    getExtendedAgentCard(options?: RequestOptions, idOverride?: number): Promise<AgentCard>;
    sendMessage(params: MessageSendParams, options?: RequestOptions, idOverride?: number): Promise<SendMessageResult>;
    sendMessageStream(params: MessageSendParams, options?: RequestOptions): AsyncGenerator<A2AStreamEventData, void, undefined>;
    setTaskPushNotificationConfig(params: TaskPushNotificationConfig, options?: RequestOptions, idOverride?: number): Promise<TaskPushNotificationConfig>;
    getTaskPushNotificationConfig(params: GetTaskPushNotificationConfigParams, options?: RequestOptions, idOverride?: number): Promise<TaskPushNotificationConfig>;
    listTaskPushNotificationConfig(params: ListTaskPushNotificationConfigParams, options?: RequestOptions, idOverride?: number): Promise<TaskPushNotificationConfig[]>;
    deleteTaskPushNotificationConfig(params: DeleteTaskPushNotificationConfigParams, options?: RequestOptions, idOverride?: number): Promise<void>;
    getTask(params: TaskQueryParams, options?: RequestOptions, idOverride?: number): Promise<Task>;
    cancelTask(params: TaskIdParams, options?: RequestOptions, idOverride?: number): Promise<Task>;
    resubscribeTask(params: TaskIdParams, options?: RequestOptions): AsyncGenerator<A2AStreamEventData, void, undefined>;
    callExtensionMethod<TExtensionParams, TExtensionResponse extends JSONRPCResponse>(method: string, params: TExtensionParams, idOverride: number, options?: RequestOptions): Promise<TExtensionResponse>;
    private _fetch;
    private _sendRpcRequest;
    private _fetchRpc;
    private _sendStreamingRequest;
    private _processSseEventData;
    private static mapToError;
}
declare class JsonRpcTransportFactoryOptions {
    fetchImpl?: typeof fetch;
}
declare class JsonRpcTransportFactory implements TransportFactory {
    private readonly options?;
    static readonly name: TransportProtocolName;
    constructor(options?: JsonRpcTransportFactoryOptions);
    get protocolName(): string;
    create(url: string, _agentCard: AgentCard): Promise<Transport>;
}

interface RestTransportOptions {
    endpoint: string;
    fetchImpl?: typeof fetch;
}
declare class RestTransport implements Transport {
    private readonly customFetchImpl?;
    private readonly endpoint;
    constructor(options: RestTransportOptions);
    getExtendedAgentCard(options?: RequestOptions): Promise<AgentCard>;
    sendMessage(params: MessageSendParams, options?: RequestOptions): Promise<SendMessageResult>;
    sendMessageStream(params: MessageSendParams, options?: RequestOptions): AsyncGenerator<A2AStreamEventData, void, undefined>;
    setTaskPushNotificationConfig(params: TaskPushNotificationConfig, options?: RequestOptions): Promise<TaskPushNotificationConfig>;
    getTaskPushNotificationConfig(params: GetTaskPushNotificationConfigParams, options?: RequestOptions): Promise<TaskPushNotificationConfig>;
    listTaskPushNotificationConfig(params: ListTaskPushNotificationConfigParams, options?: RequestOptions): Promise<TaskPushNotificationConfig[]>;
    deleteTaskPushNotificationConfig(params: DeleteTaskPushNotificationConfigParams, options?: RequestOptions): Promise<void>;
    getTask(params: TaskQueryParams, options?: RequestOptions): Promise<Task>;
    cancelTask(params: TaskIdParams, options?: RequestOptions): Promise<Task>;
    resubscribeTask(params: TaskIdParams, options?: RequestOptions): AsyncGenerator<A2AStreamEventData, void, undefined>;
    private _fetch;
    private _buildHeaders;
    private _sendRequest;
    private _handleErrorResponse;
    private _sendStreamingRequest;
    private _processSseEventData;
    private static mapToError;
}
interface RestTransportFactoryOptions {
    fetchImpl?: typeof fetch;
}
declare class RestTransportFactory implements TransportFactory {
    private readonly options?;
    static readonly name: TransportProtocolName;
    constructor(options?: RestTransportFactoryOptions);
    get protocolName(): string;
    create(url: string, _agentCard: AgentCard): Promise<Transport>;
}

export { AgentCardResolver, type AgentCardResolverOptions, AuthenticatedExtendedCardNotConfiguredError, type AuthenticationHandler, Client, ClientConfig, ClientFactory, ClientFactoryOptions, ContentTypeNotSupportedError, DefaultAgentCardResolver, type HttpHeaders, InvalidAgentResponseError, JsonRpcTransport, JsonRpcTransportFactory, type JsonRpcTransportOptions, PushNotificationNotSupportedError, RequestOptions, RestTransport, RestTransportFactory, type RestTransportOptions, TaskNotCancelableError, TaskNotFoundError, Transport, TransportFactory, UnsupportedOperationError, createAuthenticatingFetchWithRetry };
