/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef UTLL_H
#define UTLL_H 

/****************************************************************************** 
 *                      Linked List macros                 (Troy Hanson, 2003) 
 * The utility of the LL macros is that they work with any data 
 * structure having a "next" pointer within it. The LL macros allow
 * you to string together those structures into a linked list.
 *
 * LL_ADD - add an element to a list
 * LL_DEL - delete an element from a list
 * LL_FIND - find an element (for structures with a 'char *name' member)
 * LL_FIND_BY_MEMBER - find an element given a non-string member and value 
 * LL_FIND_BY_MEMBER_STR - find an element given a string member and value 
 *
 * USAGE: Suppose you have a structure like this,
 *
 *    struct mystruct {
 *       char *name;
 *       struct mystruct *next;
 *    }
 *
 * Your program declares the head of the list like:
 *     struct mystruct *list_head = NULL;
 *
 * The first parameter of each macro is the head of the list. E.g. list_head.
 * Note that this has to be an lvalue (assignable pointer) like shown above.
 *
 * The second parameter has the same type as the first, e.g. struct mystruct*.
 * Its a "scratch" variable for LL_ADD and LL_DEL. For the LL_FIND* macros it
 * stores the pointer to the resulting found object (or NULL). 
 *
 * For LL_ADD and LL_DEL, the third parameter is a pointer to the struct
 * being added or deleted. For LL_FIND, its the string to find. For 
 * LL_FIND_BY_MEMBER and LL_FIND_BY_MEMBER_STR, the third parameter is the
 * name of struct member and the fourth parameter is the value to look for.
 *
 * EXAMPLE:
 *
 *    struct mystruct *list_head = NULL, *user, *user2, *tmp;
 * 
 *    user->name = "bob";
 *    user2->name = "joe";
 *    LL_ADD(list_head, tmp, user);
 *    LL_ADD(list_head, tmp, user2);
 *    LL_FIND(list_head, tmp, "bob");
 *    if (tmp) printf("found the user with name %s\n", tmp->name);
 *    else printf("user not found\n");
 ******************************************************************************/

#define LL_ADD(head,tmp,add) tmp = head; if (tmp) {while ((tmp)->next) tmp=(tmp)->next; (tmp)->next=add;} else head=add;
#define LL_DEL(head,tmp,del) tmp = head; if (tmp == del) { head=(tmp)->next; } else { while ((tmp)->next && (tmp)->next != del) tmp=(tmp)->next; if ((tmp)->next) (tmp)->next=(del)->next; }
#define LL_FIND(head,tmp,findname) tmp=head; while (tmp) { if (strcmp((tmp)->name,(findname))) tmp=(tmp)->next; else break; }
#define LL_FIND_BY_MEMBER(head,tmp,member,findval) tmp=head; while (tmp) { if ((tmp)->member != (findval))  tmp=(tmp)->next; else break; }
#define LL_FIND_BY_MEMBER_STR(head,tmp,member,findval) tmp=head; while (tmp) { if (strcmp((tmp)->member,(findval)))  tmp=(tmp)->next; else break; }

#endif /* UTLL_H */
