//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================
#ifndef vtk_m_ClipTables_h
#define vtk_m_ClipTables_h

#include <vtkm/CellShape.h>
#include <vtkm/Types.h>

namespace vtkm
{
namespace worklet
{
namespace internal
{
namespace ClipTablesInformation
{
// Points of original cell (up to 8, for the hex)
// Note: we assume P0 is zero in several places.
// Note: we assume these values are contiguous and monotonic.
static constexpr vtkm::UInt8 P0 = 0;
static constexpr vtkm::UInt8 P1 = 1;
static constexpr vtkm::UInt8 P2 = 2;
static constexpr vtkm::UInt8 P3 = 3;
static constexpr vtkm::UInt8 P4 = 4;
static constexpr vtkm::UInt8 P5 = 5;
static constexpr vtkm::UInt8 P6 = 6;
static constexpr vtkm::UInt8 P7 = 7;

// Edges of original cell (up to 12, for the hex)
// Note: we assume these values are contiguous and monotonic.
static constexpr vtkm::UInt8 E00 = 8;
static constexpr vtkm::UInt8 E01 = 9;
static constexpr vtkm::UInt8 E02 = 10;
static constexpr vtkm::UInt8 E03 = 11;
static constexpr vtkm::UInt8 E04 = 12;
static constexpr vtkm::UInt8 E05 = 13;
static constexpr vtkm::UInt8 E06 = 14;
static constexpr vtkm::UInt8 E07 = 15;
static constexpr vtkm::UInt8 E08 = 16;
static constexpr vtkm::UInt8 E09 = 17;
static constexpr vtkm::UInt8 E10 = 18;
static constexpr vtkm::UInt8 E11 = 19;

// New interpolated point (ST_PNT outputs)
static constexpr vtkm::UInt8 N0 = 20;

// Shape types
static constexpr vtkm::UInt8 ST_VTX = vtkm::CELL_SHAPE_VERTEX;
static constexpr vtkm::UInt8 ST_LIN = vtkm::CELL_SHAPE_LINE;
static constexpr vtkm::UInt8 ST_TRI = vtkm::CELL_SHAPE_TRIANGLE;
static constexpr vtkm::UInt8 ST_QUA = vtkm::CELL_SHAPE_QUAD;
static constexpr vtkm::UInt8 ST_TET = vtkm::CELL_SHAPE_TETRA;
static constexpr vtkm::UInt8 ST_HEX = vtkm::CELL_SHAPE_HEXAHEDRON;
static constexpr vtkm::UInt8 ST_PYR = vtkm::CELL_SHAPE_PYRAMID;
static constexpr vtkm::UInt8 ST_WDG = vtkm::CELL_SHAPE_WEDGE;
// This is a special shape type for a centroid point
static constexpr vtkm::UInt8 ST_PNT = vtkm::CELL_SHAPE_EMPTY;

static constexpr vtkm::UInt8 MAX_CELL_INDICES = 8; // Hexahedron
static constexpr vtkm::UInt8 MAX_CELL_EDGES = 12;  // Hexahedron
static constexpr vtkm::UInt16 CELL_EDGES_SIZE = vtkm::NUMBER_OF_CELL_SHAPES * MAX_CELL_EDGES * 2;
}

class ClipTablesBase
{
protected:
  VTKM_EXEC inline static vtkm::Int32 GetCellClipTableIndex(vtkm::UInt8 shape)
  {
    // index into ClipTablesIndices for each shape
    VTKM_STATIC_CONSTEXPR_ARRAY vtkm::Int32 CellIndexLookup[vtkm::NUMBER_OF_CELL_SHAPES] = {
      -1,  //  0 = vtkm::CELL_SHAPE_EMPTY_CELL
      0,   //  1 = vtkm::CELL_SHAPE_VERTEX
      -1,  //  2 = vtkm::CELL_SHAPE_POLY_VERTEX
      2,   //  3 = vtkm::CELL_SHAPE_LINE
      -1,  //  4 = vtkm::CELL_SHAPE_POLY_LINE
      6,   //  5 = vtkm::CELL_SHAPE_TRIANGLE
      -1,  //  6 = vtkm::CELL_SHAPE_TRIANGLE_STRIP
      -1,  //  7 = vtkm::CELL_SHAPE_POLYGON
      -1,  //  8 = vtkm::CELL_SHAPE_PIXEL
      14,  //  9 = vtkm::CELL_SHAPE_QUAD
      30,  // 10 = vtkm::CELL_SHAPE_TETRA
      -1,  // 11 = vtkm::CELL_SHAPE_VOXEL
      46,  // 12 = vtkm::CELL_SHAPE_HEXAHEDRON
      302, // 13 = vtkm::CELL_SHAPE_WEDGE
      366  // 14 = vtkm::CELL_SHAPE_PYRAMID
    };
    return CellIndexLookup[shape];
  }

  VTKM_EXEC inline static vtkm::UInt8 GetMaxCellCase(vtkm::IdComponent numberOfPoints)
  {
    using namespace ClipTablesInformation;
    // index into ClipTablesIndices for each shape
    VTKM_STATIC_CONSTEXPR_ARRAY vtkm::UInt8 CellMaxCase[MAX_CELL_INDICES + 1] = { 0,  1,   3,
                                                                                  7,  15,  31,
                                                                                  63, 127, 255 };
    return CellMaxCase[numberOfPoints];
  }

public:
  using EdgeVec = vtkm::Vec<vtkm::UInt8, 2>;

  VTKM_EXEC inline static EdgeVec GetEdge(vtkm::UInt8 shape, vtkm::Id edgeId)
  {
    using namespace ClipTablesInformation;

    static constexpr uint8_t X = 255;
    VTKM_STATIC_CONSTEXPR_ARRAY vtkm::UInt8 CellEdges[CELL_EDGES_SIZE] = {
      X, X, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_EMPTY_CELL
      X, X, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_VERTEX
      X, X, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_POLY_VERTEX
      0, 1, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_LINE
      X, X, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_POLY_LINE
      0, 1, 1, 2, 2, 0, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_TRIANGLE
      X, X, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_TRIANGLE_STRIP
      X, X, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_POLYGON
      X, X, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_PIXEL
      0, 1, 1, 2, 3, 2, 0, 3, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_QUAD
      0, 1, 1, 2, 2, 0, 0, 3, 1, 3, 2, 3,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_TETRA
      X, X, X, X, X, X, X, X, X, X, X, X,
      X, X, X, X, X, X, X, X, X, X, X, X, // vtkm::CELL_SHAPE_VOXEL
      0, 1, 1, 2, 3, 2, 0, 3, 4, 5, 5, 6,
      7, 6, 4, 7, 0, 4, 1, 5, 3, 7, 2, 6, // vtkm::CELL_SHAPE_HEXAHEDRON
      0, 1, 1, 2, 2, 0, 3, 4, 4, 5, 5, 3,
      0, 3, 1, 4, 2, 5, X, X, X, X, X, X, // vtkm::CELL_SHAPE_WEDGE
      0, 1, 1, 2, 2, 3, 3, 0, 0, 4, 1, 4,
      2, 4, 3, 4, X, X, X, X, X, X, X, X // vtkm::CELL_SHAPE_PYRAMID
    };

    const vtkm::Id index = ((shape * MAX_CELL_EDGES) + edgeId) * 2;
    return { CellEdges[index], CellEdges[index + 1] };
  }

protected:
  ClipTablesBase() = default;
};

// Primary template declaration
template <bool Invert>
class ClipTables;

// Specialization for false
template <>
class ClipTables</*Invert=*/false> : public ClipTablesBase
{
public:
  ClipTables()
    : ClipTablesBase()
  {
  }

  /**
   * Get the value at the given index.
   */
  VTKM_EXEC inline static vtkm::UInt8 ValueAt(vtkm::Id idx)
  {
    using namespace ClipTablesInformation;
    // clang-format off
    // table format of an example case
    // number of shapes
    // {{shape-type, nverts, {point1, point2...}}, ...}, \n

    // clip table for all cases of each shape
    VTKM_STATIC_CONSTEXPR_ARRAY vtkm::UInt8 ClipTablesData[] = {
      // vtkm::CELL_SHAPE_VERTEX
      /* case 0 */ 0,
      /* case 1 */ 1,
      ST_VTX, 1, P0,
      // vtkm::CELL_SHAPE_LINE
      /* case 0 */ 0,
      /* case 1 */ 1,
      ST_LIN, 2, P0, E00,
      /* case 2 */ 1,
      ST_LIN, 2, P1, E00,
      /* case 3 */ 1,
      ST_LIN, 2, P0, P1,
      // vtkm::CELL_SHAPE_TRIANGLE
      /* case 0 */ 0,
      /* case 1 */ 1,
      ST_TRI, 3, P0, E00, E02,
      /* case 2 */ 1,
      ST_TRI, 3, P1, E01, E00,
      /* case 3 */ 1,
      ST_QUA, 4, P0, P1, E01, E02,
      /* case 4 */ 1,
      ST_TRI, 3, P2, E02, E01,
      /* case 5 */ 1,
      ST_QUA, 4, P2, P0, E00, E01,
      /* case 6 */ 1,
      ST_QUA, 4, P1, P2, E02, E00,
      /* case 7 */ 1,
      ST_TRI, 3, P0, P1, P2,
      // vtkm::CELL_SHAPE_QUAD
      /* case 0 */ 0,
      /* case 1 */ 1,
      ST_TRI, 3, P0, E00, E03,
      /* case 2 */ 1,
      ST_TRI, 3, P1, E01, E00,
      /* case 3 */ 1,
      ST_QUA, 4, P0, P1, E01, E03,
      /* case 4 */ 1,
      ST_TRI, 3, P2, E02, E01,
      /* case 5 */ 2,
      ST_QUA, 4, P2, P0, E00, E01,
      ST_QUA, 4, P0, P2, E02, E03,
      /* case 6 */ 1,
      ST_QUA, 4, P1, P2, E02, E00,
      /* case 7 */ 2,
      ST_QUA, 4, P0, P2, E02, E03,
      ST_TRI, 3, P1, P2, P0,
      /* case 8 */ 1,
      ST_TRI, 3, P3, E03, E02,
      /* case 9 */ 1,
      ST_QUA, 4, P3, P0, E00, E02,
      /* case 10 */ 2,
      ST_QUA, 4, P3, P1, E01, E02,
      ST_QUA, 4, P1, P3, E03, E00,
      /* case 11 */ 2,
      ST_QUA, 4, P3, P1, E01, E02,
      ST_TRI, 3, P0, P1, P3,
      /* case 12 */ 1,
      ST_QUA, 4, P2, P3, E03, E01,
      /* case 13 */ 2,
      ST_QUA, 4, P2, P0, E00, E01,
      ST_TRI, 3, P3, P0, P2,
      /* case 14 */ 2,
      ST_QUA, 4, P1, P3, E03, E00,
      ST_TRI, 3, P2, P3, P1,
      /* case 15 */ 1,
      ST_QUA, 4, P0, P1, P2, P3,
      // vtkm::CELL_SHAPE_TETRA
      /* case 0 */ 0,
      /* case 1 */ 1,
      ST_TET, 4, P0, E00, E02, E03,
      /* case 2 */ 1,
      ST_TET, 4, P1, E01, E00, E04,
      /* case 3 */ 1,
      ST_WDG, 6, P0, E03, E02, P1, E04, E01,
      /* case 4 */ 1,
      ST_TET, 4, P2, E02, E01, E05,
      /* case 5 */ 1,
      ST_WDG, 6, P2, E05, E01, P0, E03, E00,
      /* case 6 */ 1,
      ST_WDG, 6, P1, E04, E00, P2, E05, E02,
      /* case 7 */ 1,
      ST_WDG, 6, E03, E04, E05, P0, P1, P2,
      /* case 8 */ 1,
      ST_TET, 4, P3, E04, E03, E05,
      /* case 9 */ 1,
      ST_WDG, 6, P0, E02, E00, P3, E05, E04,
      /* case 10 */ 1,
      ST_WDG, 6, P3, E05, E03, P1, E01, E00,
      /* case 11 */ 1,
      ST_WDG, 6, P0, P1, P3, E02, E01, E05,
      /* case 12 */ 1,
      ST_WDG, 6, P2, E01, E02, P3, E04, E03,
      /* case 13 */ 1,
      ST_WDG, 6, E00, E01, E04, P0, P2, P3,
      /* case 14 */ 1,
      ST_WDG, 6, P1, P2, P3, E00, E02, E03,
      /* case 15 */ 1,
      ST_TET, 4, P0, P1, P2, P3,
      // vtkm::CELL_SHAPE_HEXAHEDRON
      /* case 0 */ 0,
      /* case 1 */ 1,
      ST_TET, 4, P0, E00, E03, E08,
      /* case 2 */ 1,
      ST_TET, 4, P1, E00, E09, E01,
      /* case 3 */ 1,
      ST_WDG, 6, P1, E01, E09, P0, E03, E08,
      /* case 4 */ 1,
      ST_TET, 4, P2, E01, E11, E02,
      /* case 5 */ 7,
      ST_PNT, 2, E08, E11,
      ST_PYR, 5, P0, P2, E02, E03, N0,
      ST_PYR, 5, E00, E01, P2, P0, N0,
      ST_TET, 4, E01, E11, P2, N0,
      ST_TET, 4, P2, E11, E02, N0,
      ST_TET, 4, E00, N0, P0, E08,
      ST_TET, 4, E03, E08, P0, N0,
      /* case 6 */ 1,
      ST_WDG, 6, P2, E02, E11, P1, E00, E09,
      /* case 7 */ 7,
      ST_PNT, 5, E08, E09, E03, E02, E11,
      ST_PYR, 5, E09, E11, P2, P1, N0,
      ST_PYR, 5, E08, E09, P1, P0, N0,
      ST_TET, 4, E03, E08, P0, N0,
      ST_TET, 4, P0, P1, P2, N0,
      ST_PYR, 5, E03, P0, P2, E02, N0,
      ST_TET, 4, P2, E11, E02, N0,
      /* case 8 */ 1,
      ST_TET, 4, P3, E03, E02, E10,
      /* case 9 */ 1,
      ST_WDG, 6, P3, E10, E02, P0, E08, E00,
      /* case 10 */ 7,
      ST_PNT, 2, E10, E09,
      ST_PYR, 5, P3, E03, E00, P1, N0,
      ST_PYR, 5, E02, P3, P1, E01, N0,
      ST_TET, 4, E01, P1, E09, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_TET, 4, E02, P3, N0, E10,
      ST_TET, 4, E03, P3, E10, N0,
      /* case 11 */ 7,
      ST_PNT, 5, E09, E08, E01, E02, E10,
      ST_PYR, 5, E08, P0, P3, E10, N0,
      ST_PYR, 5, E09, P1, P0, E08, N0,
      ST_TET, 4, E01, P1, E09, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_PYR, 5, E01, E02, P3, P1, N0,
      ST_TET, 4, P3, E02, E10, N0,
      /* case 12 */ 1,
      ST_WDG, 6, P3, E03, E10, P2, E01, E11,
      /* case 13 */ 7,
      ST_PNT, 5, E08, E10, E00, E01, E11,
      ST_PYR, 5, E10, P3, P2, E11, N0,
      ST_PYR, 5, E08, P0, P3, E10, N0,
      ST_TET, 4, E00, P0, E08, N0,
      ST_TET, 4, P0, P2, P3, N0,
      ST_PYR, 5, E00, E01, P2, P0, N0,
      ST_TET, 4, P2, E01, E11, N0,
      /* case 14 */ 7,
      ST_PNT, 5, E10, E11, E03, E00, E09,
      ST_PYR, 5, E11, P2, P1, E09, N0,
      ST_PYR, 5, E10, P3, P2, E11, N0,
      ST_TET, 4, E03, P3, E10, N0,
      ST_TET, 4, P3, P1, P2, N0,
      ST_PYR, 5, E03, E00, P1, P3, N0,
      ST_TET, 4, P1, E00, E09, N0,
      /* case 15 */ 1,
      ST_HEX, 8, P0, P1, P2, P3, E08, E09, E11, E10,
      /* case 16 */ 1,
      ST_TET, 4, P4, E04, E08, E07,
      /* case 17 */ 1,
      ST_WDG, 6, P0, E03, E00, P4, E07, E04,
      /* case 18 */ 7,
      ST_PNT, 2, E07, E01,
      ST_PYR, 5, P4, P1, E00, E08, N0,
      ST_PYR, 5, E04, E09, P1, P4, N0,
      ST_TET, 4, E09, E01, P1, N0,
      ST_TET, 4, P1, E01, E00, N0,
      ST_TET, 4, E04, N0, P4, E07,
      ST_TET, 4, E08, E07, P4, N0,
      /* case 19 */ 7,
      ST_PNT, 5, E01, E03, E09, E04, E07,
      ST_PYR, 5, E03, E07, P4, P0, N0,
      ST_PYR, 5, E01, E03, P0, P1, N0,
      ST_TET, 4, E09, E01, P1, N0,
      ST_TET, 4, P1, P0, P4, N0,
      ST_PYR, 5, E09, P1, P4, E04, N0,
      ST_TET, 4, P4, E07, E04, N0,
      /* case 20 */ 2,
      ST_TET, 4, P4, E04, E08, E07,
      ST_TET, 4, P2, E02, E01, E11,
      /* case 21 */ 8,
      ST_PNT, 4, E04, E07, E11, E11,
      ST_PYR, 5, P2, E02, E03, P0, N0,
      ST_PYR, 5, E00, E01, P2, P0, N0,
      ST_TET, 4, P2, E11, E02, N0,
      ST_TET, 4, E01, E11, P2, N0,
      ST_PYR, 5, E03, E07, P4, P0, N0,
      ST_PYR, 5, P0, P4, E04, E00, N0,
      ST_TET, 4, E04, P4, E07, N0,
      /* case 22 */ 8,
      ST_PNT, 4, E11, E02, E07, E07,
      ST_PYR, 5, P4, P1, E00, E08, N0,
      ST_PYR, 5, E09, P1, P4, E04, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_TET, 4, E04, P4, E07, N0,
      ST_PYR, 5, E00, P1, P2, E02, N0,
      ST_PYR, 5, P1, E09, E11, P2, N0,
      ST_TET, 4, E11, E02, P2, N0,
      /* case 23 */ 9,
      ST_PNT, 6, E03, E02, E11, E09, E04, E07,
      ST_TET, 4, P0, P1, P2, N0,
      ST_PYR, 5, E03, P0, P2, E02, N0,
      ST_PYR, 5, E03, E07, P4, P0, N0,
      ST_TET, 4, P4, P1, P0, N0,
      ST_TET, 4, P4, E07, E04, N0,
      ST_PYR, 5, P4, E04, E09, P1, N0,
      ST_PYR, 5, E09, E11, P2, P1, N0,
      ST_TET, 4, E11, E02, P2, N0,
      /* case 24 */ 7,
      ST_PNT, 2, E02, E04,
      ST_PYR, 5, P3, P4, E08, E03, N0,
      ST_PYR, 5, E10, E07, P4, P3, N0,
      ST_TET, 4, E07, E04, P4, N0,
      ST_TET, 4, P4, E04, E08, N0,
      ST_TET, 4, E10, N0, P3, E02,
      ST_TET, 4, E03, E02, P3, N0,
      /* case 25 */ 7,
      ST_PNT, 5, E04, E00, E07, E10, E02,
      ST_PYR, 5, E00, E02, P3, P0, N0,
      ST_PYR, 5, E04, E00, P0, P4, N0,
      ST_TET, 4, E07, E04, P4, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_PYR, 5, E07, P4, P3, E10, N0,
      ST_TET, 4, P3, E02, E10, N0,
      /* case 26 */ 5,
      ST_WDG, 6, E03, E00, E08, P3, P1, P4,
      ST_PYR, 5, P3, E10, E07, P4, E02,
      ST_PYR, 5, E04, E09, P1, P4, E01,
      ST_PYR, 5, E02, P3, P1, E01, P4,
      ST_PYR, 5, E02, E01, E04, E07, P4,
      /* case 27 */ 5,
      ST_TET, 4, P0, P1, P3, P4,
      ST_PYR, 5, E07, P4, P3, E10, E02,
      ST_PYR, 5, E04, E09, P1, P4, E01,
      ST_PYR, 5, P3, P1, E01, E02, P4,
      ST_PYR, 5, E07, E02, E01, E04, P4,
      /* case 28 */ 8,
      ST_PNT, 4, E11, E01, E04, E04,
      ST_PYR, 5, P4, E08, E03, P3, N0,
      ST_PYR, 5, E10, E07, P4, P3, N0,
      ST_TET, 4, P4, E04, E08, N0,
      ST_TET, 4, E07, E04, P4, N0,
      ST_PYR, 5, E03, E01, P2, P3, N0,
      ST_PYR, 5, P3, P2, E11, E10, N0,
      ST_TET, 4, E11, P2, E01, N0,
      /* case 29 */ 9,
      ST_PNT, 6, E00, E01, E11, E10, E07, E04,
      ST_TET, 4, P0, P2, P3, N0,
      ST_PYR, 5, E00, E01, P2, P0, N0,
      ST_PYR, 5, E00, P0, P4, E04, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_TET, 4, P4, E07, E04, N0,
      ST_PYR, 5, P4, P3, E10, E07, N0,
      ST_PYR, 5, E10, P3, P2, E11, N0,
      ST_TET, 4, E11, P2, E01, N0,
      /* case 30 */ 9,
      ST_PNT, 5, E11, E09, E10, E07, E04,
      ST_WDG, 6, E03, E00, E08, P3, P1, P4,
      ST_TET, 4, P1, P3, P4, N0,
      ST_PYR, 5, P3, E10, E07, P4, N0,
      ST_TET, 4, P2, P3, P1, N0,
      ST_PYR, 5, E09, E11, P2, P1, N0,
      ST_PYR, 5, E11, E10, P3, P2, N0,
      ST_PYR, 5, P4, E04, E09, P1, N0,
      ST_TET, 4, E07, E04, P4, N0,
      /* case 31 */ 9,
      ST_PNT, 5, E09, E11, E10, E04, E07,
      ST_PYR, 5, P0, P1, P2, P3, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_TET, 4, P4, P1, P0, N0,
      ST_PYR, 5, P4, E04, E09, P1, N0,
      ST_PYR, 5, E09, E11, P2, P1, N0,
      ST_PYR, 5, E11, E10, P3, P2, N0,
      ST_PYR, 5, E10, E07, P4, P3, N0,
      ST_TET, 4, E04, P4, E07, N0,
      /* case 32 */ 1,
      ST_TET, 4, P5, E05, E09, E04,
      /* case 33 */ 7,
      ST_PNT, 2, E03, E05,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_PYR, 5, E00, P0, P5, E09, N0,
      ST_TET, 4, E09, P5, E05, N0,
      ST_TET, 4, P5, E04, E05, N0,
      ST_TET, 4, E00, P0, N0, E03,
      ST_TET, 4, E08, P0, E03, N0,
      /* case 34 */ 1,
      ST_WDG, 6, P1, E00, E01, P5, E04, E05,
      /* case 35 */ 7,
      ST_PNT, 5, E03, E01, E08, E04, E05,
      ST_PYR, 5, E01, P1, P5, E05, N0,
      ST_PYR, 5, E03, P0, P1, E01, N0,
      ST_TET, 4, E08, P0, E03, N0,
      ST_TET, 4, P0, P5, P1, N0,
      ST_PYR, 5, E08, E04, P5, P0, N0,
      ST_TET, 4, P5, E04, E05, N0,
      /* case 36 */ 7,
      ST_PNT, 2, E02, E04,
      ST_PYR, 5, P2, E01, E09, P5, N0,
      ST_PYR, 5, E11, P2, P5, E05, N0,
      ST_TET, 4, E05, P5, E04, N0,
      ST_TET, 4, P5, E09, E04, N0,
      ST_TET, 4, E11, P2, N0, E02,
      ST_TET, 4, E01, P2, E02, N0,
      /* case 37 */ 5,
      ST_WDG, 6, P0, P5, P2, E00, E09, E01,
      ST_PYR, 5, P0, P2, E02, E03, E08,
      ST_PYR, 5, E11, P2, P5, E05, E04,
      ST_PYR, 5, E08, E04, P5, P0, P2,
      ST_PYR, 5, E08, E02, E11, E04, P2,
      /* case 38 */ 7,
      ST_PNT, 5, E04, E00, E05, E11, E02,
      ST_PYR, 5, E00, P1, P2, E02, N0,
      ST_PYR, 5, E04, P5, P1, E00, N0,
      ST_TET, 4, E05, P5, E04, N0,
      ST_TET, 4, P5, P2, P1, N0,
      ST_PYR, 5, E05, E11, P2, P5, N0,
      ST_TET, 4, P2, E11, E02, N0,
      /* case 39 */ 5,
      ST_TET, 4, P1, P0, P5, P2,
      ST_PYR, 5, E02, E03, P0, P2, E08,
      ST_PYR, 5, E11, P2, P5, E05, E04,
      ST_PYR, 5, P0, E08, E04, P5, P2,
      ST_PYR, 5, E02, E11, E04, E08, P2,
      /* case 40 */ 2,
      ST_TET, 4, P3, E03, E02, E10,
      ST_TET, 4, P5, E09, E04, E05,
      /* case 41 */ 8,
      ST_PNT, 4, E02, E10, E05, E05,
      ST_PYR, 5, P5, P0, E08, E04, N0,
      ST_PYR, 5, E00, P0, P5, E09, N0,
      ST_TET, 4, P5, E04, E05, N0,
      ST_TET, 4, E09, P5, E05, N0,
      ST_PYR, 5, E08, P0, P3, E10, N0,
      ST_PYR, 5, P0, E00, E02, P3, N0,
      ST_TET, 4, E02, E10, P3, N0,
      /* case 42 */ 8,
      ST_PNT, 4, E05, E04, E10, E10,
      ST_PYR, 5, P3, E03, E00, P1, N0,
      ST_PYR, 5, E01, E02, P3, P1, N0,
      ST_TET, 4, P3, E10, E03, N0,
      ST_TET, 4, E02, E10, P3, N0,
      ST_PYR, 5, E00, E04, P5, P1, N0,
      ST_PYR, 5, P1, P5, E05, E01, N0,
      ST_TET, 4, E05, P5, E04, N0,
      /* case 43 */ 9,
      ST_PNT, 6, E08, E04, E05, E01, E02, E10,
      ST_TET, 4, P0, P5, P1, N0,
      ST_PYR, 5, E08, E04, P5, P0, N0,
      ST_PYR, 5, E08, P0, P3, E10, N0,
      ST_TET, 4, P3, P0, P1, N0,
      ST_TET, 4, P3, E02, E10, N0,
      ST_PYR, 5, P3, P1, E01, E02, N0,
      ST_PYR, 5, E01, P1, P5, E05, N0,
      ST_TET, 4, E05, P5, E04, N0,
      /* case 44 */ 8,
      ST_PNT, 4, E10, E03, E04, E04,
      ST_PYR, 5, P5, P2, E01, E09, N0,
      ST_PYR, 5, E11, P2, P5, E05, N0,
      ST_TET, 4, P5, E09, E04, N0,
      ST_TET, 4, E05, P5, E04, N0,
      ST_PYR, 5, E01, P2, P3, E03, N0,
      ST_PYR, 5, P2, E11, E10, P3, N0,
      ST_TET, 4, E10, E03, P3, N0,
      /* case 45 */ 9,
      ST_PNT, 5, E10, E11, E08, E04, E05,
      ST_WDG, 6, E00, E01, E09, P0, P2, P5,
      ST_TET, 4, P2, P0, P5, N0,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_TET, 4, P3, P0, P2, N0,
      ST_PYR, 5, E11, E10, P3, P2, N0,
      ST_PYR, 5, E10, E08, P0, P3, N0,
      ST_PYR, 5, P5, E05, E11, P2, N0,
      ST_TET, 4, E04, E05, P5, N0,
      /* case 46 */ 9,
      ST_PNT, 6, E00, E03, E10, E11, E05, E04,
      ST_TET, 4, P1, P2, P3, N0,
      ST_PYR, 5, E00, P1, P3, E03, N0,
      ST_PYR, 5, E00, E04, P5, P1, N0,
      ST_TET, 4, P5, P2, P1, N0,
      ST_TET, 4, P5, E04, E05, N0,
      ST_PYR, 5, P5, E05, E11, P2, N0,
      ST_PYR, 5, E11, E10, P3, P2, N0,
      ST_TET, 4, E10, E03, P3, N0,
      /* case 47 */ 9,
      ST_PNT, 5, E11, E10, E08, E05, E04,
      ST_PYR, 5, P1, P2, P3, P0, N0,
      ST_TET, 4, P0, P5, P1, N0,
      ST_TET, 4, P5, P2, P1, N0,
      ST_PYR, 5, P5, E05, E11, P2, N0,
      ST_PYR, 5, E11, E10, P3, P2, N0,
      ST_PYR, 5, E10, E08, P0, P3, N0,
      ST_PYR, 5, E08, E04, P5, P0, N0,
      ST_TET, 4, E05, P5, E04, N0,
      /* case 48 */ 1,
      ST_WDG, 6, P5, E09, E05, P4, E08, E07,
      /* case 49 */ 7,
      ST_PNT, 5, E03, E07, E00, E09, E05,
      ST_PYR, 5, E07, E05, P5, P4, N0,
      ST_PYR, 5, E03, E07, P4, P0, N0,
      ST_TET, 4, E00, E03, P0, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_PYR, 5, E00, P0, P5, E09, N0,
      ST_TET, 4, P5, E05, E09, N0,
      /* case 50 */ 7,
      ST_PNT, 5, E07, E05, E08, E00, E01,
      ST_PYR, 5, E05, E01, P1, P5, N0,
      ST_PYR, 5, E07, E05, P5, P4, N0,
      ST_TET, 4, E08, E07, P4, N0,
      ST_TET, 4, P4, P5, P1, N0,
      ST_PYR, 5, E08, P4, P1, E00, N0,
      ST_TET, 4, P1, E01, E00, N0,
      /* case 51 */ 1,
      ST_HEX, 8, E03, E01, E05, E07, P0, P1, P5, P4,
      /* case 52 */ 8,
      ST_PNT, 4, E07, E08, E02, E02,
      ST_PYR, 5, P2, E01, E09, P5, N0,
      ST_PYR, 5, E05, E11, P2, P5, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_TET, 4, E11, E02, P2, N0,
      ST_PYR, 5, E09, E08, P4, P5, N0,
      ST_PYR, 5, P5, P4, E07, E05, N0,
      ST_TET, 4, E07, P4, E08, N0,
      /* case 53 */ 9,
      ST_PNT, 5, E07, E05, E03, E02, E11,
      ST_WDG, 6, P0, P5, P2, E00, E09, E01,
      ST_TET, 4, P5, P2, P0, N0,
      ST_PYR, 5, P0, P2, E02, E03, N0,
      ST_TET, 4, P4, P5, P0, N0,
      ST_PYR, 5, E05, P5, P4, E07, N0,
      ST_PYR, 5, E07, P4, P0, E03, N0,
      ST_PYR, 5, P2, P5, E05, E11, N0,
      ST_TET, 4, E02, P2, E11, N0,
      /* case 54 */ 9,
      ST_PNT, 6, E00, E08, E07, E05, E11, E02,
      ST_TET, 4, P1, P4, P5, N0,
      ST_PYR, 5, E00, E08, P4, P1, N0,
      ST_PYR, 5, E00, P1, P2, E02, N0,
      ST_TET, 4, P2, P1, P5, N0,
      ST_TET, 4, P2, E11, E02, N0,
      ST_PYR, 5, P2, P5, E05, E11, N0,
      ST_PYR, 5, E05, P5, P4, E07, N0,
      ST_TET, 4, E07, P4, E08, N0,
      /* case 55 */ 9,
      ST_PNT, 5, E05, E07, E03, E11, E02,
      ST_PYR, 5, P1, P0, P4, P5, N0,
      ST_TET, 4, P0, P1, P2, N0,
      ST_TET, 4, P2, P1, P5, N0,
      ST_PYR, 5, P2, P5, E05, E11, N0,
      ST_PYR, 5, E05, P5, P4, E07, N0,
      ST_PYR, 5, E07, P4, P0, E03, N0,
      ST_PYR, 5, E03, P0, P2, E02, N0,
      ST_TET, 4, E11, E02, P2, N0,
      /* case 56 */ 8,
      ST_PNT, 4, E05, E09, E02, E02,
      ST_PYR, 5, P3, P4, E08, E03, N0,
      ST_PYR, 5, E07, P4, P3, E10, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_TET, 4, E10, P3, E02, N0,
      ST_PYR, 5, E08, P4, P5, E09, N0,
      ST_PYR, 5, P4, E07, E05, P5, N0,
      ST_TET, 4, E05, E09, P5, N0,
      /* case 57 */ 9,
      ST_PNT, 6, E00, E09, E05, E07, E10, E02,
      ST_TET, 4, P0, P4, P5, N0,
      ST_PYR, 5, E00, P0, P5, E09, N0,
      ST_PYR, 5, E00, E02, P3, P0, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_TET, 4, P3, E02, E10, N0,
      ST_PYR, 5, P3, E10, E07, P4, N0,
      ST_PYR, 5, E07, E05, P5, P4, N0,
      ST_TET, 4, E05, E09, P5, N0,
      /* case 58 */ 9,
      ST_PNT, 5, E05, E01, E07, E10, E02,
      ST_WDG, 6, P4, P1, P3, E08, E00, E03,
      ST_TET, 4, P1, P3, P4, N0,
      ST_PYR, 5, P4, P3, E10, E07, N0,
      ST_TET, 4, P5, P1, P4, N0,
      ST_PYR, 5, E01, P1, P5, E05, N0,
      ST_PYR, 5, E05, P5, P4, E07, N0,
      ST_PYR, 5, P3, P1, E01, E02, N0,
      ST_TET, 4, E10, P3, E02, N0,
      /* case 59 */ 9,
      ST_PNT, 5, E01, E05, E07, E02, E10,
      ST_PYR, 5, P0, P4, P5, P1, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_TET, 4, P3, P0, P1, N0,
      ST_PYR, 5, P3, P1, E01, E02, N0,
      ST_PYR, 5, E01, P1, P5, E05, N0,
      ST_PYR, 5, E05, P5, P4, E07, N0,
      ST_PYR, 5, E07, P4, P3, E10, N0,
      ST_TET, 4, E02, E10, P3, N0,
      /* case 60 */ 2,
      ST_HEX, 8, P3, P4, P5, P2, E10, E07, E05, E11,
      ST_HEX, 8, E03, E08, E09, E01, P3, P4, P5, P2,
      /* case 61 */ 11,
      ST_PNT, 6, P0, P2, P3, P4, E05, E07,
      ST_WDG, 6, P0, P5, P2, E00, E09, E01,
      ST_PYR, 5, E07, E05, P5, P4, N0,
      ST_TET, 4, P4, P5, P0, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_PYR, 5, E10, E07, P4, P3, N0,
      ST_PYR, 5, E11, E10, P3, P2, N0,
      ST_TET, 4, P3, P0, P2, N0,
      ST_PYR, 5, E05, E07, E10, E11, N0,
      ST_TET, 4, P2, P0, P5, N0,
      ST_PYR, 5, E05, E11, P2, P5, N0,
      /* case 62 */ 11,
      ST_PNT, 6, P1, P3, P2, P5, E07, E05,
      ST_WDG, 6, E00, E08, E03, P1, P4, P3,
      ST_PYR, 5, E05, P5, P4, E07, N0,
      ST_TET, 4, P5, P1, P4, N0,
      ST_TET, 4, P5, P2, P1, N0,
      ST_PYR, 5, E11, P2, P5, E05, N0,
      ST_PYR, 5, E10, P3, P2, E11, N0,
      ST_TET, 4, P2, P3, P1, N0,
      ST_PYR, 5, E07, E10, E11, E05, N0,
      ST_TET, 4, P3, P4, P1, N0,
      ST_PYR, 5, E07, P4, P3, E10, N0,
      /* case 63 */ 2,
      ST_HEX, 8, P3, P4, P5, P2, E10, E07, E05, E11,
      ST_WDG, 6, P1, P2, P5, P0, P3, P4,
      /* case 64 */ 1,
      ST_TET, 4, P6, E05, E06, E11,
      /* case 65 */ 2,
      ST_TET, 4, P0, E08, E00, E03,
      ST_TET, 4, P6, E05, E06, E11,
      /* case 66 */ 7,
      ST_PNT, 2, E00, E06,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_PYR, 5, E09, E05, P6, P1, N0,
      ST_TET, 4, E05, E06, P6, N0,
      ST_TET, 4, P6, E06, E11, N0,
      ST_TET, 4, E09, N0, P1, E00,
      ST_TET, 4, E01, E00, P1, N0,
      /* case 67 */ 8,
      ST_PNT, 4, E08, E03, E06, E06,
      ST_PYR, 5, P6, E11, E01, P1, N0,
      ST_PYR, 5, E09, E05, P6, P1, N0,
      ST_TET, 4, P6, E06, E11, N0,
      ST_TET, 4, E05, E06, P6, N0,
      ST_PYR, 5, E01, E03, P0, P1, N0,
      ST_PYR, 5, P1, P0, E08, E09, N0,
      ST_TET, 4, E08, P0, E03, N0,
      /* case 68 */ 1,
      ST_WDG, 6, P2, E01, E02, P6, E05, E06,
      /* case 69 */ 8,
      ST_PNT, 4, E06, E05, E08, E08,
      ST_PYR, 5, P0, E00, E01, P2, N0,
      ST_PYR, 5, E02, E03, P0, P2, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_TET, 4, E03, E08, P0, N0,
      ST_PYR, 5, E01, E05, P6, P2, N0,
      ST_PYR, 5, P2, P6, E06, E02, N0,
      ST_TET, 4, E06, P6, E05, N0,
      /* case 70 */ 7,
      ST_PNT, 5, E00, E02, E09, E05, E06,
      ST_PYR, 5, E02, P2, P6, E06, N0,
      ST_PYR, 5, E00, P1, P2, E02, N0,
      ST_TET, 4, E09, P1, E00, N0,
      ST_TET, 4, P1, P6, P2, N0,
      ST_PYR, 5, E09, E05, P6, P1, N0,
      ST_TET, 4, P6, E05, E06, N0,
      /* case 71 */ 9,
      ST_PNT, 6, E09, E05, E06, E02, E03, E08,
      ST_TET, 4, P1, P6, P2, N0,
      ST_PYR, 5, E09, E05, P6, P1, N0,
      ST_PYR, 5, E09, P1, P0, E08, N0,
      ST_TET, 4, P0, P1, P2, N0,
      ST_TET, 4, P0, E03, E08, N0,
      ST_PYR, 5, P0, P2, E02, E03, N0,
      ST_PYR, 5, E02, P2, P6, E06, N0,
      ST_TET, 4, E06, P6, E05, N0,
      /* case 72 */ 7,
      ST_PNT, 2, E03, E05,
      ST_PYR, 5, P3, P6, E06, E10, N0,
      ST_PYR, 5, E02, E11, P6, P3, N0,
      ST_TET, 4, E11, E05, P6, N0,
      ST_TET, 4, P6, E05, E06, N0,
      ST_TET, 4, E02, N0, P3, E03,
      ST_TET, 4, E10, E03, P3, N0,
      /* case 73 */ 8,
      ST_PNT, 4, E00, E08, E05, E05,
      ST_PYR, 5, P6, E06, E10, P3, N0,
      ST_PYR, 5, E02, E11, P6, P3, N0,
      ST_TET, 4, P6, E05, E06, N0,
      ST_TET, 4, E11, E05, P6, N0,
      ST_PYR, 5, E10, E08, P0, P3, N0,
      ST_PYR, 5, P3, P0, E00, E02, N0,
      ST_TET, 4, E00, P0, E08, N0,
      /* case 74 */ 5,
      ST_WDG, 6, P1, P6, P3, E01, E11, E02,
      ST_PYR, 5, P1, P3, E03, E00, E09,
      ST_PYR, 5, E10, P3, P6, E06, E05,
      ST_PYR, 5, E09, E05, P6, P1, P3,
      ST_PYR, 5, E09, E03, E10, E05, P3,
      /* case 75 */ 9,
      ST_PNT, 5, E08, E10, E09, E05, E06,
      ST_WDG, 6, E01, E02, E11, P1, P3, P6,
      ST_TET, 4, P3, P1, P6, N0,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_TET, 4, P0, P1, P3, N0,
      ST_PYR, 5, E10, E08, P0, P3, N0,
      ST_PYR, 5, E08, E09, P1, P0, N0,
      ST_PYR, 5, P6, E06, E10, P3, N0,
      ST_TET, 4, E05, E06, P6, N0,
      /* case 76 */ 7,
      ST_PNT, 5, E03, E01, E10, E06, E05,
      ST_PYR, 5, E01, E05, P6, P2, N0,
      ST_PYR, 5, E03, E01, P2, P3, N0,
      ST_TET, 4, E10, E03, P3, N0,
      ST_TET, 4, P3, P2, P6, N0,
      ST_PYR, 5, E10, P3, P6, E06, N0,
      ST_TET, 4, P6, E05, E06, N0,
      /* case 77 */ 9,
      ST_PNT, 6, E10, E06, E05, E01, E00, E08,
      ST_TET, 4, P3, P2, P6, N0,
      ST_PYR, 5, E10, P3, P6, E06, N0,
      ST_PYR, 5, E10, E08, P0, P3, N0,
      ST_TET, 4, P0, P2, P3, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_PYR, 5, P0, E00, E01, P2, N0,
      ST_PYR, 5, E01, E05, P6, P2, N0,
      ST_TET, 4, E05, E06, P6, N0,
      /* case 78 */ 5,
      ST_TET, 4, P2, P1, P6, P3,
      ST_PYR, 5, E03, E00, P1, P3, E09,
      ST_PYR, 5, E10, P3, P6, E06, E05,
      ST_PYR, 5, P1, E09, E05, P6, P3,
      ST_PYR, 5, E03, E10, E05, E09, P3,
      /* case 79 */ 9,
      ST_PNT, 5, E10, E08, E09, E06, E05,
      ST_PYR, 5, P2, P3, P0, P1, N0,
      ST_TET, 4, P1, P6, P2, N0,
      ST_TET, 4, P6, P3, P2, N0,
      ST_PYR, 5, P6, E06, E10, P3, N0,
      ST_PYR, 5, E10, E08, P0, P3, N0,
      ST_PYR, 5, E08, E09, P1, P0, N0,
      ST_PYR, 5, E09, E05, P6, P1, N0,
      ST_TET, 4, E06, P6, E05, N0,
      /* case 80 */ 7,
      ST_PNT, 2, E08, E11,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_PYR, 5, E04, P4, P6, E05, N0,
      ST_TET, 4, E05, P6, E11, N0,
      ST_TET, 4, P6, E06, E11, N0,
      ST_TET, 4, E04, P4, N0, E08,
      ST_TET, 4, E07, P4, E08, N0,
      /* case 81 */ 8,
      ST_PNT, 4, E00, E03, E11, E11,
      ST_PYR, 5, P6, P4, E07, E06, N0,
      ST_PYR, 5, E04, P4, P6, E05, N0,
      ST_TET, 4, P6, E06, E11, N0,
      ST_TET, 4, E05, P6, E11, N0,
      ST_PYR, 5, E07, P4, P0, E03, N0,
      ST_PYR, 5, P4, E04, E00, P0, N0,
      ST_TET, 4, E00, E03, P0, N0,
      /* case 82 */ 5,
      ST_WDG, 6, E09, E05, E04, P1, P6, P4,
      ST_PYR, 5, P1, E00, E08, P4, E01,
      ST_PYR, 5, E07, E06, P6, P4, E11,
      ST_PYR, 5, E01, P1, P6, E11, P4,
      ST_PYR, 5, E01, E11, E07, E08, P4,
      /* case 83 */ 9,
      ST_PNT, 5, E03, E07, E01, E11, E06,
      ST_WDG, 6, P1, P4, P6, E09, E04, E05,
      ST_TET, 4, P4, P6, P1, N0,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_TET, 4, P0, P4, P1, N0,
      ST_PYR, 5, E07, P4, P0, E03, N0,
      ST_PYR, 5, E03, P0, P1, E01, N0,
      ST_PYR, 5, P6, P4, E07, E06, N0,
      ST_TET, 4, E11, P6, E06, N0,
      /* case 84 */ 8,
      ST_PNT, 4, E02, E01, E08, E08,
      ST_PYR, 5, P4, P6, E05, E04, N0,
      ST_PYR, 5, E06, P6, P4, E07, N0,
      ST_TET, 4, P4, E04, E08, N0,
      ST_TET, 4, E07, P4, E08, N0,
      ST_PYR, 5, E05, P6, P2, E01, N0,
      ST_PYR, 5, P6, E06, E02, P2, N0,
      ST_TET, 4, E02, E01, P2, N0,
      /* case 85 */ 2,
      ST_HEX, 8, P0, P2, P6, P4, E00, E01, E05, E04,
      ST_HEX, 8, E03, E02, E06, E07, P0, P2, P6, P4,
      /* case 86 */ 9,
      ST_PNT, 5, E02, E06, E00, E08, E07,
      ST_WDG, 6, E09, E05, E04, P1, P6, P4,
      ST_TET, 4, P6, P1, P4, N0,
      ST_PYR, 5, P1, E00, E08, P4, N0,
      ST_TET, 4, P2, P1, P6, N0,
      ST_PYR, 5, E06, E02, P2, P6, N0,
      ST_PYR, 5, E02, E00, P1, P2, N0,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_TET, 4, E08, E07, P4, N0,
      /* case 87 */ 11,
      ST_PNT, 6, P1, P4, P0, P2, E06, E02,
      ST_WDG, 6, E09, E05, E04, P1, P6, P4,
      ST_PYR, 5, E02, P2, P6, E06, N0,
      ST_TET, 4, P2, P1, P6, N0,
      ST_TET, 4, P2, P0, P1, N0,
      ST_PYR, 5, E03, P0, P2, E02, N0,
      ST_PYR, 5, E07, P4, P0, E03, N0,
      ST_TET, 4, P0, P4, P1, N0,
      ST_PYR, 5, E06, E07, E03, E02, N0,
      ST_TET, 4, P4, P6, P1, N0,
      ST_PYR, 5, E06, P6, P4, E07, N0,
      /* case 88 */ 5,
      ST_WDG, 6, E07, E06, E10, P4, P6, P3,
      ST_PYR, 5, P4, E08, E03, P3, E04,
      ST_PYR, 5, E02, E11, P6, P3, E05,
      ST_PYR, 5, E04, P4, P6, E05, P3,
      ST_PYR, 5, E04, E05, E02, E03, P3,
      /* case 89 */ 9,
      ST_PNT, 5, E00, E02, E04, E05, E11,
      ST_WDG, 6, P4, P3, P6, E07, E10, E06,
      ST_TET, 4, P3, P6, P4, N0,
      ST_PYR, 5, P4, P6, E05, E04, N0,
      ST_TET, 4, P0, P3, P4, N0,
      ST_PYR, 5, E02, P3, P0, E00, N0,
      ST_PYR, 5, E00, P0, P4, E04, N0,
      ST_PYR, 5, P6, P3, E02, E11, N0,
      ST_TET, 4, E05, P6, E11, N0,
      /* case 90 */ 5,
      ST_WDG, 6, E01, E02, E11, P1, P3, P6,
      ST_TET, 4, P1, P6, P3, P4,
      ST_WDG, 6, P4, P6, P1, E04, E05, E09,
      ST_WDG, 6, P3, P4, P1, E03, E08, E00,
      ST_WDG, 6, P6, P4, P3, E06, E07, E10,
      /* case 91 */ 5,
      ST_WDG, 6, P6, P3, P1, E11, E02, E01,
      ST_WDG, 6, E07, E06, E10, P4, P6, P3,
      ST_WDG, 6, P4, P6, P1, E04, E05, E09,
      ST_TET, 4, P6, P3, P1, P4,
      ST_TET, 4, P4, P1, P0, P3,
      /* case 92 */ 9,
      ST_PNT, 5, E01, E05, E03, E08, E04,
      ST_WDG, 6, P3, P6, P4, E10, E06, E07,
      ST_TET, 4, P6, P4, P3, N0,
      ST_PYR, 5, P3, P4, E08, E03, N0,
      ST_TET, 4, P2, P6, P3, N0,
      ST_PYR, 5, E05, P6, P2, E01, N0,
      ST_PYR, 5, E01, P2, P3, E03, N0,
      ST_PYR, 5, P4, P6, E05, E04, N0,
      ST_TET, 4, E08, P4, E04, N0,
      /* case 93 */ 11,
      ST_PNT, 6, P3, P4, P0, P2, E05, E01,
      ST_WDG, 6, P3, P6, P4, E10, E06, E07,
      ST_PYR, 5, E01, E05, P6, P2, N0,
      ST_TET, 4, P2, P6, P3, N0,
      ST_TET, 4, P2, P3, P0, N0,
      ST_PYR, 5, E00, E01, P2, P0, N0,
      ST_PYR, 5, E04, E00, P0, P4, N0,
      ST_TET, 4, P0, P3, P4, N0,
      ST_PYR, 5, E05, E01, E00, E04, N0,
      ST_TET, 4, P4, P3, P6, N0,
      ST_PYR, 5, E05, E04, P4, P6, N0,
      /* case 94 */ 5,
      ST_WDG, 6, E04, E09, E05, P4, P1, P6,
      ST_WDG, 6, P3, P4, P1, E03, E08, E00,
      ST_WDG, 6, E10, E07, E06, P3, P4, P6,
      ST_TET, 4, P4, P6, P1, P3,
      ST_TET, 4, P3, P2, P6, P1,
      /* case 95 */ 5,
      ST_WDG, 6, P4, P6, P1, E04, E05, E09,
      ST_WDG, 6, E07, E06, E10, P4, P6, P3,
      ST_PYR, 5, P0, P1, P2, P3, P4,
      ST_TET, 4, P6, P3, P2, P4,
      ST_TET, 4, P6, P2, P1, P4,
      /* case 96 */ 1,
      ST_WDG, 6, P6, E11, E06, P5, E09, E04,
      /* case 97 */ 8,
      ST_PNT, 4, E06, E11, E03, E03,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_PYR, 5, E04, P5, P0, E08, N0,
      ST_TET, 4, P0, E00, E03, N0,
      ST_TET, 4, E08, P0, E03, N0,
      ST_PYR, 5, E09, P5, P6, E11, N0,
      ST_PYR, 5, P5, E04, E06, P6, N0,
      ST_TET, 4, E06, E11, P6, N0,
      /* case 98 */ 7,
      ST_PNT, 5, E00, E04, E01, E11, E06,
      ST_PYR, 5, E04, E06, P6, P5, N0,
      ST_PYR, 5, E00, E04, P5, P1, N0,
      ST_TET, 4, E01, E00, P1, N0,
      ST_TET, 4, P1, P5, P6, N0,
      ST_PYR, 5, E01, P1, P6, E11, N0,
      ST_TET, 4, P6, E06, E11, N0,
      /* case 99 */ 9,
      ST_PNT, 6, E01, E11, E06, E04, E08, E03,
      ST_TET, 4, P1, P5, P6, N0,
      ST_PYR, 5, E01, P1, P6, E11, N0,
      ST_PYR, 5, E01, E03, P0, P1, N0,
      ST_TET, 4, P0, P5, P1, N0,
      ST_TET, 4, P0, E03, E08, N0,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_PYR, 5, E04, E06, P6, P5, N0,
      ST_TET, 4, E06, E11, P6, N0,
      /* case 100 */ 7,
      ST_PNT, 5, E02, E06, E01, E09, E04,
      ST_PYR, 5, E06, P6, P5, E04, N0,
      ST_PYR, 5, E02, P2, P6, E06, N0,
      ST_TET, 4, E01, P2, E02, N0,
      ST_TET, 4, P2, P5, P6, N0,
      ST_PYR, 5, E01, E09, P5, P2, N0,
      ST_TET, 4, P5, E09, E04, N0,
      /* case 101 */ 9,
      ST_PNT, 5, E06, E04, E02, E03, E08,
      ST_WDG, 6, E01, E09, E00, P2, P5, P0,
      ST_TET, 4, P5, P2, P0, N0,
      ST_PYR, 5, P2, E02, E03, P0, N0,
      ST_TET, 4, P6, P2, P5, N0,
      ST_PYR, 5, E04, E06, P6, P5, N0,
      ST_PYR, 5, E06, E02, P2, P6, N0,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_TET, 4, E03, E08, P0, N0,
      /* case 102 */ 1,
      ST_HEX, 8, P1, P5, P6, P2, E00, E04, E06, E02,
      /* case 103 */ 9,
      ST_PNT, 5, E04, E06, E02, E08, E03,
      ST_PYR, 5, P1, P5, P6, P2, N0,
      ST_TET, 4, P2, P0, P1, N0,
      ST_TET, 4, P0, P5, P1, N0,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_PYR, 5, E04, E06, P6, P5, N0,
      ST_PYR, 5, E06, E02, P2, P6, N0,
      ST_PYR, 5, E02, E03, P0, P2, N0,
      ST_TET, 4, E08, P0, E03, N0,
      /* case 104 */ 8,
      ST_PNT, 4, E04, E09, E03, E03,
      ST_PYR, 5, P3, E02, E11, P6, N0,
      ST_PYR, 5, E06, E10, P3, P6, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_TET, 4, E10, E03, P3, N0,
      ST_PYR, 5, E11, E09, P5, P6, N0,
      ST_PYR, 5, P6, P5, E04, E06, N0,
      ST_TET, 4, E04, P5, E09, N0,
      /* case 105 */ 2,
      ST_HEX, 8, E00, E09, E11, E02, P0, P5, P6, P3,
      ST_HEX, 8, P0, P5, P6, P3, E08, E04, E06, E10,
      /* case 106 */ 9,
      ST_PNT, 5, E04, E06, E00, E03, E10,
      ST_WDG, 6, P1, P6, P3, E01, E11, E02,
      ST_TET, 4, P6, P3, P1, N0,
      ST_PYR, 5, P1, P3, E03, E00, N0,
      ST_TET, 4, P5, P6, P1, N0,
      ST_PYR, 5, E06, P6, P5, E04, N0,
      ST_PYR, 5, E04, P5, P1, E00, N0,
      ST_PYR, 5, P3, P6, E06, E10, N0,
      ST_TET, 4, E03, P3, E10, N0,
      /* case 107 */ 11,
      ST_PNT, 6, P1, P3, P0, P5, E06, E04,
      ST_WDG, 6, P1, P6, P3, E01, E11, E02,
      ST_PYR, 5, E04, E06, P6, P5, N0,
      ST_TET, 4, P5, P6, P1, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_PYR, 5, E08, E04, P5, P0, N0,
      ST_PYR, 5, E10, E08, P0, P3, N0,
      ST_TET, 4, P0, P1, P3, N0,
      ST_PYR, 5, E06, E04, E08, E10, N0,
      ST_TET, 4, P3, P1, P6, N0,
      ST_PYR, 5, E06, E10, P3, P6, N0,
      /* case 108 */ 9,
      ST_PNT, 6, E01, E09, E04, E06, E10, E03,
      ST_TET, 4, P2, P5, P6, N0,
      ST_PYR, 5, E01, E09, P5, P2, N0,
      ST_PYR, 5, E01, P2, P3, E03, N0,
      ST_TET, 4, P3, P2, P6, N0,
      ST_TET, 4, P3, E10, E03, N0,
      ST_PYR, 5, P3, P6, E06, E10, N0,
      ST_PYR, 5, E06, P6, P5, E04, N0,
      ST_TET, 4, E04, P5, E09, N0,
      /* case 109 */ 11,
      ST_PNT, 6, P2, P0, P3, P6, E04, E06,
      ST_WDG, 6, E01, E09, E00, P2, P5, P0,
      ST_PYR, 5, E06, P6, P5, E04, N0,
      ST_TET, 4, P6, P2, P5, N0,
      ST_TET, 4, P6, P3, P2, N0,
      ST_PYR, 5, E10, P3, P6, E06, N0,
      ST_PYR, 5, E08, P0, P3, E10, N0,
      ST_TET, 4, P3, P0, P2, N0,
      ST_PYR, 5, E04, E08, E10, E06, N0,
      ST_TET, 4, P0, P5, P2, N0,
      ST_PYR, 5, E04, P5, P0, E08, N0,
      /* case 110 */ 9,
      ST_PNT, 5, E06, E04, E00, E10, E03,
      ST_PYR, 5, P2, P1, P5, P6, N0,
      ST_TET, 4, P1, P2, P3, N0,
      ST_TET, 4, P3, P2, P6, N0,
      ST_PYR, 5, P3, P6, E06, E10, N0,
      ST_PYR, 5, E06, P6, P5, E04, N0,
      ST_PYR, 5, E04, P5, P1, E00, N0,
      ST_PYR, 5, E00, P1, P3, E03, N0,
      ST_TET, 4, E10, E03, P3, N0,
      /* case 111 */ 2,
      ST_HEX, 8, P0, P5, P6, P3, E08, E04, E06, E10,
      ST_WDG, 6, P2, P3, P6, P1, P0, P5,
      /* case 112 */ 7,
      ST_PNT, 5, E08, E09, E07, E06, E11,
      ST_PYR, 5, E09, P5, P6, E11, N0,
      ST_PYR, 5, E08, P4, P5, E09, N0,
      ST_TET, 4, E07, P4, E08, N0,
      ST_TET, 4, P4, P6, P5, N0,
      ST_PYR, 5, E07, E06, P6, P4, N0,
      ST_TET, 4, P6, E06, E11, N0,
      /* case 113 */ 9,
      ST_PNT, 6, E07, E06, E11, E09, E00, E03,
      ST_TET, 4, P4, P6, P5, N0,
      ST_PYR, 5, E07, E06, P6, P4, N0,
      ST_PYR, 5, E07, P4, P0, E03, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_TET, 4, P0, E00, E03, N0,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_PYR, 5, E09, P5, P6, E11, N0,
      ST_TET, 4, E11, P6, E06, N0,
      /* case 114 */ 5,
      ST_TET, 4, P5, P6, P1, P4,
      ST_PYR, 5, E08, P4, P1, E00, E01,
      ST_PYR, 5, E07, E06, P6, P4, E11,
      ST_PYR, 5, P1, P6, E11, E01, P4,
      ST_PYR, 5, E08, E01, E11, E07, P4,
      /* case 115 */ 9,
      ST_PNT, 5, E07, E03, E01, E06, E11,
      ST_PYR, 5, P5, P1, P0, P4, N0,
      ST_TET, 4, P1, P5, P6, N0,
      ST_TET, 4, P6, P5, P4, N0,
      ST_PYR, 5, P6, P4, E07, E06, N0,
      ST_PYR, 5, E07, P4, P0, E03, N0,
      ST_PYR, 5, E03, P0, P1, E01, N0,
      ST_PYR, 5, E01, P1, P6, E11, N0,
      ST_TET, 4, E06, E11, P6, N0,
      /* case 116 */ 9,
      ST_PNT, 6, E09, E01, E02, E06, E07, E08,
      ST_TET, 4, P5, P6, P2, N0,
      ST_PYR, 5, E09, P5, P2, E01, N0,
      ST_PYR, 5, E09, E08, P4, P5, N0,
      ST_TET, 4, P4, P6, P5, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_PYR, 5, E06, E02, P2, P6, N0,
      ST_TET, 4, E02, E01, P2, N0,
      /* case 117 */ 11,
      ST_PNT, 6, P5, P0, P4, P6, E02, E06,
      ST_WDG, 6, P5, P2, P0, E09, E01, E00,
      ST_PYR, 5, E06, E02, P2, P6, N0,
      ST_TET, 4, P6, P2, P5, N0,
      ST_TET, 4, P6, P5, P4, N0,
      ST_PYR, 5, E07, E06, P6, P4, N0,
      ST_PYR, 5, E03, E07, P4, P0, N0,
      ST_TET, 4, P4, P5, P0, N0,
      ST_PYR, 5, E02, E06, E07, E03, N0,
      ST_TET, 4, P0, P5, P2, N0,
      ST_PYR, 5, E02, E03, P0, P2, N0,
      /* case 118 */ 9,
      ST_PNT, 5, E06, E02, E00, E07, E08,
      ST_PYR, 5, P5, P6, P2, P1, N0,
      ST_TET, 4, P1, P4, P5, N0,
      ST_TET, 4, P4, P6, P5, N0,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_PYR, 5, E06, E02, P2, P6, N0,
      ST_PYR, 5, E02, E00, P1, P2, N0,
      ST_PYR, 5, E00, E08, P4, P1, N0,
      ST_TET, 4, E07, P4, E08, N0,
      /* case 119 */ 2,
      ST_HEX, 8, E03, E02, E06, E07, P0, P2, P6, P4,
      ST_WDG, 6, P1, P0, P2, P5, P4, P6,
      /* case 120 */ 9,
      ST_PNT, 5, E09, E11, E08, E03, E02,
      ST_WDG, 6, E07, E06, E10, P4, P6, P3,
      ST_TET, 4, P6, P4, P3, N0,
      ST_PYR, 5, P4, E08, E03, P3, N0,
      ST_TET, 4, P5, P4, P6, N0,
      ST_PYR, 5, E11, E09, P5, P6, N0,
      ST_PYR, 5, E09, E08, P4, P5, N0,
      ST_PYR, 5, P3, E02, E11, P6, N0,
      ST_TET, 4, E03, E02, P3, N0,
      /* case 121 */ 11,
      ST_PNT, 6, P4, P3, P0, P5, E11, E09,
      ST_WDG, 6, E07, E06, E10, P4, P6, P3,
      ST_PYR, 5, E09, P5, P6, E11, N0,
      ST_TET, 4, P5, P4, P6, N0,
      ST_TET, 4, P5, P0, P4, N0,
      ST_PYR, 5, E00, P0, P5, E09, N0,
      ST_PYR, 5, E02, P3, P0, E00, N0,
      ST_TET, 4, P0, P3, P4, N0,
      ST_PYR, 5, E11, E02, E00, E09, N0,
      ST_TET, 4, P3, P6, P4, N0,
      ST_PYR, 5, E11, P6, P3, E02, N0,
      /* case 122 */ 5,
      ST_WDG, 6, P3, P1, P6, E02, E01, E11,
      ST_WDG, 6, E08, E03, E00, P4, P3, P1,
      ST_WDG, 6, P4, P3, P6, E07, E10, E06,
      ST_TET, 4, P3, P1, P6, P4,
      ST_TET, 4, P4, P6, P5, P1,
      /* case 123 */ 5,
      ST_WDG, 6, E02, E11, E01, P3, P6, P1,
      ST_WDG, 6, P3, P6, P4, E10, E06, E07,
      ST_PYR, 5, P0, P4, P5, P1, P3,
      ST_TET, 4, P6, P5, P4, P3,
      ST_TET, 4, P6, P1, P5, P3,
      /* case 124 */ 11,
      ST_PNT, 6, P6, P4, P5, P2, E03, E01,
      ST_WDG, 6, E06, E10, E07, P6, P3, P4,
      ST_PYR, 5, E01, P2, P3, E03, N0,
      ST_TET, 4, P2, P6, P3, N0,
      ST_TET, 4, P2, P5, P6, N0,
      ST_PYR, 5, E09, P5, P2, E01, N0,
      ST_PYR, 5, E08, P4, P5, E09, N0,
      ST_TET, 4, P5, P4, P6, N0,
      ST_PYR, 5, E03, E08, E09, E01, N0,
      ST_TET, 4, P4, P3, P6, N0,
      ST_PYR, 5, E03, P3, P4, E08, N0,
      /* case 125 */ 6,
      ST_WDG, 6, E07, E06, E10, P4, P6, P3,
      ST_WDG, 6, E09, E00, E01, P5, P0, P2,
      ST_TET, 4, P2, P3, P0, P5,
      ST_TET, 4, P5, P4, P6, P3,
      ST_TET, 4, P4, P5, P0, P3,
      ST_TET, 4, P5, P6, P2, P3,
      /* case 126 */ 5,
      ST_WDG, 6, E10, E07, E06, P3, P4, P6,
      ST_WDG, 6, P3, P4, P1, E03, E08, E00,
      ST_PYR, 5, P2, P1, P5, P6, P3,
      ST_TET, 4, P4, P5, P1, P3,
      ST_TET, 4, P4, P6, P5, P3,
      /* case 127 */ 9,
      ST_PNT, 7, P0, P1, P2, P3, P4, P5, P6,
      ST_WDG, 6, E07, E06, E10, P4, P6, P3,
      ST_TET, 4, P4, P3, P6, N0,
      ST_PYR, 5, P5, P6, P2, P1, N0,
      ST_TET, 4, P6, P3, P2, N0,
      ST_PYR, 5, P0, P1, P2, P3, N0,
      ST_TET, 4, P0, P3, P4, N0,
      ST_PYR, 5, P0, P4, P5, P1, N0,
      ST_TET, 4, P4, P6, P5, N0,
      /* case 128 */ 1,
      ST_TET, 4, P7, E06, E07, E10,
      /* case 129 */ 7,
      ST_PNT, 2, E00, E06,
      ST_PYR, 5, P0, E03, E10, P7, N0,
      ST_PYR, 5, E08, P0, P7, E07, N0,
      ST_TET, 4, E07, P7, E06, N0,
      ST_TET, 4, P7, E10, E06, N0,
      ST_TET, 4, E08, P0, N0, E00,
      ST_TET, 4, E03, P0, E00, N0,
      /* case 130 */ 2,
      ST_TET, 4, P1, E00, E09, E01,
      ST_TET, 4, P7, E07, E10, E06,
      /* case 131 */ 8,
      ST_PNT, 4, E09, E01, E06, E06,
      ST_PYR, 5, P7, P0, E03, E10, N0,
      ST_PYR, 5, E08, P0, P7, E07, N0,
      ST_TET, 4, P7, E10, E06, N0,
      ST_TET, 4, E07, P7, E06, N0,
      ST_PYR, 5, E03, P0, P1, E01, N0,
      ST_PYR, 5, P0, E08, E09, P1, N0,
      ST_TET, 4, E09, E01, P1, N0,
      /* case 132 */ 7,
      ST_PNT, 2, E07, E01,
      ST_PYR, 5, P7, E10, E02, P2, N0,
      ST_PYR, 5, E06, P7, P2, E11, N0,
      ST_TET, 4, E11, P2, E01, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_TET, 4, E06, P7, N0, E07,
      ST_TET, 4, E10, P7, E07, N0,
      /* case 133 */ 5,
      ST_WDG, 6, P0, P2, P7, E03, E02, E10,
      ST_PYR, 5, P0, P7, E07, E08, E00,
      ST_PYR, 5, E06, P7, P2, E11, E01,
      ST_PYR, 5, E00, E01, P2, P0, P7,
      ST_PYR, 5, E00, E07, E06, E01, P7,
      /* case 134 */ 8,
      ST_PNT, 4, E09, E00, E07, E07,
      ST_PYR, 5, P7, E10, E02, P2, N0,
      ST_PYR, 5, E11, E06, P7, P2, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_TET, 4, E06, E07, P7, N0,
      ST_PYR, 5, E02, E00, P1, P2, N0,
      ST_PYR, 5, P2, P1, E09, E11, N0,
      ST_TET, 4, E09, P1, E00, N0,
      /* case 135 */ 9,
      ST_PNT, 5, E09, E11, E08, E07, E06,
      ST_WDG, 6, P0, P2, P7, E03, E02, E10,
      ST_TET, 4, P2, P7, P0, N0,
      ST_PYR, 5, P0, P7, E07, E08, N0,
      ST_TET, 4, P1, P2, P0, N0,
      ST_PYR, 5, E11, P2, P1, E09, N0,
      ST_PYR, 5, E09, P1, P0, E08, N0,
      ST_PYR, 5, P7, P2, E11, E06, N0,
      ST_TET, 4, E07, P7, E06, N0,
      /* case 136 */ 1,
      ST_WDG, 6, P7, E07, E06, P3, E03, E02,
      /* case 137 */ 7,
      ST_PNT, 5, E00, E02, E08, E07, E06,
      ST_PYR, 5, E02, E06, P7, P3, N0,
      ST_PYR, 5, E00, E02, P3, P0, N0,
      ST_TET, 4, E08, E00, P0, N0,
      ST_TET, 4, P0, P3, P7, N0,
      ST_PYR, 5, E08, P0, P7, E07, N0,
      ST_TET, 4, P7, E06, E07, N0,
      /* case 138 */ 8,
      ST_PNT, 4, E06, E07, E09, E09,
      ST_PYR, 5, P1, P3, E03, E00, N0,
      ST_PYR, 5, E02, P3, P1, E01, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_TET, 4, E01, P1, E09, N0,
      ST_PYR, 5, E03, P3, P7, E07, N0,
      ST_PYR, 5, P3, E02, E06, P7, N0,
      ST_TET, 4, E06, E07, P7, N0,
      /* case 139 */ 9,
      ST_PNT, 6, E08, E07, E06, E02, E01, E09,
      ST_TET, 4, P0, P3, P7, N0,
      ST_PYR, 5, E08, P0, P7, E07, N0,
      ST_PYR, 5, E08, E09, P1, P0, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_TET, 4, P1, E09, E01, N0,
      ST_PYR, 5, P1, E01, E02, P3, N0,
      ST_PYR, 5, E02, E06, P7, P3, N0,
      ST_TET, 4, E06, E07, P7, N0,
      /* case 140 */ 7,
      ST_PNT, 5, E01, E03, E11, E06, E07,
      ST_PYR, 5, E03, P3, P7, E07, N0,
      ST_PYR, 5, E01, P2, P3, E03, N0,
      ST_TET, 4, E11, P2, E01, N0,
      ST_TET, 4, P2, P7, P3, N0,
      ST_PYR, 5, E11, E06, P7, P2, N0,
      ST_TET, 4, P7, E06, E07, N0,
      /* case 141 */ 5,
      ST_TET, 4, P3, P0, P2, P7,
      ST_PYR, 5, E07, E08, P0, P7, E00,
      ST_PYR, 5, E06, P7, P2, E11, E01,
      ST_PYR, 5, P0, E00, E01, P2, P7,
      ST_PYR, 5, E07, E06, E01, E00, P7,
      /* case 142 */ 9,
      ST_PNT, 6, E03, E00, E09, E11, E06, E07,
      ST_TET, 4, P3, P1, P2, N0,
      ST_PYR, 5, E03, E00, P1, P3, N0,
      ST_PYR, 5, E03, P3, P7, E07, N0,
      ST_TET, 4, P7, P3, P2, N0,
      ST_TET, 4, P7, E06, E07, N0,
      ST_PYR, 5, P7, P2, E11, E06, N0,
      ST_PYR, 5, E11, P2, P1, E09, N0,
      ST_TET, 4, E09, P1, E00, N0,
      /* case 143 */ 9,
      ST_PNT, 5, E11, E09, E08, E06, E07,
      ST_PYR, 5, P3, P0, P1, P2, N0,
      ST_TET, 4, P0, P3, P7, N0,
      ST_TET, 4, P7, P3, P2, N0,
      ST_PYR, 5, P7, P2, E11, E06, N0,
      ST_PYR, 5, E11, P2, P1, E09, N0,
      ST_PYR, 5, E09, P1, P0, E08, N0,
      ST_PYR, 5, E08, P0, P7, E07, N0,
      ST_TET, 4, E06, E07, P7, N0,
      /* case 144 */ 1,
      ST_WDG, 6, P4, E08, E04, P7, E10, E06,
      /* case 145 */ 7,
      ST_PNT, 5, E00, E04, E03, E10, E06,
      ST_PYR, 5, E04, P4, P7, E06, N0,
      ST_PYR, 5, E00, P0, P4, E04, N0,
      ST_TET, 4, E03, P0, E00, N0,
      ST_TET, 4, P0, P7, P4, N0,
      ST_PYR, 5, E03, E10, P7, P0, N0,
      ST_TET, 4, P7, E10, E06, N0,
      /* case 146 */ 8,
      ST_PNT, 4, E06, E10, E01, E01,
      ST_PYR, 5, P1, E00, E08, P4, N0,
      ST_PYR, 5, E04, E09, P1, P4, N0,
      ST_TET, 4, P1, E01, E00, N0,
      ST_TET, 4, E09, E01, P1, N0,
      ST_PYR, 5, E08, E10, P7, P4, N0,
      ST_PYR, 5, P4, P7, E06, E04, N0,
      ST_TET, 4, E06, P7, E10, N0,
      /* case 147 */ 9,
      ST_PNT, 6, E03, E10, E06, E04, E09, E01,
      ST_TET, 4, P0, P7, P4, N0,
      ST_PYR, 5, E03, E10, P7, P0, N0,
      ST_PYR, 5, E03, P0, P1, E01, N0,
      ST_TET, 4, P1, P0, P4, N0,
      ST_TET, 4, P1, E09, E01, N0,
      ST_PYR, 5, P1, P4, E04, E09, N0,
      ST_PYR, 5, E04, P4, P7, E06, N0,
      ST_TET, 4, E06, P7, E10, N0,
      /* case 148 */ 8,
      ST_PNT, 4, E04, E08, E01, E01,
      ST_PYR, 5, P2, P7, E10, E02, N0,
      ST_PYR, 5, E06, P7, P2, E11, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_TET, 4, E11, P2, E01, N0,
      ST_PYR, 5, E10, P7, P4, E08, N0,
      ST_PYR, 5, P7, E06, E04, P4, N0,
      ST_TET, 4, E04, E08, P4, N0,
      /* case 149 */ 9,
      ST_PNT, 5, E04, E06, E00, E01, E11,
      ST_WDG, 6, E03, E10, E02, P0, P7, P2,
      ST_TET, 4, P7, P0, P2, N0,
      ST_PYR, 5, P0, E00, E01, P2, N0,
      ST_TET, 4, P4, P0, P7, N0,
      ST_PYR, 5, E06, E04, P4, P7, N0,
      ST_PYR, 5, E04, E00, P0, P4, N0,
      ST_PYR, 5, P2, E11, E06, P7, N0,
      ST_TET, 4, E01, E11, P2, N0,
      /* case 150 */ 2,
      ST_HEX, 8, P4, P1, P2, P7, E04, E09, E11, E06,
      ST_HEX, 8, E08, E00, E02, E10, P4, P1, P2, P7,
      /* case 151 */ 11,
      ST_PNT, 6, P0, P7, P4, P1, E11, E09,
      ST_WDG, 6, P0, P2, P7, E03, E02, E10,
      ST_PYR, 5, E09, E11, P2, P1, N0,
      ST_TET, 4, P1, P2, P0, N0,
      ST_TET, 4, P1, P0, P4, N0,
      ST_PYR, 5, E04, E09, P1, P4, N0,
      ST_PYR, 5, E06, E04, P4, P7, N0,
      ST_TET, 4, P4, P0, P7, N0,
      ST_PYR, 5, E11, E09, E04, E06, N0,
      ST_TET, 4, P7, P0, P2, N0,
      ST_PYR, 5, E11, E06, P7, P2, N0,
      /* case 152 */ 7,
      ST_PNT, 5, E02, E06, E03, E08, E04,
      ST_PYR, 5, E06, E04, P4, P7, N0,
      ST_PYR, 5, E02, E06, P7, P3, N0,
      ST_TET, 4, E03, E02, P3, N0,
      ST_TET, 4, P3, P7, P4, N0,
      ST_PYR, 5, E03, P3, P4, E08, N0,
      ST_TET, 4, P4, E04, E08, N0,
      /* case 153 */ 1,
      ST_HEX, 8, E00, E04, E06, E02, P0, P4, P7, P3,
      /* case 154 */ 9,
      ST_PNT, 5, E06, E04, E02, E01, E09,
      ST_WDG, 6, P3, P4, P1, E03, E08, E00,
      ST_TET, 4, P4, P1, P3, N0,
      ST_PYR, 5, P3, P1, E01, E02, N0,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, E04, P4, P7, E06, N0,
      ST_PYR, 5, E06, P7, P3, E02, N0,
      ST_PYR, 5, P1, P4, E04, E09, N0,
      ST_TET, 4, E01, P1, E09, N0,
      /* case 155 */ 9,
      ST_PNT, 5, E04, E06, E02, E09, E01,
      ST_PYR, 5, P0, P3, P7, P4, N0,
      ST_TET, 4, P3, P0, P1, N0,
      ST_TET, 4, P1, P0, P4, N0,
      ST_PYR, 5, P1, P4, E04, E09, N0,
      ST_PYR, 5, E04, P4, P7, E06, N0,
      ST_PYR, 5, E06, P7, P3, E02, N0,
      ST_PYR, 5, E02, P3, P1, E01, N0,
      ST_TET, 4, E09, E01, P1, N0,
      /* case 156 */ 9,
      ST_PNT, 6, E03, E08, E04, E06, E11, E01,
      ST_TET, 4, P3, P7, P4, N0,
      ST_PYR, 5, E03, P3, P4, E08, N0,
      ST_PYR, 5, E03, E01, P2, P3, N0,
      ST_TET, 4, P2, P7, P3, N0,
      ST_TET, 4, P2, E01, E11, N0,
      ST_PYR, 5, P2, E11, E06, P7, N0,
      ST_PYR, 5, E06, E04, P4, P7, N0,
      ST_TET, 4, E04, E08, P4, N0,
      /* case 157 */ 9,
      ST_PNT, 5, E06, E04, E00, E11, E01,
      ST_PYR, 5, P3, P7, P4, P0, N0,
      ST_TET, 4, P0, P2, P3, N0,
      ST_TET, 4, P2, P7, P3, N0,
      ST_PYR, 5, P2, E11, E06, P7, N0,
      ST_PYR, 5, E06, E04, P4, P7, N0,
      ST_PYR, 5, E04, E00, P0, P4, N0,
      ST_PYR, 5, E00, E01, P2, P0, N0,
      ST_TET, 4, E11, P2, E01, N0,
      /* case 158 */ 11,
      ST_PNT, 6, P3, P4, P7, P2, E09, E11,
      ST_WDG, 6, E03, E00, E08, P3, P1, P4,
      ST_PYR, 5, E11, P2, P1, E09, N0,
      ST_TET, 4, P2, P3, P1, N0,
      ST_TET, 4, P2, P7, P3, N0,
      ST_PYR, 5, E06, P7, P2, E11, N0,
      ST_PYR, 5, E04, P4, P7, E06, N0,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, E09, E04, E06, E11, N0,
      ST_TET, 4, P4, P1, P3, N0,
      ST_PYR, 5, E09, P1, P4, E04, N0,
      /* case 159 */ 2,
      ST_HEX, 8, P4, P1, P2, P7, E04, E09, E11, E06,
      ST_WDG, 6, P3, P7, P2, P0, P4, P1,
      /* case 160 */ 7,
      ST_PNT, 2, E10, E09,
      ST_PYR, 5, P7, P5, E04, E07, N0,
      ST_PYR, 5, E06, E05, P5, P7, N0,
      ST_TET, 4, E05, E09, P5, N0,
      ST_TET, 4, P5, E09, E04, N0,
      ST_TET, 4, E06, N0, P7, E10,
      ST_TET, 4, E07, E10, P7, N0,
      /* case 161 */ 5,
      ST_WDG, 6, E08, E04, E07, P0, P5, P7,
      ST_PYR, 5, P0, E03, E10, P7, E00,
      ST_PYR, 5, E06, E05, P5, P7, E09,
      ST_PYR, 5, E00, P0, P5, E09, P7,
      ST_PYR, 5, E00, E09, E06, E10, P7,
      /* case 162 */ 8,
      ST_PNT, 4, E01, E00, E10, E10,
      ST_PYR, 5, P7, P5, E04, E07, N0,
      ST_PYR, 5, E05, P5, P7, E06, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_TET, 4, E06, P7, E10, N0,
      ST_PYR, 5, E04, P5, P1, E00, N0,
      ST_PYR, 5, P5, E05, E01, P1, N0,
      ST_TET, 4, E01, E00, P1, N0,
      /* case 163 */ 9,
      ST_PNT, 5, E01, E05, E03, E10, E06,
      ST_WDG, 6, E08, E04, E07, P0, P5, P7,
      ST_TET, 4, P5, P0, P7, N0,
      ST_PYR, 5, P0, E03, E10, P7, N0,
      ST_TET, 4, P1, P0, P5, N0,
      ST_PYR, 5, E05, E01, P1, P5, N0,
      ST_PYR, 5, E01, E03, P0, P1, N0,
      ST_PYR, 5, P7, E06, E05, P5, N0,
      ST_TET, 4, E10, E06, P7, N0,
      /* case 164 */ 5,
      ST_WDG, 6, P5, P7, P2, E05, E06, E11,
      ST_PYR, 5, P5, P2, E01, E09, E04,
      ST_PYR, 5, E02, P2, P7, E10, E07,
      ST_PYR, 5, E04, E07, P7, P5, P2,
      ST_PYR, 5, E04, E01, E02, E07, P2,
      /* case 165 */ 5,
      ST_WDG, 6, P2, P0, P5, E01, E00, E09,
      ST_TET, 4, P2, P0, P5, P7,
      ST_WDG, 6, E06, E05, E11, P7, P5, P2,
      ST_WDG, 6, E03, E10, E02, P0, P7, P2,
      ST_WDG, 6, E04, E07, E08, P5, P7, P0,
      /* case 166 */ 9,
      ST_PNT, 5, E00, E02, E04, E07, E10,
      ST_WDG, 6, E05, E11, E06, P5, P2, P7,
      ST_TET, 4, P2, P5, P7, N0,
      ST_PYR, 5, P5, E04, E07, P7, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_PYR, 5, E02, E00, P1, P2, N0,
      ST_PYR, 5, E00, E04, P5, P1, N0,
      ST_PYR, 5, P7, E10, E02, P2, N0,
      ST_TET, 4, E07, E10, P7, N0,
      /* case 167 */ 5,
      ST_WDG, 6, E07, E08, E04, P7, P0, P5,
      ST_WDG, 6, P2, P7, P0, E02, E10, E03,
      ST_WDG, 6, E11, E06, E05, P2, P7, P5,
      ST_TET, 4, P7, P5, P0, P2,
      ST_TET, 4, P2, P1, P5, P0,
      /* case 168 */ 8,
      ST_PNT, 4, E02, E03, E09, E09,
      ST_PYR, 5, P5, E04, E07, P7, N0,
      ST_PYR, 5, E06, E05, P5, P7, N0,
      ST_TET, 4, P5, E09, E04, N0,
      ST_TET, 4, E05, E09, P5, N0,
      ST_PYR, 5, E07, E03, P3, P7, N0,
      ST_PYR, 5, P7, P3, E02, E06, N0,
      ST_TET, 4, E02, P3, E03, N0,
      /* case 169 */ 9,
      ST_PNT, 5, E02, E06, E00, E09, E05,
      ST_WDG, 6, P0, P7, P5, E08, E07, E04,
      ST_TET, 4, P7, P5, P0, N0,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_TET, 4, P3, P7, P0, N0,
      ST_PYR, 5, E06, P7, P3, E02, N0,
      ST_PYR, 5, E02, P3, P0, E00, N0,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_TET, 4, E09, P5, E05, N0,
      /* case 170 */ 2,
      ST_HEX, 8, E02, E01, E05, E06, P3, P1, P5, P7,
      ST_HEX, 8, P3, P1, P5, P7, E03, E00, E04, E07,
      /* case 171 */ 11,
      ST_PNT, 6, P0, P7, P3, P1, E05, E01,
      ST_WDG, 6, E08, E04, E07, P0, P5, P7,
      ST_PYR, 5, E01, P1, P5, E05, N0,
      ST_TET, 4, P1, P0, P5, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_PYR, 5, E02, P3, P1, E01, N0,
      ST_PYR, 5, E06, P7, P3, E02, N0,
      ST_TET, 4, P3, P7, P0, N0,
      ST_PYR, 5, E05, E06, E02, E01, N0,
      ST_TET, 4, P7, P5, P0, N0,
      ST_PYR, 5, E05, P5, P7, E06, N0,
      /* case 172 */ 9,
      ST_PNT, 5, E03, E07, E01, E09, E04,
      ST_WDG, 6, E11, E06, E05, P2, P7, P5,
      ST_TET, 4, P7, P2, P5, N0,
      ST_PYR, 5, P2, E01, E09, P5, N0,
      ST_TET, 4, P3, P2, P7, N0,
      ST_PYR, 5, E07, E03, P3, P7, N0,
      ST_PYR, 5, E03, E01, P2, P3, N0,
      ST_PYR, 5, P5, E04, E07, P7, N0,
      ST_TET, 4, E09, E04, P5, N0,
      /* case 173 */ 5,
      ST_WDG, 6, E09, E00, E01, P5, P0, P2,
      ST_WDG, 6, P7, P5, P0, E07, E04, E08,
      ST_WDG, 6, E06, E05, E11, P7, P5, P2,
      ST_TET, 4, P5, P2, P0, P7,
      ST_TET, 4, P7, P3, P2, P0,
      /* case 174 */ 11,
      ST_PNT, 6, P2, P5, P1, P3, E07, E03,
      ST_WDG, 6, E11, E06, E05, P2, P7, P5,
      ST_PYR, 5, E03, P3, P7, E07, N0,
      ST_TET, 4, P3, P2, P7, N0,
      ST_TET, 4, P3, P1, P2, N0,
      ST_PYR, 5, E00, P1, P3, E03, N0,
      ST_PYR, 5, E04, P5, P1, E00, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_PYR, 5, E07, E04, E00, E03, N0,
      ST_TET, 4, P5, P7, P2, N0,
      ST_PYR, 5, E07, P7, P5, E04, N0,
      /* case 175 */ 5,
      ST_WDG, 6, E06, E05, E11, P7, P5, P2,
      ST_WDG, 6, P7, P5, P0, E07, E04, E08,
      ST_PYR, 5, P3, P0, P1, P2, P7,
      ST_TET, 4, P5, P1, P0, P7,
      ST_TET, 4, P5, P2, P1, P7,
      /* case 176 */ 7,
      ST_PNT, 5, E09, E08, E05, E06, E10,
      ST_PYR, 5, E08, E10, P7, P4, N0,
      ST_PYR, 5, E09, E08, P4, P5, N0,
      ST_TET, 4, E05, E09, P5, N0,
      ST_TET, 4, P5, P4, P7, N0,
      ST_PYR, 5, E05, P5, P7, E06, N0,
      ST_TET, 4, P7, E10, E06, N0,
      /* case 177 */ 5,
      ST_TET, 4, P4, P5, P0, P7,
      ST_PYR, 5, E10, P7, P0, E03, E00,
      ST_PYR, 5, E06, E05, P5, P7, E09,
      ST_PYR, 5, P0, P5, E09, E00, P7,
      ST_PYR, 5, E10, E00, E09, E06, P7,
      /* case 178 */ 9,
      ST_PNT, 6, E08, E00, E01, E05, E06, E10,
      ST_TET, 4, P4, P5, P1, N0,
      ST_PYR, 5, E08, P4, P1, E00, N0,
      ST_PYR, 5, E08, E10, P7, P4, N0,
      ST_TET, 4, P7, P5, P4, N0,
      ST_TET, 4, P7, E10, E06, N0,
      ST_PYR, 5, P7, E06, E05, P5, N0,
      ST_PYR, 5, E05, E01, P1, P5, N0,
      ST_TET, 4, E01, E00, P1, N0,
      /* case 179 */ 9,
      ST_PNT, 5, E05, E01, E03, E06, E10,
      ST_PYR, 5, P4, P5, P1, P0, N0,
      ST_TET, 4, P0, P7, P4, N0,
      ST_TET, 4, P7, P5, P4, N0,
      ST_PYR, 5, P7, E06, E05, P5, N0,
      ST_PYR, 5, E05, E01, P1, P5, N0,
      ST_PYR, 5, E01, E03, P0, P1, N0,
      ST_PYR, 5, E03, E10, P7, P0, N0,
      ST_TET, 4, E06, P7, E10, N0,
      /* case 180 */ 9,
      ST_PNT, 5, E08, E10, E09, E01, E02,
      ST_WDG, 6, P5, P7, P2, E05, E06, E11,
      ST_TET, 4, P7, P2, P5, N0,
      ST_PYR, 5, P5, P2, E01, E09, N0,
      ST_TET, 4, P4, P7, P5, N0,
      ST_PYR, 5, E10, P7, P4, E08, N0,
      ST_PYR, 5, E08, P4, P5, E09, N0,
      ST_PYR, 5, P2, P7, E10, E02, N0,
      ST_TET, 4, E01, P2, E02, N0,
      /* case 181 */ 5,
      ST_WDG, 6, P2, P0, P5, E01, E00, E09,
      ST_WDG, 6, E10, E02, E03, P7, P2, P0,
      ST_WDG, 6, P7, P2, P5, E06, E11, E05,
      ST_TET, 4, P2, P0, P5, P7,
      ST_TET, 4, P7, P5, P4, P0,
      /* case 182 */ 11,
      ST_PNT, 6, P5, P2, P1, P4, E10, E08,
      ST_WDG, 6, P5, P7, P2, E05, E06, E11,
      ST_PYR, 5, E08, E10, P7, P4, N0,
      ST_TET, 4, P4, P7, P5, N0,
      ST_TET, 4, P4, P5, P1, N0,
      ST_PYR, 5, E00, E08, P4, P1, N0,
      ST_PYR, 5, E02, E00, P1, P2, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_PYR, 5, E10, E08, E00, E02, N0,
      ST_TET, 4, P2, P5, P7, N0,
      ST_PYR, 5, E10, E02, P2, P7, N0,
      /* case 183 */ 5,
      ST_WDG, 6, P7, P2, P5, E06, E11, E05,
      ST_WDG, 6, E10, E02, E03, P7, P2, P0,
      ST_PYR, 5, P4, P5, P1, P0, P7,
      ST_TET, 4, P2, P0, P1, P7,
      ST_TET, 4, P2, P1, P5, P7,
      /* case 184 */ 9,
      ST_PNT, 6, E08, E03, E02, E06, E05, E09,
      ST_TET, 4, P4, P3, P7, N0,
      ST_PYR, 5, E08, E03, P3, P4, N0,
      ST_PYR, 5, E08, P4, P5, E09, N0,
      ST_TET, 4, P5, P4, P7, N0,
      ST_TET, 4, P5, E05, E09, N0,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_PYR, 5, E06, P7, P3, E02, N0,
      ST_TET, 4, E02, P3, E03, N0,
      /* case 185 */ 9,
      ST_PNT, 5, E06, E02, E00, E05, E09,
      ST_PYR, 5, P4, P0, P3, P7, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_TET, 4, P5, P4, P7, N0,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_PYR, 5, E06, P7, P3, E02, N0,
      ST_PYR, 5, E02, P3, P0, E00, N0,
      ST_PYR, 5, E00, P0, P5, E09, N0,
      ST_TET, 4, E05, E09, P5, N0,
      /* case 186 */ 11,
      ST_PNT, 6, P4, P3, P7, P5, E01, E05,
      ST_WDG, 6, P4, P1, P3, E08, E00, E03,
      ST_PYR, 5, E05, E01, P1, P5, N0,
      ST_TET, 4, P5, P1, P4, N0,
      ST_TET, 4, P5, P4, P7, N0,
      ST_PYR, 5, E06, E05, P5, P7, N0,
      ST_PYR, 5, E02, E06, P7, P3, N0,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, E01, E05, E06, E02, N0,
      ST_TET, 4, P3, P4, P1, N0,
      ST_PYR, 5, E01, E02, P3, P1, N0,
      /* case 187 */ 2,
      ST_HEX, 8, E02, E01, E05, E06, P3, P1, P5, P7,
      ST_WDG, 6, P0, P3, P1, P4, P7, P5,
      /* case 188 */ 11,
      ST_PNT, 6, P7, P5, P4, P3, E01, E03,
      ST_WDG, 6, P7, P2, P5, E06, E11, E05,
      ST_PYR, 5, E03, E01, P2, P3, N0,
      ST_TET, 4, P3, P2, P7, N0,
      ST_TET, 4, P3, P7, P4, N0,
      ST_PYR, 5, E08, E03, P3, P4, N0,
      ST_PYR, 5, E09, E08, P4, P5, N0,
      ST_TET, 4, P4, P7, P5, N0,
      ST_PYR, 5, E01, E03, E08, E09, N0,
      ST_TET, 4, P5, P7, P2, N0,
      ST_PYR, 5, E01, E09, P5, P2, N0,
      /* case 189 */ 5,
      ST_WDG, 6, P2, P5, P7, E11, E05, E06,
      ST_WDG, 6, E01, E09, E00, P2, P5, P0,
      ST_PYR, 5, P3, P7, P4, P0, P2,
      ST_TET, 4, P5, P0, P4, P2,
      ST_TET, 4, P5, P4, P7, P2,
      /* case 190 */ 6,
      ST_WDG, 6, E03, E00, E08, P3, P1, P4,
      ST_WDG, 6, E11, E06, E05, P2, P7, P5,
      ST_TET, 4, P5, P4, P7, P2,
      ST_TET, 4, P2, P3, P1, P4,
      ST_TET, 4, P3, P2, P7, P4,
      ST_TET, 4, P2, P1, P5, P4,
      /* case 191 */ 9,
      ST_PNT, 7, P1, P0, P3, P2, P5, P4, P7,
      ST_WDG, 6, P5, P7, P2, E05, E06, E11,
      ST_TET, 4, P5, P7, P2, N0,
      ST_PYR, 5, P4, P0, P3, P7, N0,
      ST_TET, 4, P7, P3, P2, N0,
      ST_PYR, 5, P1, P2, P3, P0, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_PYR, 5, P1, P0, P4, P5, N0,
      ST_TET, 4, P5, P4, P7, N0,
      /* case 192 */ 1,
      ST_WDG, 6, P6, E05, E11, P7, E07, E10,
      /* case 193 */ 8,
      ST_PNT, 4, E11, E05, E00, E00,
      ST_PYR, 5, P0, P7, E07, E08, N0,
      ST_PYR, 5, E10, P7, P0, E03, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_TET, 4, E03, P0, E00, N0,
      ST_PYR, 5, E07, P7, P6, E05, N0,
      ST_PYR, 5, P7, E10, E11, P6, N0,
      ST_TET, 4, E11, E05, P6, N0,
      /* case 194 */ 8,
      ST_PNT, 4, E10, E07, E00, E00,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_PYR, 5, E11, E01, P1, P6, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_TET, 4, E01, E00, P1, N0,
      ST_PYR, 5, E05, E07, P7, P6, N0,
      ST_PYR, 5, P6, P7, E10, E11, N0,
      ST_TET, 4, E10, P7, E07, N0,
      /* case 195 */ 2,
      ST_HEX, 8, E08, E07, E05, E09, P0, P7, P6, P1,
      ST_HEX, 8, P0, P7, P6, P1, E03, E10, E11, E01,
      /* case 196 */ 7,
      ST_PNT, 5, E07, E05, E10, E02, E01,
      ST_PYR, 5, E05, P6, P2, E01, N0,
      ST_PYR, 5, E07, P7, P6, E05, N0,
      ST_TET, 4, E10, P7, E07, N0,
      ST_TET, 4, P7, P2, P6, N0,
      ST_PYR, 5, E10, E02, P2, P7, N0,
      ST_TET, 4, P2, E02, E01, N0,
      /* case 197 */ 9,
      ST_PNT, 5, E05, E01, E07, E08, E00,
      ST_WDG, 6, E10, E02, E03, P7, P2, P0,
      ST_TET, 4, P2, P7, P0, N0,
      ST_PYR, 5, P7, E07, E08, P0, N0,
      ST_TET, 4, P6, P7, P2, N0,
      ST_PYR, 5, E01, E05, P6, P2, N0,
      ST_PYR, 5, E05, E07, P7, P6, N0,
      ST_PYR, 5, P0, E00, E01, P2, N0,
      ST_TET, 4, E08, E00, P0, N0,
      /* case 198 */ 9,
      ST_PNT, 6, E02, E10, E07, E05, E09, E00,
      ST_TET, 4, P2, P6, P7, N0,
      ST_PYR, 5, E02, P2, P7, E10, N0,
      ST_PYR, 5, E02, E00, P1, P2, N0,
      ST_TET, 4, P1, P6, P2, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_PYR, 5, E05, E07, P7, P6, N0,
      ST_TET, 4, E07, E10, P7, N0,
      /* case 199 */ 11,
      ST_PNT, 6, P2, P0, P1, P6, E07, E05,
      ST_WDG, 6, P2, P7, P0, E02, E10, E03,
      ST_PYR, 5, E05, E07, P7, P6, N0,
      ST_TET, 4, P6, P7, P2, N0,
      ST_TET, 4, P6, P2, P1, N0,
      ST_PYR, 5, E09, E05, P6, P1, N0,
      ST_PYR, 5, E08, E09, P1, P0, N0,
      ST_TET, 4, P1, P2, P0, N0,
      ST_PYR, 5, E07, E05, E09, E08, N0,
      ST_TET, 4, P0, P2, P7, N0,
      ST_PYR, 5, E07, E08, P0, P7, N0,
      /* case 200 */ 7,
      ST_PNT, 5, E03, E07, E02, E11, E05,
      ST_PYR, 5, E07, P7, P6, E05, N0,
      ST_PYR, 5, E03, P3, P7, E07, N0,
      ST_TET, 4, E02, P3, E03, N0,
      ST_TET, 4, P3, P6, P7, N0,
      ST_PYR, 5, E02, E11, P6, P3, N0,
      ST_TET, 4, P6, E11, E05, N0,
      /* case 201 */ 9,
      ST_PNT, 6, E02, E11, E05, E07, E08, E00,
      ST_TET, 4, P3, P6, P7, N0,
      ST_PYR, 5, E02, E11, P6, P3, N0,
      ST_PYR, 5, E02, P3, P0, E00, N0,
      ST_TET, 4, P0, P3, P7, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_PYR, 5, P0, P7, E07, E08, N0,
      ST_PYR, 5, E07, P7, P6, E05, N0,
      ST_TET, 4, E05, P6, E11, N0,
      /* case 202 */ 9,
      ST_PNT, 5, E07, E05, E03, E00, E09,
      ST_WDG, 6, E02, E11, E01, P3, P6, P1,
      ST_TET, 4, P6, P3, P1, N0,
      ST_PYR, 5, P3, E03, E00, P1, N0,
      ST_TET, 4, P7, P3, P6, N0,
      ST_PYR, 5, E05, E07, P7, P6, N0,
      ST_PYR, 5, E07, E03, P3, P7, N0,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_TET, 4, E00, E09, P1, N0,
      /* case 203 */ 11,
      ST_PNT, 6, P3, P1, P0, P7, E05, E07,
      ST_WDG, 6, E02, E11, E01, P3, P6, P1,
      ST_PYR, 5, E07, P7, P6, E05, N0,
      ST_TET, 4, P7, P3, P6, N0,
      ST_TET, 4, P7, P0, P3, N0,
      ST_PYR, 5, E08, P0, P7, E07, N0,
      ST_PYR, 5, E09, P1, P0, E08, N0,
      ST_TET, 4, P0, P1, P3, N0,
      ST_PYR, 5, E05, E09, E08, E07, N0,
      ST_TET, 4, P1, P6, P3, N0,
      ST_PYR, 5, E05, P6, P1, E09, N0,
      /* case 204 */ 1,
      ST_HEX, 8, P3, P2, P6, P7, E03, E01, E05, E07,
      /* case 205 */ 9,
      ST_PNT, 5, E01, E05, E07, E00, E08,
      ST_PYR, 5, P3, P2, P6, P7, N0,
      ST_TET, 4, P7, P0, P3, N0,
      ST_TET, 4, P0, P2, P3, N0,
      ST_PYR, 5, P0, E00, E01, P2, N0,
      ST_PYR, 5, E01, E05, P6, P2, N0,
      ST_PYR, 5, E05, E07, P7, P6, N0,
      ST_PYR, 5, E07, E08, P0, P7, N0,
      ST_TET, 4, E00, P0, E08, N0,
      /* case 206 */ 9,
      ST_PNT, 5, E05, E07, E03, E09, E00,
      ST_PYR, 5, P2, P6, P7, P3, N0,
      ST_TET, 4, P3, P1, P2, N0,
      ST_TET, 4, P1, P6, P2, N0,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_PYR, 5, E05, E07, P7, P6, N0,
      ST_PYR, 5, E07, E03, P3, P7, N0,
      ST_PYR, 5, E03, E00, P1, P3, N0,
      ST_TET, 4, E09, P1, E00, N0,
      /* case 207 */ 2,
      ST_HEX, 8, E08, E07, E05, E09, P0, P7, P6, P1,
      ST_WDG, 6, P3, P0, P7, P2, P1, P6,
      /* case 208 */ 7,
      ST_PNT, 5, E08, E10, E04, E05, E11,
      ST_PYR, 5, E10, E11, P6, P7, N0,
      ST_PYR, 5, E08, E10, P7, P4, N0,
      ST_TET, 4, E04, E08, P4, N0,
      ST_TET, 4, P4, P7, P6, N0,
      ST_PYR, 5, E04, P4, P6, E05, N0,
      ST_TET, 4, P6, E11, E05, N0,
      /* case 209 */ 9,
      ST_PNT, 6, E04, E05, E11, E10, E03, E00,
      ST_TET, 4, P4, P7, P6, N0,
      ST_PYR, 5, E04, P4, P6, E05, N0,
      ST_PYR, 5, E04, E00, P0, P4, N0,
      ST_TET, 4, P0, P7, P4, N0,
      ST_TET, 4, P0, E00, E03, N0,
      ST_PYR, 5, P0, E03, E10, P7, N0,
      ST_PYR, 5, E10, E11, P6, P7, N0,
      ST_TET, 4, E11, E05, P6, N0,
      /* case 210 */ 9,
      ST_PNT, 5, E10, E11, E08, E00, E01,
      ST_WDG, 6, P4, P6, P1, E04, E05, E09,
      ST_TET, 4, P6, P1, P4, N0,
      ST_PYR, 5, P4, P1, E00, E08, N0,
      ST_TET, 4, P7, P6, P4, N0,
      ST_PYR, 5, E11, P6, P7, E10, N0,
      ST_PYR, 5, E10, P7, P4, E08, N0,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_TET, 4, E00, P1, E01, N0,
      /* case 211 */ 11,
      ST_PNT, 6, P4, P1, P0, P7, E11, E10,
      ST_WDG, 6, P4, P6, P1, E04, E05, E09,
      ST_PYR, 5, E10, E11, P6, P7, N0,
      ST_TET, 4, P7, P6, P4, N0,
      ST_TET, 4, P7, P4, P0, N0,
      ST_PYR, 5, E03, E10, P7, P0, N0,
      ST_PYR, 5, E01, E03, P0, P1, N0,
      ST_TET, 4, P0, P4, P1, N0,
      ST_PYR, 5, E11, E10, E03, E01, N0,
      ST_TET, 4, P1, P4, P6, N0,
      ST_PYR, 5, E11, E01, P1, P6, N0,
      /* case 212 */ 9,
      ST_PNT, 6, E10, E02, E01, E05, E04, E08,
      ST_TET, 4, P7, P2, P6, N0,
      ST_PYR, 5, E10, E02, P2, P7, N0,
      ST_PYR, 5, E10, P7, P4, E08, N0,
      ST_TET, 4, P4, P7, P6, N0,
      ST_TET, 4, P4, E04, E08, N0,
      ST_PYR, 5, P4, P6, E05, E04, N0,
      ST_PYR, 5, E05, P6, P2, E01, N0,
      ST_TET, 4, E01, P2, E02, N0,
      /* case 213 */ 11,
      ST_PNT, 6, P7, P0, P4, P6, E01, E05,
      ST_WDG, 6, E10, E02, E03, P7, P2, P0,
      ST_PYR, 5, E05, P6, P2, E01, N0,
      ST_TET, 4, P6, P7, P2, N0,
      ST_TET, 4, P6, P4, P7, N0,
      ST_PYR, 5, E04, P4, P6, E05, N0,
      ST_PYR, 5, E00, P0, P4, E04, N0,
      ST_TET, 4, P4, P0, P7, N0,
      ST_PYR, 5, E01, E00, E04, E05, N0,
      ST_TET, 4, P0, P2, P7, N0,
      ST_PYR, 5, E01, P2, P0, E00, N0,
      /* case 214 */ 11,
      ST_PNT, 6, P6, P1, P2, P7, E08, E10,
      ST_WDG, 6, E05, E04, E09, P6, P4, P1,
      ST_PYR, 5, E10, P7, P4, E08, N0,
      ST_TET, 4, P7, P6, P4, N0,
      ST_TET, 4, P7, P2, P6, N0,
      ST_PYR, 5, E02, P2, P7, E10, N0,
      ST_PYR, 5, E00, P1, P2, E02, N0,
      ST_TET, 4, P2, P1, P6, N0,
      ST_PYR, 5, E08, E00, E02, E10, N0,
      ST_TET, 4, P1, P4, P6, N0,
      ST_PYR, 5, E08, P4, P1, E00, N0,
      /* case 215 */ 6,
      ST_WDG, 6, E09, E05, E04, P1, P6, P4,
      ST_WDG, 6, E02, E03, E10, P2, P0, P7,
      ST_TET, 4, P7, P4, P0, P2,
      ST_TET, 4, P2, P1, P6, P4,
      ST_TET, 4, P1, P2, P0, P4,
      ST_TET, 4, P2, P6, P7, P4,
      /* case 216 */ 5,
      ST_TET, 4, P7, P6, P4, P3,
      ST_PYR, 5, E03, P3, P4, E08, E04,
      ST_PYR, 5, E02, E11, P6, P3, E05,
      ST_PYR, 5, P4, P6, E05, E04, P3,
      ST_PYR, 5, E03, E04, E05, E02, P3,
      /* case 217 */ 9,
      ST_PNT, 5, E02, E00, E04, E11, E05,
      ST_PYR, 5, P7, P4, P0, P3, N0,
      ST_TET, 4, P4, P7, P6, N0,
      ST_TET, 4, P6, P7, P3, N0,
      ST_PYR, 5, P6, P3, E02, E11, N0,
      ST_PYR, 5, E02, P3, P0, E00, N0,
      ST_PYR, 5, E00, P0, P4, E04, N0,
      ST_PYR, 5, E04, P4, P6, E05, N0,
      ST_TET, 4, E11, E05, P6, N0,
      /* case 218 */ 5,
      ST_WDG, 6, P1, P4, P6, E09, E04, E05,
      ST_WDG, 6, E03, E00, E08, P3, P1, P4,
      ST_WDG, 6, P3, P1, P6, E02, E01, E11,
      ST_TET, 4, P1, P4, P6, P3,
      ST_TET, 4, P3, P6, P7, P4,
      /* case 219 */ 5,
      ST_WDG, 6, E09, E05, E04, P1, P6, P4,
      ST_WDG, 6, P1, P6, P3, E01, E11, E02,
      ST_PYR, 5, P0, P3, P7, P4, P1,
      ST_TET, 4, P6, P7, P3, P1,
      ST_TET, 4, P6, P4, P7, P1,
      /* case 220 */ 9,
      ST_PNT, 5, E05, E01, E03, E04, E08,
      ST_PYR, 5, P7, P3, P2, P6, N0,
      ST_TET, 4, P3, P7, P4, N0,
      ST_TET, 4, P4, P7, P6, N0,
      ST_PYR, 5, P4, P6, E05, E04, N0,
      ST_PYR, 5, E05, P6, P2, E01, N0,
      ST_PYR, 5, E01, P2, P3, E03, N0,
      ST_PYR, 5, E03, P3, P4, E08, N0,
      ST_TET, 4, E04, E08, P4, N0,
      /* case 221 */ 2,
      ST_HEX, 8, P0, P2, P6, P4, E00, E01, E05, E04,
      ST_WDG, 6, P7, P4, P6, P3, P0, P2,
      /* case 222 */ 5,
      ST_WDG, 6, E04, E09, E05, P4, P1, P6,
      ST_WDG, 6, P4, P1, P3, E08, E00, E03,
      ST_PYR, 5, P7, P3, P2, P6, P4,
      ST_TET, 4, P1, P2, P3, P4,
      ST_TET, 4, P1, P6, P2, P4,
      /* case 223 */ 9,
      ST_PNT, 7, P0, P3, P7, P4, P1, P2, P6,
      ST_WDG, 6, E09, E05, E04, P1, P6, P4,
      ST_TET, 4, P1, P4, P6, N0,
      ST_PYR, 5, P2, P6, P7, P3, N0,
      ST_TET, 4, P6, P4, P7, N0,
      ST_PYR, 5, P0, P3, P7, P4, N0,
      ST_TET, 4, P0, P4, P1, N0,
      ST_PYR, 5, P0, P1, P2, P3, N0,
      ST_TET, 4, P1, P6, P2, N0,
      /* case 224 */ 7,
      ST_PNT, 5, E10, E11, E07, E04, E09,
      ST_PYR, 5, E11, E09, P5, P6, N0,
      ST_PYR, 5, E10, E11, P6, P7, N0,
      ST_TET, 4, E07, E10, P7, N0,
      ST_TET, 4, P7, P6, P5, N0,
      ST_PYR, 5, E07, P7, P5, E04, N0,
      ST_TET, 4, P5, E09, E04, N0,
      /* case 225 */ 9,
      ST_PNT, 5, E11, E09, E10, E03, E00,
      ST_WDG, 6, P7, P5, P0, E07, E04, E08,
      ST_TET, 4, P5, P0, P7, N0,
      ST_PYR, 5, P7, P0, E03, E10, N0,
      ST_TET, 4, P6, P5, P7, N0,
      ST_PYR, 5, E09, P5, P6, E11, N0,
      ST_PYR, 5, E11, P6, P7, E10, N0,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_TET, 4, E03, P0, E00, N0,
      /* case 226 */ 9,
      ST_PNT, 6, E04, E07, E10, E11, E01, E00,
      ST_TET, 4, P5, P7, P6, N0,
      ST_PYR, 5, E04, E07, P7, P5, N0,
      ST_PYR, 5, E04, P5, P1, E00, N0,
      ST_TET, 4, P1, P5, P6, N0,
      ST_TET, 4, P1, E01, E00, N0,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_PYR, 5, E11, P6, P7, E10, N0,
      ST_TET, 4, E10, P7, E07, N0,
      /* case 227 */ 11,
      ST_PNT, 6, P5, P0, P1, P6, E10, E11,
      ST_WDG, 6, E04, E07, E08, P5, P7, P0,
      ST_PYR, 5, E11, P6, P7, E10, N0,
      ST_TET, 4, P6, P5, P7, N0,
      ST_TET, 4, P6, P1, P5, N0,
      ST_PYR, 5, E01, P1, P6, E11, N0,
      ST_PYR, 5, E03, P0, P1, E01, N0,
      ST_TET, 4, P1, P0, P5, N0,
      ST_PYR, 5, E10, E03, E01, E11, N0,
      ST_TET, 4, P0, P7, P5, N0,
      ST_PYR, 5, E10, P7, P0, E03, N0,
      /* case 228 */ 5,
      ST_TET, 4, P6, P5, P7, P2,
      ST_PYR, 5, E01, E09, P5, P2, E04,
      ST_PYR, 5, E02, P2, P7, E10, E07,
      ST_PYR, 5, P5, E04, E07, P7, P2,
      ST_PYR, 5, E01, E02, E07, E04, P2,
      /* case 229 */ 5,
      ST_WDG, 6, E08, E04, E07, P0, P5, P7,
      ST_WDG, 6, P2, P0, P5, E01, E00, E09,
      ST_WDG, 6, E02, E03, E10, P2, P0, P7,
      ST_TET, 4, P0, P7, P5, P2,
      ST_TET, 4, P2, P6, P7, P5,
      /* case 230 */ 9,
      ST_PNT, 5, E02, E00, E04, E10, E07,
      ST_PYR, 5, P6, P2, P1, P5, N0,
      ST_TET, 4, P5, P7, P6, N0,
      ST_TET, 4, P7, P2, P6, N0,
      ST_PYR, 5, P7, E10, E02, P2, N0,
      ST_PYR, 5, E02, E00, P1, P2, N0,
      ST_PYR, 5, E00, E04, P5, P1, N0,
      ST_PYR, 5, E04, E07, P7, P5, N0,
      ST_TET, 4, E10, P7, E07, N0,
      /* case 231 */ 5,
      ST_WDG, 6, P0, P7, P5, E08, E07, E04,
      ST_WDG, 6, E03, E10, E02, P0, P7, P2,
      ST_PYR, 5, P1, P5, P6, P2, P0,
      ST_TET, 4, P7, P2, P6, P0,
      ST_TET, 4, P7, P6, P5, P0,
      /* case 232 */ 9,
      ST_PNT, 6, E07, E04, E09, E11, E02, E03,
      ST_TET, 4, P7, P6, P5, N0,
      ST_PYR, 5, E07, P7, P5, E04, N0,
      ST_PYR, 5, E07, E03, P3, P7, N0,
      ST_TET, 4, P3, P6, P7, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_PYR, 5, P3, E02, E11, P6, N0,
      ST_PYR, 5, E11, E09, P5, P6, N0,
      ST_TET, 4, E09, E04, P5, N0,
      /* case 233 */ 11,
      ST_PNT, 6, P7, P0, P3, P6, E09, E11,
      ST_WDG, 6, P7, P5, P0, E07, E04, E08,
      ST_PYR, 5, E11, E09, P5, P6, N0,
      ST_TET, 4, P6, P5, P7, N0,
      ST_TET, 4, P6, P7, P3, N0,
      ST_PYR, 5, E02, E11, P6, P3, N0,
      ST_PYR, 5, E00, E02, P3, P0, N0,
      ST_TET, 4, P3, P7, P0, N0,
      ST_PYR, 5, E09, E11, E02, E00, N0,
      ST_TET, 4, P0, P7, P5, N0,
      ST_PYR, 5, E09, E00, P0, P5, N0,
      /* case 234 */ 11,
      ST_PNT, 6, P6, P1, P5, P7, E03, E07,
      ST_WDG, 6, P6, P3, P1, E11, E02, E01,
      ST_PYR, 5, E07, E03, P3, P7, N0,
      ST_TET, 4, P7, P3, P6, N0,
      ST_TET, 4, P7, P6, P5, N0,
      ST_PYR, 5, E04, E07, P7, P5, N0,
      ST_PYR, 5, E00, E04, P5, P1, N0,
      ST_TET, 4, P5, P6, P1, N0,
      ST_PYR, 5, E03, E07, E04, E00, N0,
      ST_TET, 4, P1, P6, P3, N0,
      ST_PYR, 5, E03, E00, P1, P3, N0,
      /* case 235 */ 6,
      ST_WDG, 6, P1, P6, P3, E01, E11, E02,
      ST_WDG, 6, P5, P0, P7, E04, E08, E07,
      ST_TET, 4, P7, P0, P3, P5,
      ST_TET, 4, P5, P6, P1, P3,
      ST_TET, 4, P1, P0, P5, P3,
      ST_TET, 4, P5, P7, P6, P3,
      /* case 236 */ 9,
      ST_PNT, 5, E07, E03, E01, E04, E09,
      ST_PYR, 5, P6, P7, P3, P2, N0,
      ST_TET, 4, P2, P5, P6, N0,
      ST_TET, 4, P5, P7, P6, N0,
      ST_PYR, 5, P5, E04, E07, P7, N0,
      ST_PYR, 5, E07, E03, P3, P7, N0,
      ST_PYR, 5, E03, E01, P2, P3, N0,
      ST_PYR, 5, E01, E09, P5, P2, N0,
      ST_TET, 4, E04, P5, E09, N0,
      /* case 237 */ 5,
      ST_WDG, 6, P0, P5, P2, E00, E09, E01,
      ST_WDG, 6, E08, E04, E07, P0, P5, P7,
      ST_PYR, 5, P3, P2, P6, P7, P0,
      ST_TET, 4, P5, P7, P6, P0,
      ST_TET, 4, P5, P6, P2, P0,
      /* case 238 */ 2,
      ST_HEX, 8, E00, E03, E07, E04, P1, P3, P7, P5,
      ST_WDG, 6, P2, P1, P3, P6, P5, P7,
      /* case 239 */ 9,
      ST_PNT, 7, P3, P2, P1, P0, P7, P6, P5,
      ST_WDG, 6, P7, P5, P0, E07, E04, E08,
      ST_TET, 4, P7, P5, P0, N0,
      ST_PYR, 5, P6, P2, P1, P5, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_PYR, 5, P3, P0, P1, P2, N0,
      ST_TET, 4, P3, P7, P0, N0,
      ST_PYR, 5, P3, P2, P6, P7, N0,
      ST_TET, 4, P7, P6, P5, N0,
      /* case 240 */ 1,
      ST_HEX, 8, E08, E09, E11, E10, P4, P5, P6, P7,
      /* case 241 */ 9,
      ST_PNT, 5, E09, E11, E10, E00, E03,
      ST_PYR, 5, P4, P7, P6, P5, N0,
      ST_TET, 4, P7, P4, P0, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_PYR, 5, E09, P5, P6, E11, N0,
      ST_PYR, 5, E11, P6, P7, E10, N0,
      ST_PYR, 5, E10, P7, P0, E03, N0,
      ST_TET, 4, E00, E03, P0, N0,
      /* case 242 */ 9,
      ST_PNT, 5, E11, E10, E08, E01, E00,
      ST_PYR, 5, P5, P4, P7, P6, N0,
      ST_TET, 4, P4, P5, P1, N0,
      ST_TET, 4, P1, P5, P6, N0,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_PYR, 5, E11, P6, P7, E10, N0,
      ST_PYR, 5, E10, P7, P4, E08, N0,
      ST_PYR, 5, E08, P4, P1, E00, N0,
      ST_TET, 4, E01, E00, P1, N0,
      /* case 243 */ 2,
      ST_HEX, 8, P0, P7, P6, P1, E03, E10, E11, E01,
      ST_WDG, 6, P5, P1, P6, P4, P0, P7,
      /* case 244 */ 9,
      ST_PNT, 5, E10, E08, E09, E02, E01,
      ST_PYR, 5, P6, P5, P4, P7, N0,
      ST_TET, 4, P5, P6, P2, N0,
      ST_TET, 4, P2, P6, P7, N0,
      ST_PYR, 5, P2, P7, E10, E02, N0,
      ST_PYR, 5, E10, P7, P4, E08, N0,
      ST_PYR, 5, E08, P4, P5, E09, N0,
      ST_PYR, 5, E09, P5, P2, E01, N0,
      ST_TET, 4, E02, E01, P2, N0,
      /* case 245 */ 5,
      ST_WDG, 6, E00, E01, E09, P0, P2, P5,
      ST_WDG, 6, P0, P2, P7, E03, E02, E10,
      ST_PYR, 5, P4, P7, P6, P5, P0,
      ST_TET, 4, P2, P6, P7, P0,
      ST_TET, 4, P2, P5, P6, P0,
      /* case 246 */ 2,
      ST_HEX, 8, P1, P4, P7, P2, E00, E08, E10, E02,
      ST_WDG, 6, P6, P2, P7, P5, P1, P4,
      /* case 247 */ 9,
      ST_PNT, 7, P4, P5, P1, P0, P7, P6, P2,
      ST_WDG, 6, E10, E02, E03, P7, P2, P0,
      ST_TET, 4, P7, P0, P2, N0,
      ST_PYR, 5, P6, P2, P1, P5, N0,
      ST_TET, 4, P2, P0, P1, N0,
      ST_PYR, 5, P4, P5, P1, P0, N0,
      ST_TET, 4, P4, P0, P7, N0,
      ST_PYR, 5, P4, P7, P6, P5, N0,
      ST_TET, 4, P7, P2, P6, N0,
      /* case 248 */ 9,
      ST_PNT, 5, E11, E09, E08, E02, E03,
      ST_PYR, 5, P7, P6, P5, P4, N0,
      ST_TET, 4, P4, P3, P7, N0,
      ST_TET, 4, P3, P6, P7, N0,
      ST_PYR, 5, P3, E02, E11, P6, N0,
      ST_PYR, 5, E11, E09, P5, P6, N0,
      ST_PYR, 5, E09, E08, P4, P5, N0,
      ST_PYR, 5, E08, E03, P3, P4, N0,
      ST_TET, 4, E02, P3, E03, N0,
      /* case 249 */ 2,
      ST_HEX, 8, E00, E09, E11, E02, P0, P5, P6, P3,
      ST_WDG, 6, P4, P0, P5, P7, P3, P6,
      /* case 250 */ 5,
      ST_WDG, 6, P3, P1, P6, E02, E01, E11,
      ST_WDG, 6, E03, E00, E08, P3, P1, P4,
      ST_PYR, 5, P7, P6, P5, P4, P3,
      ST_TET, 4, P1, P4, P5, P3,
      ST_TET, 4, P1, P5, P6, P3,
      /* case 251 */ 9,
      ST_PNT, 7, P0, P4, P7, P3, P1, P5, P6,
      ST_WDG, 6, P1, P6, P3, E01, E11, E02,
      ST_TET, 4, P1, P6, P3, N0,
      ST_PYR, 5, P5, P4, P7, P6, N0,
      ST_TET, 4, P6, P7, P3, N0,
      ST_PYR, 5, P0, P3, P7, P4, N0,
      ST_TET, 4, P0, P1, P3, N0,
      ST_PYR, 5, P0, P4, P5, P1, N0,
      ST_TET, 4, P1, P5, P6, N0,
      /* case 252 */ 2,
      ST_HEX, 8, P4, P3, P2, P5, E08, E03, E01, E09,
      ST_WDG, 6, P6, P5, P2, P7, P4, P3,
      /* case 253 */ 9,
      ST_PNT, 7, P3, P7, P4, P0, P2, P6, P5,
      ST_WDG, 6, E01, E09, E00, P2, P5, P0,
      ST_TET, 4, P2, P0, P5, N0,
      ST_PYR, 5, P6, P5, P4, P7, N0,
      ST_TET, 4, P5, P0, P4, N0,
      ST_PYR, 5, P3, P7, P4, P0, N0,
      ST_TET, 4, P3, P0, P2, N0,
      ST_PYR, 5, P3, P2, P6, P7, N0,
      ST_TET, 4, P2, P5, P6, N0,
      /* case 254 */ 9,
      ST_PNT, 7, P7, P6, P5, P4, P3, P2, P1,
      ST_WDG, 6, E03, E00, E08, P3, P1, P4,
      ST_TET, 4, P3, P4, P1, N0,
      ST_PYR, 5, P2, P1, P5, P6, N0,
      ST_TET, 4, P1, P4, P5, N0,
      ST_PYR, 5, P7, P6, P5, P4, N0,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, P7, P3, P2, P6, N0,
      ST_TET, 4, P3, P1, P2, N0,
      /* case 255 */ 1,
      ST_HEX, 8, P0, P1, P2, P3, P4, P5, P6, P7,
      // vtkm::CELL_SHAPE_WEDGE
      /* case 0 */ 0,
      /* case 1 */ 1,
      ST_TET, 4, E06, E00, E02, P0,
      /* case 2 */ 1,
      ST_TET, 4, E07, E01, E00, P1,
      /* case 3 */ 1,
      ST_WDG, 6, E02, E06, P0, E01, E07, P1,
      /* case 4 */ 1,
      ST_TET, 4, E08, E02, E01, P2,
      /* case 5 */ 1,
      ST_WDG, 6, E01, E08, P2, E00, E06, P0,
      /* case 6 */ 1,
      ST_WDG, 6, E00, E07, P1, E02, E08, P2,
      /* case 7 */ 1,
      ST_WDG, 6, P0, P1, P2, E06, E07, E08,
      /* case 8 */ 1,
      ST_TET, 4, E06, E05, E03, P3,
      /* case 9 */ 1,
      ST_WDG, 6, P0, E00, E02, P3, E03, E05,
      /* case 10 */ 7,
      ST_PNT, 6, E00, E01, E07, E03, E05, E06,
      ST_PYR, 5, E06, E00, P1, P3, N0,
      ST_PYR, 5, P3, P1, E07, E03, N0,
      ST_TET, 4, P3, E03, E05, N0,
      ST_TET, 4, E05, E06, P3, N0,
      ST_TET, 4, P1, E01, E07, N0,
      ST_TET, 4, P1, E00, E01, N0,
      /* case 11 */ 7,
      ST_PNT, 5, E01, E02, E05, E03, E07,
      ST_TET, 4, P0, P1, P3, N0,
      ST_PYR, 5, E02, E01, P1, P0, N0,
      ST_PYR, 5, E02, P0, P3, E05, N0,
      ST_TET, 4, E05, P3, E03, N0,
      ST_PYR, 5, P3, P1, E07, E03, N0,
      ST_TET, 4, P1, E01, E07, N0,
      /* case 12 */ 7,
      ST_PNT, 6, E05, E03, E06, E02, E01, E08,
      ST_PYR, 5, E08, P2, P3, E05, N0,
      ST_PYR, 5, P2, E02, E06, P3, N0,
      ST_TET, 4, P2, E01, E02, N0,
      ST_TET, 4, E01, P2, E08, N0,
      ST_TET, 4, P3, E06, E03, N0,
      ST_TET, 4, P3, E03, E05, N0,
      /* case 13 */ 7,
      ST_PNT, 5, E01, E00, E03, E05, E08,
      ST_TET, 4, P0, P3, P2, N0,
      ST_PYR, 5, E00, P0, P2, E01, N0,
      ST_PYR, 5, E00, E03, P3, P0, N0,
      ST_TET, 4, E03, E05, P3, N0,
      ST_PYR, 5, P3, E05, E08, P2, N0,
      ST_TET, 4, P2, E08, E01, N0,
      /* case 14 */ 8,
      ST_PNT, 7, E03, E05, E08, E07, P3, P2, P1,
      ST_WDG, 6, P2, P1, P3, E02, E00, E06,
      ST_PYR, 5, E05, E03, E07, E08, N0,
      ST_PYR, 5, E07, P1, P2, E08, N0,
      ST_TET, 4, P3, P2, P1, N0,
      ST_TET, 4, P3, E03, E05, N0,
      ST_PYR, 5, E03, P3, P1, E07, N0,
      ST_PYR, 5, E08, P2, P3, E05, N0,
      /* case 15 */ 8,
      ST_PNT, 7, P1, P2, P3, E05, E03, E07, E08,
      ST_TET, 4, P0, P2, P1, P3,
      ST_PYR, 5, E05, E03, E07, E08, N0,
      ST_PYR, 5, E08, E07, P1, P2, N0,
      ST_TET, 4, P3, P2, P1, N0,
      ST_TET, 4, P3, E03, E05, N0,
      ST_PYR, 5, P3, P1, E07, E03, N0,
      ST_PYR, 5, P2, P3, E05, E08, N0,
      /* case 16 */ 1,
      ST_TET, 4, E07, E03, E04, P4,
      /* case 17 */ 7,
      ST_PNT, 6, E03, E04, E07, E00, E02, E06,
      ST_PYR, 5, E06, P0, P4, E03, N0,
      ST_PYR, 5, P0, E00, E07, P4, N0,
      ST_TET, 4, P0, E02, E00, N0,
      ST_TET, 4, E02, P0, E06, N0,
      ST_TET, 4, P4, E07, E04, N0,
      ST_TET, 4, P4, E04, E03, N0,
      /* case 18 */ 1,
      ST_WDG, 6, P1, E01, E00, P4, E04, E03,
      /* case 19 */ 7,
      ST_PNT, 5, E02, E01, E04, E03, E06,
      ST_TET, 4, P1, P4, P0, N0,
      ST_PYR, 5, E01, P1, P0, E02, N0,
      ST_PYR, 5, E01, E04, P4, P1, N0,
      ST_TET, 4, E04, E03, P4, N0,
      ST_PYR, 5, P4, E03, E06, P0, N0,
      ST_TET, 4, P0, E06, E02, N0,
      /* case 20 */ 7,
      ST_PNT, 6, E01, E02, E08, E04, E03, E07,
      ST_PYR, 5, E07, E01, P2, P4, N0,
      ST_PYR, 5, P4, P2, E08, E04, N0,
      ST_TET, 4, P4, E04, E03, N0,
      ST_TET, 4, E03, E07, P4, N0,
      ST_TET, 4, P2, E02, E08, N0,
      ST_TET, 4, P2, E01, E02, N0,
      /* case 21 */ 8,
      ST_PNT, 7, E04, E03, E06, E08, P4, P0, P2,
      ST_WDG, 6, P0, P2, P4, E00, E01, E07,
      ST_PYR, 5, E03, E04, E08, E06, N0,
      ST_PYR, 5, E08, P2, P0, E06, N0,
      ST_TET, 4, P4, P0, P2, N0,
      ST_TET, 4, P4, E04, E03, N0,
      ST_PYR, 5, E04, P4, P2, E08, N0,
      ST_PYR, 5, E06, P0, P4, E03, N0,
      /* case 22 */ 7,
      ST_PNT, 5, E02, E00, E03, E04, E08,
      ST_TET, 4, P1, P2, P4, N0,
      ST_PYR, 5, E00, E02, P2, P1, N0,
      ST_PYR, 5, E00, P1, P4, E03, N0,
      ST_TET, 4, E03, P4, E04, N0,
      ST_PYR, 5, P4, P2, E08, E04, N0,
      ST_TET, 4, P2, E02, E08, N0,
      /* case 23 */ 8,
      ST_PNT, 7, P2, P0, P4, E03, E04, E08, E06,
      ST_TET, 4, P1, P0, P2, P4,
      ST_PYR, 5, E03, E04, E08, E06, N0,
      ST_PYR, 5, E06, E08, P2, P0, N0,
      ST_TET, 4, P4, P0, P2, N0,
      ST_TET, 4, P4, E04, E03, N0,
      ST_PYR, 5, P4, P2, E08, E04, N0,
      ST_PYR, 5, P0, P4, E03, E06, N0,
      /* case 24 */ 1,
      ST_WDG, 6, E04, E07, P4, E05, E06, P3,
      /* case 25 */ 7,
      ST_PNT, 5, E04, E05, E02, E00, E07,
      ST_TET, 4, P3, P0, P4, N0,
      ST_PYR, 5, E05, P3, P4, E04, N0,
      ST_PYR, 5, E05, E02, P0, P3, N0,
      ST_TET, 4, E02, E00, P0, N0,
      ST_PYR, 5, P0, E00, E07, P4, N0,
      ST_TET, 4, P4, E07, E04, N0,
      /* case 26 */ 7,
      ST_PNT, 5, E05, E04, E01, E00, E06,
      ST_TET, 4, P4, P3, P1, N0,
      ST_PYR, 5, E04, E05, P3, P4, N0,
      ST_PYR, 5, E04, P4, P1, E01, N0,
      ST_TET, 4, E01, P1, E00, N0,
      ST_PYR, 5, P1, P3, E06, E00, N0,
      ST_TET, 4, P3, E05, E06, N0,
      /* case 27 */ 1,
      ST_HEX, 8, P3, P4, E04, E05, P0, P1, E01, E02,
      /* case 28 */ 8,
      ST_PNT, 7, E02, E01, E07, E06, P2, P4, P3,
      ST_WDG, 6, E04, E05, E08, P4, P3, P2,
      ST_PYR, 5, E01, E07, E06, E02, N0,
      ST_PYR, 5, E06, E07, P4, P3, N0,
      ST_TET, 4, P2, P3, P4, N0,
      ST_TET, 4, P2, E01, E02, N0,
      ST_PYR, 5, E02, E06, P3, P2, N0,
      ST_PYR, 5, E07, E01, P2, P4, N0,
      /* case 29 */ 3,
      ST_WDG, 6, P2, P3, P4, E08, E05, E04,
      ST_TET, 4, P2, P3, P4, P0,
      ST_WDG, 6, P2, P4, P0, E01, E07, E00,
      /* case 30 */ 3,
      ST_WDG, 6, E06, E00, E02, P3, P1, P2,
      ST_TET, 4, P3, P2, P1, P4,
      ST_WDG, 6, E05, E08, E04, P3, P2, P4,
      /* case 31 */ 2,
      ST_WDG, 6, E08, E04, E05, P2, P4, P3,
      ST_PYR, 5, P0, P1, P4, P3, P2,
      /* case 32 */ 1,
      ST_TET, 4, E08, E04, E05, P5,
      /* case 33 */ 7,
      ST_PNT, 6, E02, E00, E06, E05, E04, E08,
      ST_PYR, 5, E08, E02, P0, P5, N0,
      ST_PYR, 5, P5, P0, E06, E05, N0,
      ST_TET, 4, P5, E05, E04, N0,
      ST_TET, 4, E04, E08, P5, N0,
      ST_TET, 4, P0, E00, E06, N0,
      ST_TET, 4, P0, E02, E00, N0,
      /* case 34 */ 7,
      ST_PNT, 6, E04, E05, E08, E01, E00, E07,
      ST_PYR, 5, E07, P1, P5, E04, N0,
      ST_PYR, 5, P1, E01, E08, P5, N0,
      ST_TET, 4, P1, E00, E01, N0,
      ST_TET, 4, E00, P1, E07, N0,
      ST_TET, 4, P5, E08, E05, N0,
      ST_TET, 4, P5, E05, E04, N0,
      /* case 35 */ 8,
      ST_PNT, 7, E05, E04, E07, E06, P5, P1, P0,
      ST_WDG, 6, P1, P0, P5, E01, E02, E08,
      ST_PYR, 5, E04, E05, E06, E07, N0,
      ST_PYR, 5, E06, P0, P1, E07, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_TET, 4, P5, E05, E04, N0,
      ST_PYR, 5, E05, P5, P0, E06, N0,
      ST_PYR, 5, E07, P1, P5, E04, N0,
      /* case 36 */ 1,
      ST_WDG, 6, P2, E02, E01, P5, E05, E04,
      /* case 37 */ 7,
      ST_PNT, 5, E00, E01, E04, E05, E06,
      ST_TET, 4, P2, P0, P5, N0,
      ST_PYR, 5, E01, E00, P0, P2, N0,
      ST_PYR, 5, E01, P2, P5, E04, N0,
      ST_TET, 4, E04, P5, E05, N0,
      ST_PYR, 5, P5, P0, E06, E05, N0,
      ST_TET, 4, P0, E00, E06, N0,
      /* case 38 */ 7,
      ST_PNT, 5, E00, E02, E05, E04, E07,
      ST_TET, 4, P2, P5, P1, N0,
      ST_PYR, 5, E02, P2, P1, E00, N0,
      ST_PYR, 5, E02, E05, P5, P2, N0,
      ST_TET, 4, E05, E04, P5, N0,
      ST_PYR, 5, P5, E04, E07, P1, N0,
      ST_TET, 4, P1, E07, E00, N0,
      /* case 39 */ 8,
      ST_PNT, 7, P0, P1, P5, E04, E05, E06, E07,
      ST_TET, 4, P2, P1, P0, P5,
      ST_PYR, 5, E04, E05, E06, E07, N0,
      ST_PYR, 5, E07, E06, P0, P1, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_TET, 4, P5, E05, E04, N0,
      ST_PYR, 5, P5, P0, E06, E05, N0,
      ST_PYR, 5, P1, P5, E04, E07, N0,
      /* case 40 */ 1,
      ST_WDG, 6, E03, E06, P3, E04, E08, P5,
      /* case 41 */ 7,
      ST_PNT, 5, E04, E03, E00, E02, E08,
      ST_TET, 4, P3, P5, P0, N0,
      ST_PYR, 5, E03, E04, P5, P3, N0,
      ST_PYR, 5, E03, P3, P0, E00, N0,
      ST_TET, 4, E00, P0, E02, N0,
      ST_PYR, 5, P0, P5, E08, E02, N0,
      ST_TET, 4, P5, E04, E08, N0,
      /* case 42 */ 8,
      ST_PNT, 7, E01, E00, E06, E08, P1, P3, P5,
      ST_WDG, 6, E03, E04, E07, P3, P5, P1,
      ST_PYR, 5, E00, E06, E08, E01, N0,
      ST_PYR, 5, E08, E06, P3, P5, N0,
      ST_TET, 4, P1, P5, P3, N0,
      ST_TET, 4, P1, E00, E01, N0,
      ST_PYR, 5, E01, E08, P5, P1, N0,
      ST_PYR, 5, E06, E00, P1, P3, N0,
      /* case 43 */ 3,
      ST_WDG, 6, E08, E02, E01, P5, P0, P1,
      ST_TET, 4, P5, P1, P0, P3,
      ST_WDG, 6, E04, E07, E03, P5, P1, P3,
      /* case 44 */ 7,
      ST_PNT, 5, E03, E04, E01, E02, E06,
      ST_TET, 4, P5, P2, P3, N0,
      ST_PYR, 5, E04, P5, P3, E03, N0,
      ST_PYR, 5, E04, E01, P2, P5, N0,
      ST_TET, 4, E01, E02, P2, N0,
      ST_PYR, 5, P2, E02, E06, P3, N0,
      ST_TET, 4, P3, E06, E03, N0,
      /* case 45 */ 1,
      ST_HEX, 8, P5, P3, E03, E04, P2, P0, E00, E01,
      /* case 46 */ 3,
      ST_WDG, 6, P1, P5, P3, E07, E04, E03,
      ST_TET, 4, P1, P5, P3, P2,
      ST_WDG, 6, P1, P3, P2, E00, E06, E02,
      /* case 47 */ 2,
      ST_WDG, 6, E07, E03, E04, P1, P3, P5,
      ST_PYR, 5, P2, P0, P3, P5, P1,
      /* case 48 */ 1,
      ST_WDG, 6, E05, E08, P5, E03, E07, P4,
      /* case 49 */ 8,
      ST_PNT, 7, E00, E02, E08, E07, P0, P5, P4,
      ST_WDG, 6, E05, E03, E06, P5, P4, P0,
      ST_PYR, 5, E02, E08, E07, E00, N0,
      ST_PYR, 5, E07, E08, P5, P4, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_TET, 4, P0, E02, E00, N0,
      ST_PYR, 5, E00, E07, P4, P0, N0,
      ST_PYR, 5, E08, E02, P0, P5, N0,
      /* case 50 */ 7,
      ST_PNT, 5, E05, E03, E00, E01, E08,
      ST_TET, 4, P4, P1, P5, N0,
      ST_PYR, 5, E03, P4, P5, E05, N0,
      ST_PYR, 5, E03, E00, P1, P4, N0,
      ST_TET, 4, E00, E01, P1, N0,
      ST_PYR, 5, P1, E01, E08, P5, N0,
      ST_TET, 4, P5, E08, E05, N0,
      /* case 51 */ 3,
      ST_WDG, 6, P0, P4, P5, E06, E03, E05,
      ST_TET, 4, P0, P4, P5, P1,
      ST_WDG, 6, P0, P5, P1, E02, E08, E01,
      /* case 52 */ 7,
      ST_PNT, 5, E03, E05, E02, E01, E07,
      ST_TET, 4, P5, P4, P2, N0,
      ST_PYR, 5, E05, E03, P4, P5, N0,
      ST_PYR, 5, E05, P5, P2, E02, N0,
      ST_TET, 4, E02, P2, E01, N0,
      ST_PYR, 5, P2, P4, E07, E01, N0,
      ST_TET, 4, P4, E03, E07, N0,
      /* case 53 */ 3,
      ST_WDG, 6, E07, E01, E00, P4, P2, P0,
      ST_TET, 4, P4, P0, P2, P5,
      ST_WDG, 6, E03, E06, E05, P4, P0, P5,
      /* case 54 */ 1,
      ST_HEX, 8, P4, P5, E05, E03, P1, P2, E02, E00,
      /* case 55 */ 2,
      ST_WDG, 6, E06, E05, E03, P0, P5, P4,
      ST_PYR, 5, P1, P2, P5, P4, P0,
      /* case 56 */ 1,
      ST_WDG, 6, E06, E07, E08, P3, P4, P5,
      /* case 57 */ 8,
      ST_PNT, 7, P4, P5, P0, E02, E00, E07, E08,
      ST_TET, 4, P3, P4, P5, P0,
      ST_PYR, 5, E02, E08, E07, E00, N0,
      ST_PYR, 5, E08, P5, P4, E07, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_TET, 4, P0, E02, E00, N0,
      ST_PYR, 5, P0, E00, E07, P4, N0,
      ST_PYR, 5, P5, E08, E02, P0, N0,
      /* case 58 */ 8,
      ST_PNT, 7, P5, P3, P1, E00, E01, E08, E06,
      ST_TET, 4, P4, P5, P3, P1,
      ST_PYR, 5, E00, E06, E08, E01, N0,
      ST_PYR, 5, E06, P3, P5, E08, N0,
      ST_TET, 4, P1, P5, P3, N0,
      ST_TET, 4, P1, E00, E01, N0,
      ST_PYR, 5, P1, E01, E08, P5, N0,
      ST_PYR, 5, P3, E06, E00, P1, N0,
      /* case 59 */ 2,
      ST_WDG, 6, P5, P1, P0, E08, E01, E02,
      ST_PYR, 5, P3, P0, P1, P4, P5,
      /* case 60 */ 8,
      ST_PNT, 7, P3, P4, P2, E01, E02, E06, E07,
      ST_TET, 4, P5, P3, P4, P2,
      ST_PYR, 5, E01, E07, E06, E02, N0,
      ST_PYR, 5, E07, P4, P3, E06, N0,
      ST_TET, 4, P2, P3, P4, N0,
      ST_TET, 4, P2, E01, E02, N0,
      ST_PYR, 5, P2, E02, E06, P3, N0,
      ST_PYR, 5, P4, E07, E01, P2, N0,
      /* case 61 */ 2,
      ST_WDG, 6, P4, P0, P2, E07, E00, E01,
      ST_PYR, 5, P5, P2, P0, P3, P4,
      /* case 62 */ 2,
      ST_WDG, 6, P3, P2, P1, E06, E02, E00,
      ST_PYR, 5, P4, P1, P2, P5, P3,
      /* case 63 */ 1,
      ST_WDG, 6, P0, P1, P2, P3, P4, P5,
      // vtkm::CELL_SHAPE_PYRAMID
      /* case 0 */ 0,
      /* case 1 */ 1,
      ST_TET, 4, P0, E00, E03, E04,
      /* case 2 */ 1,
      ST_TET, 4, P1, E01, E00, E05,
      /* case 3 */ 1,
      ST_WDG, 6, E01, E05, P1, E03, E04, P0,
      /* case 4 */ 1,
      ST_TET, 4, P2, E02, E01, E06,
      /* case 5 */ 2,
      ST_WDG, 6, P0, E00, E04, P2, E01, E06,
      ST_WDG, 6, E03, P0, E04, E02, P2, E06,
      /* case 6 */ 1,
      ST_WDG, 6, E02, E06, P2, E00, E05, P1,
      /* case 7 */ 2,
      ST_WDG, 6, E04, E05, E06, P0, P1, P2,
      ST_WDG, 6, P2, E02, E06, P0, E03, E04,
      /* case 8 */ 1,
      ST_TET, 4, P3, E03, E02, E07,
      /* case 9 */ 1,
      ST_WDG, 6, E00, E04, P0, E02, E07, P3,
      /* case 10 */ 2,
      ST_WDG, 6, P3, E03, E07, P1, E00, E05,
      ST_WDG, 6, E02, P3, E07, E01, P1, E05,
      /* case 11 */ 2,
      ST_WDG, 6, E07, E04, E05, P3, P0, P1,
      ST_WDG, 6, P1, E01, E05, P3, E02, E07,
      /* case 12 */ 1,
      ST_WDG, 6, E03, E07, P3, E01, E06, P2,
      /* case 13 */ 2,
      ST_WDG, 6, E06, E07, E04, P2, P3, P0,
      ST_WDG, 6, P0, E00, E04, P2, E01, E06,
      /* case 14 */ 2,
      ST_WDG, 6, E05, E06, E07, P1, P2, P3,
      ST_WDG, 6, P3, E03, E07, P1, E00, E05,
      /* case 15 */ 1,
      ST_HEX, 8, P0, P1, P2, P3, E04, E05, E06, E07,
      /* case 16 */ 1,
      ST_PYR, 5, E04, E05, E06, E07, P4,
      /* case 17 */ 2,
      ST_WDG, 6, P4, E05, E07, P0, E00, E03,
      ST_TET, 4, E05, E06, E07, P4,
      /* case 18 */ 2,
      ST_WDG, 6, P4, E06, E04, P1, E01, E00,
      ST_TET, 4, E06, E07, E04, P4,
      /* case 19 */ 7,
      ST_PNT, 7, E07, E06, E03, E01, P0, P1, P4,
      ST_PYR, 5, E06, E07, E03, E01, N0,
      ST_PYR, 5, E03, P0, P1, E01, N0,
      ST_TET, 4, P0, P4, P1, N0,
      ST_TET, 4, E07, E06, P4, N0,
      ST_PYR, 5, E07, P4, P0, E03, N0,
      ST_PYR, 5, P4, E06, E01, P1, N0,
      /* case 20 */ 2,
      ST_WDG, 6, P4, E07, E05, P2, E02, E01,
      ST_TET, 4, E07, E04, E05, P4,
      /* case 21 */ 2,
      ST_WDG, 6, E00, E01, E05, P0, P2, P4,
      ST_WDG, 6, E02, E03, E07, P2, P0, P4,
      /* case 22 */ 7,
      ST_PNT, 7, E04, E07, E00, E02, P1, P2, P4,
      ST_PYR, 5, E07, E04, E00, E02, N0,
      ST_PYR, 5, E00, P1, P2, E02, N0,
      ST_TET, 4, P1, P4, P2, N0,
      ST_TET, 4, E04, E07, P4, N0,
      ST_PYR, 5, E04, P4, P1, E00, N0,
      ST_PYR, 5, P4, E07, E02, P2, N0,
      /* case 23 */ 2,
      ST_WDG, 6, P0, P2, P4, E03, E02, E07,
      ST_TET, 4, P0, P1, P2, P4,
      /* case 24 */ 2,
      ST_WDG, 6, P4, E04, E06, P3, E03, E02,
      ST_TET, 4, E04, E05, E06, P4,
      /* case 25 */ 7,
      ST_PNT, 7, E06, E05, E02, E00, P3, P0, P4,
      ST_PYR, 5, E05, E06, E02, E00, N0,
      ST_PYR, 5, E02, P3, P0, E00, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_TET, 4, E06, E05, P4, N0,
      ST_PYR, 5, E06, P4, P3, E02, N0,
      ST_PYR, 5, P4, E05, E00, P0, N0,
      /* case 26 */ 2,
      ST_WDG, 6, E03, E00, E04, P3, P1, P4,
      ST_WDG, 6, E01, E02, E06, P1, P3, P4,
      /* case 27 */ 2,
      ST_WDG, 6, P3, P1, P4, E02, E01, E06,
      ST_TET, 4, P3, P0, P1, P4,
      /* case 28 */ 7,
      ST_PNT, 7, E05, E04, E01, E03, P2, P3, P4,
      ST_PYR, 5, E04, E05, E01, E03, N0,
      ST_PYR, 5, E01, P2, P3, E03, N0,
      ST_TET, 4, P2, P4, P3, N0,
      ST_TET, 4, E05, E04, P4, N0,
      ST_PYR, 5, E05, P4, P2, E01, N0,
      ST_PYR, 5, P4, E04, E03, P3, N0,
      /* case 29 */ 2,
      ST_WDG, 6, P2, P0, P4, E01, E00, E05,
      ST_TET, 4, P2, P3, P0, P4,
      /* case 30 */ 2,
      ST_WDG, 6, P1, P3, P4, E00, E03, E04,
      ST_TET, 4, P1, P2, P3, P4,
      /* case 31 */ 1,
      ST_PYR, 5, P0, P1, P2, P3, P4,
    };
    // clang-format on
    return ClipTablesData[idx];
  }

  /**
   * Get the the case index for the given shape and caseId.
   */
  VTKM_EXEC inline static vtkm::Id GetCaseIndex(vtkm::UInt8 shape, vtkm::Id caseId)
  {
    // clang-format off
    // offset into ShapeCases for each shape
    VTKM_STATIC_CONSTEXPR_ARRAY vtkm::UInt16 ClipTablesIndices[] = {
      // vtkm::CELL_SHAPE_VERTEX
      0, 1, // case 0 - 1
      // vtkm::CELL_SHAPE_LINE
      5, 6, 11, 16, // case 0 - 3
      // vtkm::CELL_SHAPE_TRIANGLE
      21, 22, 28, 34, 41, 47, 54, 61, // case 0 - 7
      // vtkm::CELL_SHAPE_QUAD
      67, 68, 74, 80, 87, 93, 106, 113, // case 0 - 7
      125, 131, 138, 151, 163, 170, 182, 194, // case 8 - 15
      // vtkm::CELL_SHAPE_TETRA
      201, 202, 209, 216, 225, 232, 241, 250, // case 0 - 7
      259, 266, 275, 284, 293, 302, 311, 320, // case 8 - 15
      // vtkm::CELL_SHAPE_HEXAHEDRON
      327, 328, 335, 342, 351, 358, 401, 410, // case 0 - 7
      457, 464, 473, 516, 563, 572, 619, 666, // case 8 - 15
      677, 684, 693, 736, 783, 796, 849, 902, // case 16 - 23
      963, 1006, 1053, 1090, 1125, 1178, 1239, 1301, // case 24 - 31
      1362, 1369, 1412, 1421, 1468, 1511, 1548, 1595, // case 32 - 39
      1630, 1643, 1696, 1749, 1810, 1863, 1925, 1986, // case 40 - 47
      2047, 2056, 2103, 2150, 2161, 2214, 2276, 2337, // case 48 - 55
      2398, 2451, 2512, 2574, 2635, 2656, 2732, 2808, // case 56 - 63
      2827, 2834, 2847, 2890, 2943, 2952, 3005, 3052, // case 64 - 71
      3113, 3156, 3209, 3246, 3308, 3355, 3416, 3451, // case 72 - 79
      3512, 3555, 3608, 3645, 3707, 3760, 3781, 3843, // case 80 - 87
      3919, 3956, 4018, 4057, 4094, 4156, 4232, 4269, // case 88 - 95
      4305, 4314, 4367, 4414, 4475, 4522, 4584, 4595, // case 96 - 103
      4656, 4709, 4730, 4792, 4868, 4929, 5005, 5066, // case 104 - 111
      5085, 5132, 5193, 5228, 5289, 5350, 5426, 5487, // case 112 - 119
      5506, 5568, 5644, 5681, 5717, 5793, 5834, 5870, // case 120 - 127
      5933, 5940, 5983, 5996, 6049, 6092, 6129, 6182, // case 128 - 135
      6244, 6253, 6300, 6353, 6414, 6461, 6496, 6557, // case 136 - 143
      6618, 6627, 6674, 6727, 6788, 6841, 6903, 6924, // case 144 - 151
      7000, 7047, 7058, 7120, 7181, 7242, 7303, 7379, // case 152 - 159
      7398, 7441, 7478, 7531, 7593, 7630, 7669, 7731, // case 160 - 167
      7768, 7821, 7883, 7904, 7980, 8042, 8079, 8155, // case 168 - 175
      8191, 8238, 8273, 8334, 8395, 8457, 8494, 8570, // case 176 - 183
      8606, 8667, 8728, 8804, 8823, 8899, 8935, 8976, // case 184 - 191
      9039, 9048, 9101, 9154, 9175, 9222, 9284, 9345, // case 192 - 199
      9421, 9468, 9529, 9591, 9667, 9678, 9739, 9800, // case 200 - 207
      9819, 9866, 9927, 9989, 10065, 10126, 10202, 10278, // case 208 - 215
      10319, 10354, 10415, 10452, 10488, 10549, 10568, 10604, // case 216 - 223
      10667, 10714, 10776, 10837, 10913, 10948, 10985, 11046, // case 224 - 231
      11082, 11143, 11219, 11295, 11336, 11397, 11433, 11452, // case 232 - 239
      11515, 11526, 11587, 11648, 11667, 11728, 11764, 11783, // case 240 - 247
      11846, 11907, 11926, 11962, 12025, 12044, 12107, 12170, // case 248 - 255
      // vtkm::CELL_SHAPE_WEDGE
      12181, 12182, 12189, 12196, 12205, 12212, 12221, 12230, // case 0 - 7
      12239, 12246, 12255, 12302, 12349, 12396, 12443, 12501, // case 8 - 15
      12557, 12564, 12611, 12620, 12667, 12714, 12772, 12819, // case 16 - 23
      12875, 12884, 12931, 12978, 12989, 13047, 13070, 13093, // case 24 - 31
      13109, 13116, 13163, 13210, 13268, 13277, 13324, 13371, // case 32 - 39
      13427, 13436, 13483, 13541, 13564, 13611, 13622, 13645, // case 40 - 47
      13661, 13670, 13728, 13775, 13798, 13845, 13868, 13879, // case 48 - 55
      13895, 13904, 13960, 14016, 14032, 14088, 14104, 14120, // case 56 - 63
      // vtkm::CELL_SHAPE_PYRAMID
      14129, 14130, 14137, 14144, 14153, 14160, 14177, 14186, // case 0 - 7
      14203, 14210, 14219, 14236, 14253, 14262, 14279, 14296, // case 8 - 15
      14307, 14315, 14330, 14345, 14395, 14410, 14427, 14477, // case 16 - 23
      14492, 14507, 14557, 14574, 14589, 14639, 14654, 14669, // case 24 - 31
    };
    // clang-format on
    return ClipTablesIndices[ClipTablesBase::GetCellClipTableIndex(shape) + caseId];
  }

  /**
   * Given the number of points and a case index, return if the cell is discarded.
   */
  VTKM_EXEC inline static constexpr bool IsCellDiscarded(
    vtkm::IdComponent vtkmNotUsed(numberOfPoints),
    vtkm::UInt8 caseIndex)
  {
    return caseIndex == 0;
  }

  /**
   * Given the number of points and a case index, return if the cell is kept.
   */
  VTKM_EXEC inline static bool IsCellKept(vtkm::IdComponent numberOfPoints, vtkm::UInt8 caseIndex)
  {
    return caseIndex == ClipTablesBase::GetMaxCellCase(numberOfPoints);
  }

  /**
   * Get the discarded cell case index.
   */
  VTKM_EXEC inline static constexpr vtkm::UInt8 GetDiscardedCellCase() { return 0; }

  /**
   * Get the kept cell case index.
   */
  VTKM_EXEC inline static constexpr vtkm::UInt8 GetKeptCellCase() { return 255; }
};


// Specialization for true
template <>
class ClipTables</*Invert=*/true> : public ClipTablesBase
{
public:
  ClipTables()
    : ClipTablesBase()
  {
  }

  /**
   * Get the value at the given index.
   */
  VTKM_EXEC inline static vtkm::UInt8 ValueAt(vtkm::Id idx)
  {
    using namespace ClipTablesInformation;
    // clang-format off
    // table format of an example case
    // number of shapes
    // {{shape-type, nverts, {point1, point2...}}, ...}, \n

    // clip table for all cases of each shape
    VTKM_STATIC_CONSTEXPR_ARRAY vtkm::UInt8 ClipTablesData[] = {
      // vtkm::CELL_SHAPE_VERTEX
      /* case 0 */ 1,
      ST_VTX, 1, P0,
      /* case 1 */ 0,
      // vtkm::CELL_SHAPE_LINE
      /* case 0 */ 1,
      ST_LIN, 2, P0, P1,
      /* case 1 */ 1,
      ST_LIN, 2, E00, P1,
      /* case 2 */ 1,
      ST_LIN, 2, E00, P0,
      /* case 3 */ 0,
      // vtkm::CELL_SHAPE_TRIANGLE
      /* case 0 */ 1,
      ST_TRI, 3, P0, P1, P2,
      /* case 1 */ 1,
      ST_QUA, 4, P1, P2, E02, E00,
      /* case 2 */ 1,
      ST_QUA, 4, P2, P0, E00, E01,
      /* case 3 */ 1,
      ST_TRI, 3, E02, E01, P2,
      /* case 4 */ 1,
      ST_QUA, 4, P0, P1, E01, E02,
      /* case 5 */ 1,
      ST_TRI, 3, E01, E00, P1,
      /* case 6 */ 1,
      ST_TRI, 3, E00, E02, P0,
      /* case 7 */ 0,
      // vtkm::CELL_SHAPE_QUAD
      /* case 0 */ 1,
      ST_QUA, 4, P0, P1, P2, P3,
      /* case 1 */ 2,
      ST_QUA, 4, E03, E00, P1, P3,
      ST_TRI, 3, P3, P1, P2,
      /* case 2 */ 2,
      ST_QUA, 4, E00, E01, P2, P0,
      ST_TRI, 3, P0, P2, P3,
      /* case 3 */ 1,
      ST_QUA, 4, E03, E01, P2, P3,
      /* case 4 */ 2,
      ST_QUA, 4, E01, E02, P3, P1,
      ST_TRI, 3, P1, P3, P0,
      /* case 5 */ 2,
      ST_TRI, 3, E03, E02, P3,
      ST_TRI, 3, E01, E00, P1,
      /* case 6 */ 1,
      ST_QUA, 4, E00, E02, P3, P0,
      /* case 7 */ 1,
      ST_TRI, 3, E03, E02, P3,
      /* case 8 */ 2,
      ST_QUA, 4, E02, E03, P0, P2,
      ST_TRI, 3, P2, P0, P1,
      /* case 9 */ 1,
      ST_QUA, 4, E02, E00, P1, P2,
      /* case 10 */ 2,
      ST_TRI, 3, E00, E03, P0,
      ST_TRI, 3, E02, E01, P2,
      /* case 11 */ 1,
      ST_TRI, 3, E02, E01, P2,
      /* case 12 */ 1,
      ST_QUA, 4, E01, E03, P0, P1,
      /* case 13 */ 1,
      ST_TRI, 3, E01, E00, P1,
      /* case 14 */ 1,
      ST_TRI, 3, E00, E03, P0,
      /* case 15 */ 0,
      // vtkm::CELL_SHAPE_TETRA
      /* case 0 */ 1,
      ST_TET, 4, P0, P1, P2, P3,
      /* case 1 */ 1,
      ST_WDG, 6, E00, E03, E02, P1, P3, P2,
      /* case 2 */ 1,
      ST_WDG, 6, P0, P3, P2, E00, E04, E01,
      /* case 3 */ 1,
      ST_WDG, 6, P3, E03, E04, P2, E02, E01,
      /* case 4 */ 1,
      ST_WDG, 6, E02, E05, E01, P0, P3, P1,
      /* case 5 */ 1,
      ST_WDG, 6, P1, E00, E01, P3, E03, E05,
      /* case 6 */ 1,
      ST_WDG, 6, P3, E04, E05, P0, E00, E02,
      /* case 7 */ 1,
      ST_TET, 4, E03, E04, E05, P3,
      /* case 8 */ 1,
      ST_WDG, 6, P0, P2, P1, E03, E05, E04,
      /* case 9 */ 1,
      ST_WDG, 6, P2, E02, E05, P1, E00, E04,
      /* case 10 */ 1,
      ST_WDG, 6, P0, E00, E03, P2, E01, E05,
      /* case 11 */ 1,
      ST_TET, 4, E02, E05, E01, P2,
      /* case 12 */ 1,
      ST_WDG, 6, P1, E01, E04, P0, E02, E03,
      /* case 13 */ 1,
      ST_TET, 4, E00, E01, E04, P1,
      /* case 14 */ 1,
      ST_TET, 4, E00, E03, E02, P0,
      /* case 15 */ 0,
      // vtkm::CELL_SHAPE_HEXAHEDRON
      /* case 0 */ 1,
      ST_HEX, 8, P0, P1, P2, P3, P4, P5, P6, P7,
      /* case 1 */ 9,
      ST_PNT, 7, P1, P2, P3, P4, P5, P6, P7,
      ST_WDG, 6, P1, P3, P4, E00, E03, E08,
      ST_TET, 4, P1, P3, P4, N0,
      ST_TET, 4, P1, P2, P3, N0,
      ST_PYR, 5, P6, P7, P3, P2, N0,
      ST_PYR, 5, P5, P6, P2, P1, N0,
      ST_PYR, 5, P4, P7, P6, P5, N0,
      ST_TET, 4, P3, P7, P4, N0,
      ST_TET, 4, P4, P5, P1, N0,
      /* case 2 */ 9,
      ST_PNT, 7, P5, P4, P0, P2, P6, P7, P3,
      ST_WDG, 6, E09, E00, E01, P5, P0, P2,
      ST_TET, 4, P5, P2, P0, N0,
      ST_TET, 4, P5, P0, P4, N0,
      ST_PYR, 5, P7, P4, P0, P3, N0,
      ST_PYR, 5, P6, P5, P4, P7, N0,
      ST_PYR, 5, P2, P6, P7, P3, N0,
      ST_TET, 4, P0, P2, P3, N0,
      ST_TET, 4, P2, P5, P6, N0,
      /* case 3 */ 2,
      ST_HEX, 8, E01, P2, P3, E03, E09, P5, P4, E08,
      ST_WDG, 6, P2, P6, P5, P3, P7, P4,
      /* case 4 */ 9,
      ST_PNT, 7, P6, P5, P1, P3, P7, P4, P0,
      ST_WDG, 6, E11, E01, E02, P6, P1, P3,
      ST_TET, 4, P6, P3, P1, N0,
      ST_TET, 4, P6, P1, P5, N0,
      ST_PYR, 5, P4, P5, P1, P0, N0,
      ST_PYR, 5, P7, P6, P5, P4, N0,
      ST_PYR, 5, P3, P7, P4, P0, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_TET, 4, P3, P6, P7, N0,
      /* case 5 */ 12,
      ST_PNT, 2, E08, E11,
      ST_PYR, 5, P4, P7, P6, P5, N0,
      ST_TET, 4, P5, P6, P1, N0,
      ST_TET, 4, P4, P5, P1, N0,
      ST_TET, 4, P3, P7, P4, N0,
      ST_TET, 4, P6, P7, P3, N0,
      ST_PYR, 5, P6, P3, E02, E11, N0,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_TET, 4, P1, E01, E00, N0,
      ST_PYR, 5, P4, P1, E00, E08, N0,
      ST_PYR, 5, P4, E08, E03, P3, N0,
      ST_TET, 4, P3, E03, E02, N0,
      /* case 6 */ 2,
      ST_HEX, 8, E02, P3, P0, E00, E11, P6, P5, E09,
      ST_WDG, 6, P3, P7, P6, P0, P4, P5,
      /* case 7 */ 9,
      ST_PNT, 5, E08, E09, E03, E02, E11,
      ST_PYR, 5, P4, P7, P6, P5, N0,
      ST_TET, 4, P6, P3, N0, P7,
      ST_PYR, 5, P5, P6, E11, E09, N0,
      ST_PYR, 5, E08, P4, P5, E09, N0,
      ST_TET, 4, P3, P7, P4, N0,
      ST_PYR, 5, P3, P4, E08, E03, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_PYR, 5, E11, P6, P3, E02, N0,
      /* case 8 */ 9,
      ST_PNT, 7, P2, P1, P0, P7, P6, P5, P4,
      ST_WDG, 6, E02, E03, E10, P2, P0, P7,
      ST_TET, 4, P2, P7, P0, N0,
      ST_TET, 4, P2, P0, P1, N0,
      ST_PYR, 5, P5, P1, P0, P4, N0,
      ST_PYR, 5, P6, P2, P1, P5, N0,
      ST_PYR, 5, P7, P6, P5, P4, N0,
      ST_TET, 4, P0, P7, P4, N0,
      ST_TET, 4, P7, P2, P6, N0,
      /* case 9 */ 2,
      ST_HEX, 8, E10, P7, P4, E08, E02, P2, P1, E00,
      ST_WDG, 6, P7, P6, P2, P4, P5, P1,
      /* case 10 */ 12,
      ST_PNT, 2, E10, E09,
      ST_PYR, 5, P7, P6, P5, P4, N0,
      ST_TET, 4, P6, P2, P5, N0,
      ST_TET, 4, P7, P2, P6, N0,
      ST_TET, 4, P0, P7, P4, N0,
      ST_TET, 4, P5, P0, P4, N0,
      ST_PYR, 5, P5, E09, E00, P0, N0,
      ST_PYR, 5, P2, E01, E09, P5, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_PYR, 5, P7, E10, E02, P2, N0,
      ST_PYR, 5, P7, P0, E03, E10, N0,
      ST_TET, 4, P0, E00, E03, N0,
      /* case 11 */ 9,
      ST_PNT, 5, E09, E08, E01, E02, E10,
      ST_PYR, 5, P5, P4, P7, P6, N0,
      ST_TET, 4, P7, N0, P2, P6,
      ST_PYR, 5, P4, E08, E10, P7, N0,
      ST_PYR, 5, E09, E08, P4, P5, N0,
      ST_TET, 4, P2, P5, P6, N0,
      ST_PYR, 5, P2, E01, E09, P5, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_PYR, 5, E10, E02, P2, P7, N0,
      /* case 12 */ 2,
      ST_HEX, 8, E11, P6, P7, E10, E01, P1, P0, E03,
      ST_WDG, 6, P0, P4, P7, P1, P5, P6,
      /* case 13 */ 9,
      ST_PNT, 5, E08, E10, E00, E01, E11,
      ST_PYR, 5, P4, P7, P6, P5, N0,
      ST_TET, 4, P6, N0, P1, P5,
      ST_PYR, 5, P7, E10, E11, P6, N0,
      ST_PYR, 5, E08, E10, P7, P4, N0,
      ST_TET, 4, P1, P4, P5, N0,
      ST_PYR, 5, P1, E00, E08, P4, N0,
      ST_TET, 4, P1, E01, E00, N0,
      ST_PYR, 5, E11, E01, P1, P6, N0,
      /* case 14 */ 9,
      ST_PNT, 5, E10, E11, E03, E00, E09,
      ST_PYR, 5, P7, P6, P5, P4, N0,
      ST_TET, 4, P5, N0, P0, P4,
      ST_PYR, 5, P6, E11, E09, P5, N0,
      ST_PYR, 5, E10, E11, P6, P7, N0,
      ST_TET, 4, P0, P7, P4, N0,
      ST_PYR, 5, P0, E03, E10, P7, N0,
      ST_TET, 4, P0, E00, E03, N0,
      ST_PYR, 5, E09, E00, P0, P5, N0,
      /* case 15 */ 1,
      ST_HEX, 8, E08, E09, E11, E10, P4, P5, P6, P7,
      /* case 16 */ 9,
      ST_PNT, 7, P5, P1, P0, P7, P6, P2, P3,
      ST_WDG, 6, P5, P0, P7, E04, E08, E07,
      ST_TET, 4, P5, P0, P7, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_PYR, 5, P2, P3, P0, P1, N0,
      ST_PYR, 5, P6, P2, P1, P5, N0,
      ST_PYR, 5, P7, P3, P2, P6, N0,
      ST_TET, 4, P0, P3, P7, N0,
      ST_TET, 4, P7, P6, P5, N0,
      /* case 17 */ 2,
      ST_HEX, 8, E04, P5, P1, E00, E07, P7, P3, E03,
      ST_WDG, 6, P3, P2, P1, P7, P6, P5,
      /* case 18 */ 12,
      ST_PNT, 2, E07, E01,
      ST_PYR, 5, P7, P3, P2, P6, N0,
      ST_TET, 4, P6, P2, P5, N0,
      ST_TET, 4, P7, P6, P5, N0,
      ST_TET, 4, P0, P3, P7, N0,
      ST_TET, 4, P2, P3, P0, N0,
      ST_PYR, 5, P2, P0, E00, E01, N0,
      ST_PYR, 5, P5, P2, E01, E09, N0,
      ST_TET, 4, P5, E09, E04, N0,
      ST_PYR, 5, P7, P5, E04, E07, N0,
      ST_PYR, 5, P7, E07, E08, P0, N0,
      ST_TET, 4, P0, E08, E00, N0,
      /* case 19 */ 9,
      ST_PNT, 5, E01, E03, E09, E04, E07,
      ST_PYR, 5, P2, P6, P7, P3, N0,
      ST_TET, 4, P7, P5, N0, P6,
      ST_PYR, 5, P3, P7, E07, E03, N0,
      ST_PYR, 5, E01, P2, P3, E03, N0,
      ST_TET, 4, P5, P6, P2, N0,
      ST_PYR, 5, P5, P2, E01, E09, N0,
      ST_TET, 4, P5, E09, E04, N0,
      ST_PYR, 5, E07, P7, P5, E04, N0,
      /* case 20 */ 6,
      ST_WDG, 6, E01, E02, E11, P1, P3, P6,
      ST_WDG, 6, P0, P7, P5, E08, E07, E04,
      ST_TET, 4, P3, P1, P6, P7,
      ST_TET, 4, P5, P7, P6, P1,
      ST_TET, 4, P0, P5, P1, P7,
      ST_TET, 4, P3, P7, P0, P1,
      /* case 21 */ 11,
      ST_PNT, 4, E04, E07, E11, E11,
      ST_PYR, 5, P6, P3, E02, E11, N0,
      ST_TET, 4, E02, P3, E03, N0,
      ST_PYR, 5, P7, E07, E03, P3, N0,
      ST_TET, 4, P6, P7, P3, N0,
      ST_TET, 4, P1, E01, E00, N0,
      ST_TET, 4, P5, P6, P1, N0,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_TET, 4, P5, P7, P6, N0,
      ST_PYR, 5, P5, E04, E07, P7, N0,
      ST_PYR, 5, P5, P1, E00, E04, N0,
      /* case 22 */ 11,
      ST_PNT, 4, E11, E02, E07, E07,
      ST_PYR, 5, P7, E07, E08, P0, N0,
      ST_TET, 4, E08, E00, P0, N0,
      ST_PYR, 5, P3, P0, E00, E02, N0,
      ST_TET, 4, P7, P0, P3, N0,
      ST_TET, 4, P5, E09, E04, N0,
      ST_TET, 4, P6, P5, P7, N0,
      ST_PYR, 5, P5, E04, E07, P7, N0,
      ST_TET, 4, P6, P7, P3, N0,
      ST_PYR, 5, P6, P3, E02, E11, N0,
      ST_PYR, 5, P6, E11, E09, P5, N0,
      /* case 23 */ 9,
      ST_PNT, 6, E03, E02, E11, E09, E04, E07,
      ST_TET, 4, P6, P5, P7, N0,
      ST_PYR, 5, P6, E11, E09, P5, N0,
      ST_TET, 4, P3, P6, P7, N0,
      ST_PYR, 5, P3, E02, E11, P6, N0,
      ST_TET, 4, E03, E02, P3, N0,
      ST_PYR, 5, P3, P7, E07, E03, N0,
      ST_PYR, 5, E07, P7, P5, E04, N0,
      ST_TET, 4, P5, E09, E04, N0,
      /* case 24 */ 12,
      ST_PNT, 2, E02, E04,
      ST_PYR, 5, P2, P1, P5, P6, N0,
      ST_TET, 4, P6, P5, P7, N0,
      ST_TET, 4, P2, P6, P7, N0,
      ST_TET, 4, P0, P1, P2, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_PYR, 5, P5, P0, E08, E04, N0,
      ST_PYR, 5, P7, P5, E04, E07, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_PYR, 5, P2, P7, E10, E02, N0,
      ST_PYR, 5, P2, E02, E03, P0, N0,
      ST_TET, 4, P0, E03, E08, N0,
      /* case 25 */ 9,
      ST_PNT, 5, E04, E00, E07, E10, E02,
      ST_PYR, 5, P5, P6, P2, P1, N0,
      ST_TET, 4, P2, P7, N0, P6,
      ST_PYR, 5, P1, P2, E02, E00, N0,
      ST_PYR, 5, E04, P5, P1, E00, N0,
      ST_TET, 4, P7, P6, P5, N0,
      ST_PYR, 5, P7, P5, E04, E07, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_PYR, 5, E02, P2, P7, E10, N0,
      /* case 26 */ 6,
      ST_TET, 4, P0, E00, E03, E08,
      ST_TET, 4, P5, P7, P6, P2,
      ST_PYR, 5, E02, P2, P7, E10, E07,
      ST_PYR, 5, E01, E09, P5, P2, E04,
      ST_PYR, 5, P7, P5, E04, E07, P2,
      ST_PYR, 5, E07, E04, E01, E02, P2,
      /* case 27 */ 5,
      ST_TET, 4, P5, P7, P6, P2,
      ST_PYR, 5, E02, P2, P7, E10, E07,
      ST_PYR, 5, E01, E09, P5, P2, E04,
      ST_PYR, 5, P7, P5, E04, E07, P2,
      ST_PYR, 5, E07, E04, E01, E02, P2,
      /* case 28 */ 11,
      ST_PNT, 4, E11, E01, E04, E04,
      ST_PYR, 5, P5, P0, E08, E04, N0,
      ST_TET, 4, E08, P0, E03, N0,
      ST_PYR, 5, P1, E01, E03, P0, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_TET, 4, P6, P5, P7, N0,
      ST_PYR, 5, P7, P5, E04, E07, N0,
      ST_TET, 4, P6, P1, P5, N0,
      ST_PYR, 5, P6, E11, E01, P1, N0,
      ST_PYR, 5, P6, P7, E10, E11, N0,
      /* case 29 */ 9,
      ST_PNT, 6, E00, E01, E11, E10, E07, E04,
      ST_TET, 4, P6, P5, P7, N0,
      ST_PYR, 5, P6, P7, E10, E11, N0,
      ST_TET, 4, P1, P5, P6, N0,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_TET, 4, E00, P1, E01, N0,
      ST_PYR, 5, P1, E00, E04, P5, N0,
      ST_PYR, 5, E04, E07, P7, P5, N0,
      ST_TET, 4, P7, E07, E10, N0,
      /* case 30 */ 8,
      ST_PNT, 5, E11, E09, E10, E07, E04,
      ST_TET, 4, P0, E00, E03, E08,
      ST_PYR, 5, P5, P6, E11, E09, N0,
      ST_PYR, 5, P6, P7, E10, E11, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_TET, 4, P6, P5, P7, N0,
      ST_PYR, 5, P7, P5, E04, E07, N0,
      ST_TET, 4, P5, E09, E04, N0,
      /* case 31 */ 7,
      ST_PNT, 5, E09, E11, E10, E04, E07,
      ST_PYR, 5, P6, P7, E10, E11, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_PYR, 5, P5, P6, E11, E09, N0,
      ST_TET, 4, E04, P5, E09, N0,
      ST_PYR, 5, E07, P7, P5, E04, N0,
      ST_TET, 4, P7, P6, P5, N0,
      /* case 32 */ 9,
      ST_PNT, 7, P6, P2, P1, P4, P7, P3, P0,
      ST_WDG, 6, P6, P1, P4, E05, E09, E04,
      ST_TET, 4, P6, P1, P4, N0,
      ST_TET, 4, P6, P2, P1, N0,
      ST_PYR, 5, P3, P0, P1, P2, N0,
      ST_PYR, 5, P7, P3, P2, P6, N0,
      ST_PYR, 5, P4, P0, P3, P7, N0,
      ST_TET, 4, P1, P0, P4, N0,
      ST_TET, 4, P4, P7, P6, N0,
      /* case 33 */ 12,
      ST_PNT, 2, E03, E05,
      ST_PYR, 5, P3, P2, P6, P7, N0,
      ST_TET, 4, P2, P1, P6, N0,
      ST_TET, 4, P3, P1, P2, N0,
      ST_TET, 4, P4, P3, P7, N0,
      ST_TET, 4, P6, P4, P7, N0,
      ST_PYR, 5, P6, E05, E04, P4, N0,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_PYR, 5, P3, E03, E00, P1, N0,
      ST_PYR, 5, P3, P4, E08, E03, N0,
      ST_TET, 4, P4, E04, E08, N0,
      /* case 34 */ 2,
      ST_HEX, 8, E05, P6, P2, E01, E04, P4, P0, E00,
      ST_WDG, 6, P0, P3, P2, P4, P7, P6,
      /* case 35 */ 9,
      ST_PNT, 5, E03, E01, E08, E04, E05,
      ST_PYR, 5, P3, P2, P6, P7, N0,
      ST_TET, 4, P6, N0, P4, P7,
      ST_PYR, 5, P2, E01, E05, P6, N0,
      ST_PYR, 5, E03, E01, P2, P3, N0,
      ST_TET, 4, P4, P3, P7, N0,
      ST_PYR, 5, P4, E08, E03, P3, N0,
      ST_TET, 4, P4, E04, E08, N0,
      ST_PYR, 5, E05, E04, P4, P6, N0,
      /* case 36 */ 12,
      ST_PNT, 2, E02, E04,
      ST_PYR, 5, P3, P7, P4, P0, N0,
      ST_TET, 4, P7, P6, P4, N0,
      ST_TET, 4, P3, P6, P7, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_TET, 4, P4, P1, P0, N0,
      ST_PYR, 5, P4, E04, E09, P1, N0,
      ST_PYR, 5, P6, E05, E04, P4, N0,
      ST_TET, 4, P6, E11, E05, N0,
      ST_PYR, 5, P3, E02, E11, P6, N0,
      ST_PYR, 5, P3, P1, E01, E02, N0,
      ST_TET, 4, P1, E09, E01, N0,
      /* case 37 */ 6,
      ST_TET, 4, P1, E00, E09, E01,
      ST_TET, 4, P6, P7, P3, P4,
      ST_PYR, 5, E08, E03, P3, P4, E02,
      ST_PYR, 5, E04, P4, P6, E05, E11,
      ST_PYR, 5, P3, E02, E11, P6, P4,
      ST_PYR, 5, E02, E08, E04, E11, P4,
      /* case 38 */ 9,
      ST_PNT, 5, E04, E00, E05, E11, E02,
      ST_PYR, 5, P4, P0, P3, P7, N0,
      ST_TET, 4, P3, N0, P6, P7,
      ST_PYR, 5, P0, E00, E02, P3, N0,
      ST_PYR, 5, E04, E00, P0, P4, N0,
      ST_TET, 4, P6, P4, P7, N0,
      ST_PYR, 5, P6, E05, E04, P4, N0,
      ST_TET, 4, P6, E11, E05, N0,
      ST_PYR, 5, E02, E11, P6, P3, N0,
      /* case 39 */ 5,
      ST_TET, 4, P6, P7, P3, P4,
      ST_PYR, 5, E08, E03, P3, P4, E02,
      ST_PYR, 5, E04, P4, P6, E05, E11,
      ST_PYR, 5, P3, E02, E11, P6, P4,
      ST_PYR, 5, E02, E08, E04, E11, P4,
      /* case 40 */ 6,
      ST_WDG, 6, P1, P4, P6, E09, E04, E05,
      ST_WDG, 6, E03, E10, E02, P0, P7, P2,
      ST_TET, 4, P4, P6, P1, P7,
      ST_TET, 4, P2, P6, P7, P1,
      ST_TET, 4, P0, P1, P2, P7,
      ST_TET, 4, P4, P0, P7, P1,
      /* case 41 */ 11,
      ST_PNT, 4, E02, E10, E05, E05,
      ST_PYR, 5, P6, E05, E04, P4, N0,
      ST_TET, 4, E04, E08, P4, N0,
      ST_PYR, 5, P7, P4, E08, E10, N0,
      ST_TET, 4, P6, P4, P7, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_TET, 4, P2, P1, P6, N0,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_TET, 4, P2, P6, P7, N0,
      ST_PYR, 5, P2, P7, E10, E02, N0,
      ST_PYR, 5, P2, E02, E00, P1, N0,
      /* case 42 */ 11,
      ST_PNT, 4, E05, E04, E10, E10,
      ST_PYR, 5, P7, P0, E03, E10, N0,
      ST_TET, 4, E03, P0, E00, N0,
      ST_PYR, 5, P4, E04, E00, P0, N0,
      ST_TET, 4, P7, P4, P0, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_TET, 4, P6, P7, P2, N0,
      ST_PYR, 5, P2, P7, E10, E02, N0,
      ST_TET, 4, P6, P4, P7, N0,
      ST_PYR, 5, P6, E05, E04, P4, N0,
      ST_PYR, 5, P6, P2, E01, E05, N0,
      /* case 43 */ 9,
      ST_PNT, 6, E08, E04, E05, E01, E02, E10,
      ST_TET, 4, P6, P7, P2, N0,
      ST_PYR, 5, P6, P2, E01, E05, N0,
      ST_TET, 4, P4, P7, P6, N0,
      ST_PYR, 5, P4, P6, E05, E04, N0,
      ST_TET, 4, E08, P4, E04, N0,
      ST_PYR, 5, P4, E08, E10, P7, N0,
      ST_PYR, 5, E10, E02, P2, P7, N0,
      ST_TET, 4, P2, E02, E01, N0,
      /* case 44 */ 11,
      ST_PNT, 4, E10, E03, E04, E04,
      ST_PYR, 5, P4, E04, E09, P1, N0,
      ST_TET, 4, E09, E01, P1, N0,
      ST_PYR, 5, P0, P1, E01, E03, N0,
      ST_TET, 4, P4, P1, P0, N0,
      ST_TET, 4, P6, E11, E05, N0,
      ST_TET, 4, P7, P6, P4, N0,
      ST_PYR, 5, P6, E05, E04, P4, N0,
      ST_TET, 4, P7, P4, P0, N0,
      ST_PYR, 5, P7, P0, E03, E10, N0,
      ST_PYR, 5, P7, E10, E11, P6, N0,
      /* case 45 */ 8,
      ST_PNT, 5, E10, E11, E08, E04, E05,
      ST_TET, 4, P1, E01, E00, E09,
      ST_PYR, 5, P6, P7, E10, E11, N0,
      ST_PYR, 5, P7, P4, E08, E10, N0,
      ST_TET, 4, P4, E04, E08, N0,
      ST_TET, 4, P7, P6, P4, N0,
      ST_PYR, 5, P4, P6, E05, E04, N0,
      ST_TET, 4, P6, E11, E05, N0,
      /* case 46 */ 9,
      ST_PNT, 6, E00, E03, E10, E11, E05, E04,
      ST_TET, 4, P7, P6, P4, N0,
      ST_PYR, 5, P7, E10, E11, P6, N0,
      ST_TET, 4, P0, P7, P4, N0,
      ST_PYR, 5, P0, E03, E10, P7, N0,
      ST_TET, 4, E00, E03, P0, N0,
      ST_PYR, 5, P0, P4, E04, E00, N0,
      ST_PYR, 5, E04, P4, P6, E05, N0,
      ST_TET, 4, P6, E11, E05, N0,
      /* case 47 */ 7,
      ST_PNT, 5, E11, E10, E08, E05, E04,
      ST_PYR, 5, P7, P4, E08, E10, N0,
      ST_TET, 4, P4, E04, E08, N0,
      ST_PYR, 5, P6, P7, E10, E11, N0,
      ST_TET, 4, E05, P6, E11, N0,
      ST_PYR, 5, E04, P4, P6, E05, N0,
      ST_TET, 4, P4, P7, P6, N0,
      /* case 48 */ 2,
      ST_HEX, 8, E09, P1, P0, E08, E05, P6, P7, E07,
      ST_WDG, 6, P1, P2, P6, P0, P3, P7,
      /* case 49 */ 9,
      ST_PNT, 5, E03, E07, E00, E09, E05,
      ST_PYR, 5, P3, P2, P6, P7, N0,
      ST_TET, 4, P6, P1, N0, P2,
      ST_PYR, 5, P7, P6, E05, E07, N0,
      ST_PYR, 5, E03, P3, P7, E07, N0,
      ST_TET, 4, P1, P2, P3, N0,
      ST_PYR, 5, P1, P3, E03, E00, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_PYR, 5, E05, P6, P1, E09, N0,
      /* case 50 */ 9,
      ST_PNT, 5, E07, E05, E08, E00, E01,
      ST_PYR, 5, P7, P3, P2, P6, N0,
      ST_TET, 4, P2, P0, N0, P3,
      ST_PYR, 5, P6, P2, E01, E05, N0,
      ST_PYR, 5, E07, P7, P6, E05, N0,
      ST_TET, 4, P0, P3, P7, N0,
      ST_PYR, 5, P0, P7, E07, E08, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_PYR, 5, E01, P2, P0, E00, N0,
      /* case 51 */ 1,
      ST_HEX, 8, P3, P2, P6, P7, E03, E01, E05, E07,
      /* case 52 */ 11,
      ST_PNT, 4, E07, E08, E02, E02,
      ST_PYR, 5, P3, P1, E01, E02, N0,
      ST_TET, 4, E01, P1, E09, N0,
      ST_PYR, 5, P0, E08, E09, P1, N0,
      ST_TET, 4, P3, P0, P1, N0,
      ST_TET, 4, P6, E11, E05, N0,
      ST_TET, 4, P7, P3, P6, N0,
      ST_PYR, 5, P6, P3, E02, E11, N0,
      ST_TET, 4, P7, P0, P3, N0,
      ST_PYR, 5, P7, E07, E08, P0, N0,
      ST_PYR, 5, P7, P6, E05, E07, N0,
      /* case 53 */ 8,
      ST_PNT, 5, E07, E05, E03, E02, E11,
      ST_TET, 4, P1, E00, E09, E01,
      ST_PYR, 5, P6, E05, E07, P7, N0,
      ST_PYR, 5, P7, E07, E03, P3, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_TET, 4, P7, P3, P6, N0,
      ST_PYR, 5, P3, E02, E11, P6, N0,
      ST_TET, 4, P6, E11, E05, N0,
      /* case 54 */ 9,
      ST_PNT, 6, E00, E08, E07, E05, E11, E02,
      ST_TET, 4, P7, P3, P6, N0,
      ST_PYR, 5, P7, P6, E05, E07, N0,
      ST_TET, 4, P0, P3, P7, N0,
      ST_PYR, 5, P0, P7, E07, E08, N0,
      ST_TET, 4, E00, P0, E08, N0,
      ST_PYR, 5, P0, E00, E02, P3, N0,
      ST_PYR, 5, E02, E11, P6, P3, N0,
      ST_TET, 4, P6, E11, E05, N0,
      /* case 55 */ 7,
      ST_PNT, 5, E05, E07, E03, E11, E02,
      ST_PYR, 5, P7, E07, E03, P3, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_PYR, 5, P6, E05, E07, P7, N0,
      ST_TET, 4, E11, E05, P6, N0,
      ST_PYR, 5, E02, E11, P6, P3, N0,
      ST_TET, 4, P3, P6, P7, N0,
      /* case 56 */ 11,
      ST_PNT, 4, E05, E09, E02, E02,
      ST_PYR, 5, P2, E02, E03, P0, N0,
      ST_TET, 4, E03, E08, P0, N0,
      ST_PYR, 5, P1, P0, E08, E09, N0,
      ST_TET, 4, P2, P0, P1, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_TET, 4, P6, P7, P2, N0,
      ST_PYR, 5, P7, E10, E02, P2, N0,
      ST_TET, 4, P6, P2, P1, N0,
      ST_PYR, 5, P6, P1, E09, E05, N0,
      ST_PYR, 5, P6, E05, E07, P7, N0,
      /* case 57 */ 9,
      ST_PNT, 6, E00, E09, E05, E07, E10, E02,
      ST_TET, 4, P6, P7, P2, N0,
      ST_PYR, 5, P6, E05, E07, P7, N0,
      ST_TET, 4, P1, P6, P2, N0,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_TET, 4, E00, E09, P1, N0,
      ST_PYR, 5, P1, P2, E02, E00, N0,
      ST_PYR, 5, E02, P2, P7, E10, N0,
      ST_TET, 4, P7, E07, E10, N0,
      /* case 58 */ 8,
      ST_PNT, 5, E05, E01, E07, E10, E02,
      ST_TET, 4, P0, E08, E00, E03,
      ST_PYR, 5, P2, E01, E05, P6, N0,
      ST_PYR, 5, P6, E05, E07, P7, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_TET, 4, P6, P7, P2, N0,
      ST_PYR, 5, P7, E10, E02, P2, N0,
      ST_TET, 4, P2, E02, E01, N0,
      /* case 59 */ 7,
      ST_PNT, 5, E01, E05, E07, E02, E10,
      ST_PYR, 5, P6, E05, E07, P7, N0,
      ST_TET, 4, P7, E07, E10, N0,
      ST_PYR, 5, P2, E01, E05, P6, N0,
      ST_TET, 4, E02, E01, P2, N0,
      ST_PYR, 5, E10, E02, P2, P7, N0,
      ST_TET, 4, P7, P2, P6, N0,
      /* case 60 */ 2,
      ST_WDG, 6, P1, E01, E09, P0, E03, E08,
      ST_WDG, 6, P6, E05, E11, P7, E07, E10,
      /* case 61 */ 2,
      ST_WDG, 6, E07, P7, E10, E05, P6, E11,
      ST_TET, 4, E00, P1, E01, E09,
      /* case 62 */ 2,
      ST_WDG, 6, E07, P7, E10, E05, P6, E11,
      ST_TET, 4, E00, E03, P0, E08,
      /* case 63 */ 1,
      ST_WDG, 6, P7, E10, E07, P6, E11, E05,
      /* case 64 */ 9,
      ST_PNT, 7, P7, P4, P5, P2, P3, P0, P1,
      ST_WDG, 6, E06, E05, E11, P7, P5, P2,
      ST_TET, 4, P7, P2, P5, N0,
      ST_TET, 4, P7, P5, P4, N0,
      ST_PYR, 5, P0, P4, P5, P1, N0,
      ST_PYR, 5, P3, P7, P4, P0, N0,
      ST_PYR, 5, P2, P3, P0, P1, N0,
      ST_TET, 4, P5, P2, P1, N0,
      ST_TET, 4, P2, P7, P3, N0,
      /* case 65 */ 6,
      ST_WDG, 6, P5, P7, P2, E05, E06, E11,
      ST_WDG, 6, E08, E03, E00, P4, P3, P1,
      ST_TET, 4, P7, P2, P5, P3,
      ST_TET, 4, P1, P2, P3, P5,
      ST_TET, 4, P4, P5, P1, P3,
      ST_TET, 4, P7, P4, P3, P5,
      /* case 66 */ 12,
      ST_PNT, 2, E00, E06,
      ST_PYR, 5, P0, P3, P7, P4, N0,
      ST_TET, 4, P4, P7, P5, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_TET, 4, P2, P3, P0, N0,
      ST_TET, 4, P7, P3, P2, N0,
      ST_PYR, 5, P7, P2, E11, E06, N0,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_TET, 4, P5, E05, E09, N0,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_PYR, 5, P0, E00, E01, P2, N0,
      ST_TET, 4, P2, E01, E11, N0,
      /* case 67 */ 11,
      ST_PNT, 4, E08, E03, E06, E06,
      ST_PYR, 5, P7, P2, E11, E06, N0,
      ST_TET, 4, E11, P2, E01, N0,
      ST_PYR, 5, P3, E03, E01, P2, N0,
      ST_TET, 4, P7, P3, P2, N0,
      ST_TET, 4, P5, E05, E09, N0,
      ST_TET, 4, P4, P7, P5, N0,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_TET, 4, P4, P3, P7, N0,
      ST_PYR, 5, P4, E08, E03, P3, N0,
      ST_PYR, 5, P4, P5, E09, E08, N0,
      /* case 68 */ 2,
      ST_HEX, 8, E06, P7, P3, E02, E05, P5, P1, E01,
      ST_WDG, 6, P1, P0, P3, P5, P4, P7,
      /* case 69 */ 11,
      ST_PNT, 4, E06, E05, E08, E08,
      ST_PYR, 5, P4, P1, E00, E08, N0,
      ST_TET, 4, E00, P1, E01, N0,
      ST_PYR, 5, P5, E05, E01, P1, N0,
      ST_TET, 4, P4, P5, P1, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, P3, P4, E08, E03, N0,
      ST_TET, 4, P7, P5, P4, N0,
      ST_PYR, 5, P7, E06, E05, P5, N0,
      ST_PYR, 5, P7, P3, E02, E06, N0,
      /* case 70 */ 9,
      ST_PNT, 5, E00, E02, E09, E05, E06,
      ST_PYR, 5, P0, P3, P7, P4, N0,
      ST_TET, 4, P7, N0, P5, P4,
      ST_PYR, 5, P3, E02, E06, P7, N0,
      ST_PYR, 5, E00, E02, P3, P0, N0,
      ST_TET, 4, P5, P0, P4, N0,
      ST_PYR, 5, P5, E09, E00, P0, N0,
      ST_TET, 4, P5, E05, E09, N0,
      ST_PYR, 5, E06, E05, P5, P7, N0,
      /* case 71 */ 9,
      ST_PNT, 6, E09, E05, E06, E02, E03, E08,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, P7, P3, E02, E06, N0,
      ST_TET, 4, P5, P4, P7, N0,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_TET, 4, E09, P5, E05, N0,
      ST_PYR, 5, P5, E09, E08, P4, N0,
      ST_PYR, 5, E08, E03, P3, P4, N0,
      ST_TET, 4, P3, E03, E02, N0,
      /* case 72 */ 12,
      ST_PNT, 2, E03, E05,
      ST_PYR, 5, P0, P4, P5, P1, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_TET, 4, P0, P1, P2, N0,
      ST_TET, 4, P7, P4, P0, N0,
      ST_TET, 4, P5, P4, P7, N0,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_PYR, 5, P2, P5, E05, E11, N0,
      ST_TET, 4, P2, E11, E02, N0,
      ST_PYR, 5, P0, P2, E02, E03, N0,
      ST_PYR, 5, P0, E03, E10, P7, N0,
      ST_TET, 4, P7, E10, E06, N0,
      /* case 73 */ 11,
      ST_PNT, 4, E00, E08, E05, E05,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_TET, 4, E06, P7, E10, N0,
      ST_PYR, 5, P4, E08, E10, P7, N0,
      ST_TET, 4, P5, P4, P7, N0,
      ST_TET, 4, P2, E11, E02, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_PYR, 5, P2, P5, E05, E11, N0,
      ST_TET, 4, P1, P4, P5, N0,
      ST_PYR, 5, P1, E00, E08, P4, N0,
      ST_PYR, 5, P1, P2, E02, E00, N0,
      /* case 74 */ 6,
      ST_TET, 4, P2, E01, E11, E02,
      ST_TET, 4, P7, P4, P0, P5,
      ST_PYR, 5, E09, E00, P0, P5, E03,
      ST_PYR, 5, E05, P5, P7, E06, E10,
      ST_PYR, 5, P0, E03, E10, P7, P5,
      ST_PYR, 5, E03, E09, E05, E10, P5,
      /* case 75 */ 8,
      ST_PNT, 5, E08, E10, E09, E05, E06,
      ST_TET, 4, P2, E02, E01, E11,
      ST_PYR, 5, P7, P4, E08, E10, N0,
      ST_PYR, 5, P4, P5, E09, E08, N0,
      ST_TET, 4, P5, E05, E09, N0,
      ST_TET, 4, P4, P7, P5, N0,
      ST_PYR, 5, P5, P7, E06, E05, N0,
      ST_TET, 4, P7, E10, E06, N0,
      /* case 76 */ 9,
      ST_PNT, 5, E03, E01, E10, E06, E05,
      ST_PYR, 5, P0, P4, P5, P1, N0,
      ST_TET, 4, P5, P7, N0, P4,
      ST_PYR, 5, P1, P5, E05, E01, N0,
      ST_PYR, 5, E03, P0, P1, E01, N0,
      ST_TET, 4, P7, P4, P0, N0,
      ST_PYR, 5, P7, P0, E03, E10, N0,
      ST_TET, 4, P7, E10, E06, N0,
      ST_PYR, 5, E05, P5, P7, E06, N0,
      /* case 77 */ 9,
      ST_PNT, 6, E10, E06, E05, E01, E00, E08,
      ST_TET, 4, P5, P1, P4, N0,
      ST_PYR, 5, P5, E05, E01, P1, N0,
      ST_TET, 4, P7, P5, P4, N0,
      ST_PYR, 5, P7, E06, E05, P5, N0,
      ST_TET, 4, E10, E06, P7, N0,
      ST_PYR, 5, P7, P4, E08, E10, N0,
      ST_PYR, 5, E08, P4, P1, E00, N0,
      ST_TET, 4, P1, E01, E00, N0,
      /* case 78 */ 5,
      ST_TET, 4, P7, P4, P0, P5,
      ST_PYR, 5, E09, E00, P0, P5, E03,
      ST_PYR, 5, E05, P5, P7, E06, E10,
      ST_PYR, 5, P0, E03, E10, P7, P5,
      ST_PYR, 5, E03, E09, E05, E10, P5,
      /* case 79 */ 7,
      ST_PNT, 5, E10, E08, E09, E06, E05,
      ST_PYR, 5, P4, P5, E09, E08, N0,
      ST_TET, 4, P5, E05, E09, N0,
      ST_PYR, 5, P7, P4, E08, E10, N0,
      ST_TET, 4, E06, P7, E10, N0,
      ST_PYR, 5, E05, P5, P7, E06, N0,
      ST_TET, 4, P5, P4, P7, N0,
      /* case 80 */ 12,
      ST_PNT, 2, E08, E11,
      ST_PYR, 5, P0, P1, P2, P3, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_TET, 4, P0, P5, P1, N0,
      ST_TET, 4, P7, P0, P3, N0,
      ST_TET, 4, P2, P7, P3, N0,
      ST_PYR, 5, P2, E11, E06, P7, N0,
      ST_PYR, 5, P5, E05, E11, P2, N0,
      ST_TET, 4, P5, E04, E05, N0,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_PYR, 5, P0, P7, E07, E08, N0,
      ST_TET, 4, P7, E06, E07, N0,
      /* case 81 */ 11,
      ST_PNT, 4, E00, E03, E11, E11,
      ST_PYR, 5, P2, E11, E06, P7, N0,
      ST_TET, 4, E06, E07, P7, N0,
      ST_PYR, 5, P3, P7, E07, E03, N0,
      ST_TET, 4, P2, P7, P3, N0,
      ST_TET, 4, P5, E04, E05, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_PYR, 5, P5, E05, E11, P2, N0,
      ST_TET, 4, P1, P2, P3, N0,
      ST_PYR, 5, P1, P3, E03, E00, N0,
      ST_PYR, 5, P1, E00, E04, P5, N0,
      /* case 82 */ 6,
      ST_TET, 4, P5, E05, E09, E04,
      ST_TET, 4, P7, P0, P3, P2,
      ST_PYR, 5, E01, P2, P0, E00, E08,
      ST_PYR, 5, E11, E06, P7, P2, E07,
      ST_PYR, 5, P0, P7, E07, E08, P2,
      ST_PYR, 5, E08, E07, E11, E01, P2,
      /* case 83 */ 8,
      ST_PNT, 5, E03, E07, E01, E11, E06,
      ST_TET, 4, P5, E09, E04, E05,
      ST_PYR, 5, P7, E07, E03, P3, N0,
      ST_PYR, 5, P3, E03, E01, P2, N0,
      ST_TET, 4, P2, E01, E11, N0,
      ST_TET, 4, P3, P2, P7, N0,
      ST_PYR, 5, P2, E11, E06, P7, N0,
      ST_TET, 4, P7, E06, E07, N0,
      /* case 84 */ 11,
      ST_PNT, 4, E02, E01, E08, E08,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_TET, 4, E04, E05, P5, N0,
      ST_PYR, 5, P1, P5, E05, E01, N0,
      ST_TET, 4, P0, P5, P1, N0,
      ST_TET, 4, P7, E06, E07, N0,
      ST_TET, 4, P3, P7, P0, N0,
      ST_PYR, 5, P7, E07, E08, P0, N0,
      ST_TET, 4, P3, P0, P1, N0,
      ST_PYR, 5, P3, P1, E01, E02, N0,
      ST_PYR, 5, P3, E02, E06, P7, N0,
      /* case 85 */ 2,
      ST_WDG, 6, P7, E07, E06, P3, E03, E02,
      ST_WDG, 6, P5, E05, E04, P1, E01, E00,
      /* case 86 */ 8,
      ST_PNT, 5, E02, E06, E00, E08, E07,
      ST_TET, 4, P5, E05, E09, E04,
      ST_PYR, 5, P7, P3, E02, E06, N0,
      ST_PYR, 5, P3, P0, E00, E02, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_TET, 4, P3, P7, P0, N0,
      ST_PYR, 5, P0, P7, E07, E08, N0,
      ST_TET, 4, P7, E06, E07, N0,
      /* case 87 */ 2,
      ST_WDG, 6, E06, P7, E07, E02, P3, E03,
      ST_TET, 4, E09, E04, P5, E05,
      /* case 88 */ 6,
      ST_TET, 4, P7, E06, E07, E10,
      ST_TET, 4, P2, P0, P1, P5,
      ST_PYR, 5, E04, P5, P0, E08, E03,
      ST_PYR, 5, E05, E11, P2, P5, E02,
      ST_PYR, 5, P0, P2, E02, E03, P5,
      ST_PYR, 5, E03, E02, E05, E04, P5,
      /* case 89 */ 8,
      ST_PNT, 5, E00, E02, E04, E05, E11,
      ST_TET, 4, P7, E07, E10, E06,
      ST_PYR, 5, P2, E02, E00, P1, N0,
      ST_PYR, 5, P1, E00, E04, P5, N0,
      ST_TET, 4, P5, E04, E05, N0,
      ST_TET, 4, P1, P5, P2, N0,
      ST_PYR, 5, P5, E05, E11, P2, N0,
      ST_TET, 4, P2, E11, E02, N0,
      /* case 90 */ 4,
      ST_TET, 4, E07, E06, E10, P7,
      ST_TET, 4, E08, E03, E00, P0,
      ST_TET, 4, E04, E09, E05, P5,
      ST_TET, 4, E01, E02, E11, P2,
      /* case 91 */ 3,
      ST_TET, 4, E04, E09, E05, P5,
      ST_TET, 4, E07, E10, P7, E06,
      ST_TET, 4, E01, P2, E02, E11,
      /* case 92 */ 8,
      ST_PNT, 5, E01, E05, E03, E08, E04,
      ST_TET, 4, P7, E10, E06, E07,
      ST_PYR, 5, P5, E05, E01, P1, N0,
      ST_PYR, 5, P1, E01, E03, P0, N0,
      ST_TET, 4, P0, E03, E08, N0,
      ST_TET, 4, P1, P0, P5, N0,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_TET, 4, P5, E04, E05, N0,
      /* case 93 */ 2,
      ST_WDG, 6, E01, P1, E00, E05, P5, E04,
      ST_TET, 4, E10, P7, E07, E06,
      /* case 94 */ 3,
      ST_TET, 4, E10, E07, E06, P7,
      ST_TET, 4, E03, P0, E00, E08,
      ST_TET, 4, E05, E09, P5, E04,
      /* case 95 */ 2,
      ST_TET, 4, E06, E10, E07, P7,
      ST_TET, 4, E05, E04, E09, P5,
      /* case 96 */ 2,
      ST_HEX, 8, E11, P2, P1, E09, E06, P7, P4, E04,
      ST_WDG, 6, P2, P3, P7, P1, P0, P4,
      /* case 97 */ 11,
      ST_PNT, 4, E06, E11, E03, E03,
      ST_PYR, 5, P3, E03, E00, P1, N0,
      ST_TET, 4, E00, E09, P1, N0,
      ST_PYR, 5, P2, P1, E09, E11, N0,
      ST_TET, 4, P3, P1, P2, N0,
      ST_TET, 4, P4, E04, E08, N0,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, P4, E08, E03, P3, N0,
      ST_TET, 4, P7, P3, P2, N0,
      ST_PYR, 5, P7, P2, E11, E06, N0,
      ST_PYR, 5, P7, E06, E04, P4, N0,
      /* case 98 */ 9,
      ST_PNT, 5, E00, E04, E01, E11, E06,
      ST_PYR, 5, P0, P3, P7, P4, N0,
      ST_TET, 4, P7, P2, N0, P3,
      ST_PYR, 5, P4, P7, E06, E04, N0,
      ST_PYR, 5, E00, P0, P4, E04, N0,
      ST_TET, 4, P2, P3, P0, N0,
      ST_PYR, 5, P2, P0, E00, E01, N0,
      ST_TET, 4, P2, E01, E11, N0,
      ST_PYR, 5, E06, P7, P2, E11, N0,
      /* case 99 */ 9,
      ST_PNT, 6, E01, E11, E06, E04, E08, E03,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, P7, E06, E04, P4, N0,
      ST_TET, 4, P2, P7, P3, N0,
      ST_PYR, 5, P2, E11, E06, P7, N0,
      ST_TET, 4, E01, E11, P2, N0,
      ST_PYR, 5, P2, P3, E03, E01, N0,
      ST_PYR, 5, E03, P3, P4, E08, N0,
      ST_TET, 4, P4, E04, E08, N0,
      /* case 100 */ 9,
      ST_PNT, 5, E02, E06, E01, E09, E04,
      ST_PYR, 5, P3, P7, P4, P0, N0,
      ST_TET, 4, P4, N0, P1, P0,
      ST_PYR, 5, P7, E06, E04, P4, N0,
      ST_PYR, 5, E02, E06, P7, P3, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_PYR, 5, P1, E01, E02, P3, N0,
      ST_TET, 4, P1, E09, E01, N0,
      ST_PYR, 5, E04, E09, P1, P4, N0,
      /* case 101 */ 8,
      ST_PNT, 5, E06, E04, E02, E03, E08,
      ST_TET, 4, P1, E09, E01, E00,
      ST_PYR, 5, P4, P7, E06, E04, N0,
      ST_PYR, 5, P7, P3, E02, E06, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_TET, 4, P7, P4, P3, N0,
      ST_PYR, 5, P3, P4, E08, E03, N0,
      ST_TET, 4, P4, E04, E08, N0,
      /* case 102 */ 1,
      ST_HEX, 8, E00, E04, E06, E02, P0, P4, P7, P3,
      /* case 103 */ 7,
      ST_PNT, 5, E04, E06, E02, E08, E03,
      ST_PYR, 5, P7, P3, E02, E06, N0,
      ST_TET, 4, P3, E03, E02, N0,
      ST_PYR, 5, P4, P7, E06, E04, N0,
      ST_TET, 4, E08, P4, E04, N0,
      ST_PYR, 5, E03, P3, P4, E08, N0,
      ST_TET, 4, P3, P7, P4, N0,
      /* case 104 */ 11,
      ST_PNT, 4, E04, E09, E03, E03,
      ST_PYR, 5, P0, P2, E02, E03, N0,
      ST_TET, 4, E02, P2, E11, N0,
      ST_PYR, 5, P1, E09, E11, P2, N0,
      ST_TET, 4, P0, P1, P2, N0,
      ST_TET, 4, P7, E10, E06, N0,
      ST_TET, 4, P4, P0, P7, N0,
      ST_PYR, 5, P7, P0, E03, E10, N0,
      ST_TET, 4, P4, P1, P0, N0,
      ST_PYR, 5, P4, E04, E09, P1, N0,
      ST_PYR, 5, P4, P7, E06, E04, N0,
      /* case 105 */ 2,
      ST_WDG, 6, P4, E08, E04, P7, E10, E06,
      ST_WDG, 6, P1, E09, E00, P2, E11, E02,
      /* case 106 */ 8,
      ST_PNT, 5, E04, E06, E00, E03, E10,
      ST_TET, 4, P2, E01, E11, E02,
      ST_PYR, 5, P7, E06, E04, P4, N0,
      ST_PYR, 5, P4, E04, E00, P0, N0,
      ST_TET, 4, P0, E00, E03, N0,
      ST_TET, 4, P4, P0, P7, N0,
      ST_PYR, 5, P0, E03, E10, P7, N0,
      ST_TET, 4, P7, E10, E06, N0,
      /* case 107 */ 2,
      ST_WDG, 6, E04, P4, E08, E06, P7, E10,
      ST_TET, 4, E01, P2, E02, E11,
      /* case 108 */ 9,
      ST_PNT, 6, E01, E09, E04, E06, E10, E03,
      ST_TET, 4, P4, P0, P7, N0,
      ST_PYR, 5, P4, P7, E06, E04, N0,
      ST_TET, 4, P1, P0, P4, N0,
      ST_PYR, 5, P1, P4, E04, E09, N0,
      ST_TET, 4, E01, P1, E09, N0,
      ST_PYR, 5, P1, E01, E03, P0, N0,
      ST_PYR, 5, E03, E10, P7, P0, N0,
      ST_TET, 4, P7, E10, E06, N0,
      /* case 109 */ 2,
      ST_WDG, 6, E04, P4, E08, E06, P7, E10,
      ST_TET, 4, E01, E00, P1, E09,
      /* case 110 */ 7,
      ST_PNT, 5, E06, E04, E00, E10, E03,
      ST_PYR, 5, P4, E04, E00, P0, N0,
      ST_TET, 4, P0, E00, E03, N0,
      ST_PYR, 5, P7, E06, E04, P4, N0,
      ST_TET, 4, E10, E06, P7, N0,
      ST_PYR, 5, E03, E10, P7, P0, N0,
      ST_TET, 4, P0, P7, P4, N0,
      /* case 111 */ 1,
      ST_WDG, 6, P4, E08, E04, P7, E10, E06,
      /* case 112 */ 9,
      ST_PNT, 5, E08, E09, E07, E06, E11,
      ST_PYR, 5, P0, P1, P2, P3, N0,
      ST_TET, 4, P2, N0, P7, P3,
      ST_PYR, 5, P1, E09, E11, P2, N0,
      ST_PYR, 5, E08, E09, P1, P0, N0,
      ST_TET, 4, P7, P0, P3, N0,
      ST_PYR, 5, P7, E07, E08, P0, N0,
      ST_TET, 4, P7, E06, E07, N0,
      ST_PYR, 5, E11, E06, P7, P2, N0,
      /* case 113 */ 9,
      ST_PNT, 6, E07, E06, E11, E09, E00, E03,
      ST_TET, 4, P2, P3, P1, N0,
      ST_PYR, 5, P2, P1, E09, E11, N0,
      ST_TET, 4, P7, P3, P2, N0,
      ST_PYR, 5, P7, P2, E11, E06, N0,
      ST_TET, 4, E07, P7, E06, N0,
      ST_PYR, 5, P7, E07, E03, P3, N0,
      ST_PYR, 5, E03, E00, P1, P3, N0,
      ST_TET, 4, P1, E00, E09, N0,
      /* case 114 */ 5,
      ST_TET, 4, P7, P0, P3, P2,
      ST_PYR, 5, E01, P2, P0, E00, E08,
      ST_PYR, 5, E11, E06, P7, P2, E07,
      ST_PYR, 5, P0, P7, E07, E08, P2,
      ST_PYR, 5, E08, E07, E11, E01, P2,
      /* case 115 */ 7,
      ST_PNT, 5, E07, E03, E01, E06, E11,
      ST_PYR, 5, P3, E03, E01, P2, N0,
      ST_TET, 4, P2, E01, E11, N0,
      ST_PYR, 5, P7, E07, E03, P3, N0,
      ST_TET, 4, E06, E07, P7, N0,
      ST_PYR, 5, E11, E06, P7, P2, N0,
      ST_TET, 4, P2, P7, P3, N0,
      /* case 116 */ 9,
      ST_PNT, 6, E09, E01, E02, E06, E07, E08,
      ST_TET, 4, P3, P7, P0, N0,
      ST_PYR, 5, P3, E02, E06, P7, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_PYR, 5, P1, E01, E02, P3, N0,
      ST_TET, 4, E09, E01, P1, N0,
      ST_PYR, 5, P1, P0, E08, E09, N0,
      ST_PYR, 5, E08, P0, P7, E07, N0,
      ST_TET, 4, P7, E06, E07, N0,
      /* case 117 */ 2,
      ST_WDG, 6, E06, P7, E07, E02, P3, E03,
      ST_TET, 4, E09, P1, E00, E01,
      /* case 118 */ 7,
      ST_PNT, 5, E06, E02, E00, E07, E08,
      ST_PYR, 5, P3, P0, E00, E02, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_PYR, 5, P7, P3, E02, E06, N0,
      ST_TET, 4, E07, P7, E06, N0,
      ST_PYR, 5, E08, P0, P7, E07, N0,
      ST_TET, 4, P0, P3, P7, N0,
      /* case 119 */ 1,
      ST_WDG, 6, P7, E07, E06, P3, E03, E02,
      /* case 120 */ 8,
      ST_PNT, 5, E09, E11, E08, E03, E02,
      ST_TET, 4, P7, E06, E07, E10,
      ST_PYR, 5, P2, P1, E09, E11, N0,
      ST_PYR, 5, P1, P0, E08, E09, N0,
      ST_TET, 4, P0, E03, E08, N0,
      ST_TET, 4, P1, P2, P0, N0,
      ST_PYR, 5, P0, P2, E02, E03, N0,
      ST_TET, 4, P2, E11, E02, N0,
      /* case 121 */ 2,
      ST_WDG, 6, E11, P2, E02, E09, P1, E00,
      ST_TET, 4, E07, E10, P7, E06,
      /* case 122 */ 3,
      ST_TET, 4, E07, E06, E10, P7,
      ST_TET, 4, E08, E00, P0, E03,
      ST_TET, 4, E11, P2, E01, E02,
      /* case 123 */ 2,
      ST_TET, 4, E06, E10, E07, P7,
      ST_TET, 4, E11, E01, E02, P2,
      /* case 124 */ 2,
      ST_WDG, 6, E03, P0, E08, E01, P1, E09,
      ST_TET, 4, E06, E07, P7, E10,
      /* case 125 */ 2,
      ST_TET, 4, E06, E10, E07, P7,
      ST_TET, 4, E09, E00, E01, P1,
      /* case 126 */ 2,
      ST_TET, 4, E08, E03, E00, P0,
      ST_TET, 4, E07, E06, E10, P7,
      /* case 127 */ 1,
      ST_TET, 4, E07, E06, E10, P7,
      /* case 128 */ 9,
      ST_PNT, 7, P6, P5, P4, P3, P2, P1, P0,
      ST_WDG, 6, P6, P4, P3, E06, E07, E10,
      ST_TET, 4, P6, P4, P3, N0,
      ST_TET, 4, P6, P5, P4, N0,
      ST_PYR, 5, P1, P0, P4, P5, N0,
      ST_PYR, 5, P2, P1, P5, P6, N0,
      ST_PYR, 5, P3, P0, P1, P2, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_TET, 4, P3, P2, P6, N0,
      /* case 129 */ 12,
      ST_PNT, 2, E00, E06,
      ST_PYR, 5, P1, P5, P6, P2, N0,
      ST_TET, 4, P5, P4, P6, N0,
      ST_TET, 4, P1, P4, P5, N0,
      ST_TET, 4, P3, P1, P2, N0,
      ST_TET, 4, P6, P3, P2, N0,
      ST_PYR, 5, P6, E06, E10, P3, N0,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_PYR, 5, P1, E00, E08, P4, N0,
      ST_PYR, 5, P1, P3, E03, E00, N0,
      ST_TET, 4, P3, E10, E03, N0,
      /* case 130 */ 6,
      ST_WDG, 6, P4, P3, P6, E07, E10, E06,
      ST_WDG, 6, E00, E01, E09, P0, P2, P5,
      ST_TET, 4, P3, P6, P4, P2,
      ST_TET, 4, P5, P6, P2, P4,
      ST_TET, 4, P0, P4, P5, P2,
      ST_TET, 4, P3, P0, P2, P4,
      /* case 131 */ 11,
      ST_PNT, 4, E09, E01, E06, E06,
      ST_PYR, 5, P6, E06, E10, P3, N0,
      ST_TET, 4, E10, E03, P3, N0,
      ST_PYR, 5, P2, P3, E03, E01, N0,
      ST_TET, 4, P6, P3, P2, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_TET, 4, P5, P4, P6, N0,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_TET, 4, P5, P6, P2, N0,
      ST_PYR, 5, P5, P2, E01, E09, N0,
      ST_PYR, 5, P5, E09, E08, P4, N0,
      /* case 132 */ 12,
      ST_PNT, 2, E07, E01,
      ST_PYR, 5, P4, P5, P1, P0, N0,
      ST_TET, 4, P5, P6, P1, N0,
      ST_TET, 4, P4, P6, P5, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_PYR, 5, P1, E01, E02, P3, N0,
      ST_PYR, 5, P6, E11, E01, P1, N0,
      ST_TET, 4, P6, E06, E11, N0,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_PYR, 5, P4, P3, E10, E07, N0,
      ST_TET, 4, P3, E02, E10, N0,
      /* case 133 */ 6,
      ST_TET, 4, P3, E03, E02, E10,
      ST_TET, 4, P6, P5, P4, P1,
      ST_PYR, 5, E00, E08, P4, P1, E07,
      ST_PYR, 5, E01, P1, P6, E11, E06,
      ST_PYR, 5, P4, E07, E06, P6, P1,
      ST_PYR, 5, E07, E00, E01, E06, P1,
      /* case 134 */ 11,
      ST_PNT, 4, E09, E00, E07, E07,
      ST_PYR, 5, P4, P3, E10, E07, N0,
      ST_TET, 4, E10, P3, E02, N0,
      ST_PYR, 5, P0, E00, E02, P3, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_TET, 4, P6, E06, E11, N0,
      ST_TET, 4, P5, P4, P6, N0,
      ST_PYR, 5, P6, P4, E07, E06, N0,
      ST_TET, 4, P5, P0, P4, N0,
      ST_PYR, 5, P5, E09, E00, P0, N0,
      ST_PYR, 5, P5, P6, E11, E09, N0,
      /* case 135 */ 8,
      ST_PNT, 5, E09, E11, E08, E07, E06,
      ST_TET, 4, P3, E03, E02, E10,
      ST_PYR, 5, P6, E11, E09, P5, N0,
      ST_PYR, 5, P5, E09, E08, P4, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_TET, 4, P5, P4, P6, N0,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_TET, 4, P6, E06, E11, N0,
      /* case 136 */ 2,
      ST_HEX, 8, E07, P4, P0, E03, E06, P6, P2, E02,
      ST_WDG, 6, P4, P5, P6, P0, P1, P2,
      /* case 137 */ 9,
      ST_PNT, 5, E00, E02, E08, E07, E06,
      ST_PYR, 5, P1, P5, P6, P2, N0,
      ST_TET, 4, P6, P4, N0, P5,
      ST_PYR, 5, P2, P6, E06, E02, N0,
      ST_PYR, 5, E00, P1, P2, E02, N0,
      ST_TET, 4, P4, P5, P1, N0,
      ST_PYR, 5, P4, P1, E00, E08, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_PYR, 5, E06, P6, P4, E07, N0,
      /* case 138 */ 11,
      ST_PNT, 4, E06, E07, E09, E09,
      ST_PYR, 5, P5, E09, E00, P0, N0,
      ST_TET, 4, E00, E03, P0, N0,
      ST_PYR, 5, P4, P0, E03, E07, N0,
      ST_TET, 4, P5, P0, P4, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_TET, 4, P6, P2, P5, N0,
      ST_PYR, 5, P2, E01, E09, P5, N0,
      ST_TET, 4, P6, P5, P4, N0,
      ST_PYR, 5, P6, P4, E07, E06, N0,
      ST_PYR, 5, P6, E06, E02, P2, N0,
      /* case 139 */ 9,
      ST_PNT, 6, E08, E07, E06, E02, E01, E09,
      ST_TET, 4, P6, P2, P5, N0,
      ST_PYR, 5, P6, E06, E02, P2, N0,
      ST_TET, 4, P4, P6, P5, N0,
      ST_PYR, 5, P4, E07, E06, P6, N0,
      ST_TET, 4, E08, E07, P4, N0,
      ST_PYR, 5, P4, P5, E09, E08, N0,
      ST_PYR, 5, E09, P5, P2, E01, N0,
      ST_TET, 4, P2, E02, E01, N0,
      /* case 140 */ 9,
      ST_PNT, 5, E01, E03, E11, E06, E07,
      ST_PYR, 5, P1, P0, P4, P5, N0,
      ST_TET, 4, P4, N0, P6, P5,
      ST_PYR, 5, P0, E03, E07, P4, N0,
      ST_PYR, 5, E01, E03, P0, P1, N0,
      ST_TET, 4, P6, P1, P5, N0,
      ST_PYR, 5, P6, E11, E01, P1, N0,
      ST_TET, 4, P6, E06, E11, N0,
      ST_PYR, 5, E07, E06, P6, P4, N0,
      /* case 141 */ 5,
      ST_TET, 4, P6, P5, P4, P1,
      ST_PYR, 5, E00, E08, P4, P1, E07,
      ST_PYR, 5, E01, P1, P6, E11, E06,
      ST_PYR, 5, P4, E07, E06, P6, P1,
      ST_PYR, 5, E07, E00, E01, E06, P1,
      /* case 142 */ 9,
      ST_PNT, 6, E03, E00, E09, E11, E06, E07,
      ST_TET, 4, P5, P4, P6, N0,
      ST_PYR, 5, P5, P6, E11, E09, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_TET, 4, E03, P0, E00, N0,
      ST_PYR, 5, P0, E03, E07, P4, N0,
      ST_PYR, 5, E07, E06, P6, P4, N0,
      ST_TET, 4, P6, E06, E11, N0,
      /* case 143 */ 7,
      ST_PNT, 5, E11, E09, E08, E06, E07,
      ST_PYR, 5, P5, E09, E08, P4, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_PYR, 5, P6, E11, E09, P5, N0,
      ST_TET, 4, E06, E11, P6, N0,
      ST_PYR, 5, E07, E06, P6, P4, N0,
      ST_TET, 4, P4, P6, P5, N0,
      /* case 144 */ 2,
      ST_HEX, 8, E06, P6, P5, E04, E10, P3, P0, E08,
      ST_WDG, 6, P0, P1, P5, P3, P2, P6,
      /* case 145 */ 9,
      ST_PNT, 5, E00, E04, E03, E10, E06,
      ST_PYR, 5, P1, P5, P6, P2, N0,
      ST_TET, 4, P6, N0, P3, P2,
      ST_PYR, 5, P5, E04, E06, P6, N0,
      ST_PYR, 5, E00, E04, P5, P1, N0,
      ST_TET, 4, P3, P1, P2, N0,
      ST_PYR, 5, P3, E03, E00, P1, N0,
      ST_TET, 4, P3, E10, E03, N0,
      ST_PYR, 5, E06, E10, P3, P6, N0,
      /* case 146 */ 11,
      ST_PNT, 4, E06, E10, E01, E01,
      ST_PYR, 5, P2, P0, E00, E01, N0,
      ST_TET, 4, E00, P0, E08, N0,
      ST_PYR, 5, P3, E10, E08, P0, N0,
      ST_TET, 4, P2, P3, P0, N0,
      ST_TET, 4, P5, E09, E04, N0,
      ST_TET, 4, P6, P2, P5, N0,
      ST_PYR, 5, P5, P2, E01, E09, N0,
      ST_TET, 4, P6, P3, P2, N0,
      ST_PYR, 5, P6, E06, E10, P3, N0,
      ST_PYR, 5, P6, P5, E04, E06, N0,
      /* case 147 */ 9,
      ST_PNT, 6, E03, E10, E06, E04, E09, E01,
      ST_TET, 4, P6, P2, P5, N0,
      ST_PYR, 5, P6, P5, E04, E06, N0,
      ST_TET, 4, P3, P2, P6, N0,
      ST_PYR, 5, P3, P6, E06, E10, N0,
      ST_TET, 4, E03, P3, E10, N0,
      ST_PYR, 5, P3, E03, E01, P2, N0,
      ST_PYR, 5, E01, E09, P5, P2, N0,
      ST_TET, 4, P5, E09, E04, N0,
      /* case 148 */ 11,
      ST_PNT, 4, E04, E08, E01, E01,
      ST_PYR, 5, P1, E01, E02, P3, N0,
      ST_TET, 4, E02, E10, P3, N0,
      ST_PYR, 5, P0, P3, E10, E08, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_TET, 4, P6, E06, E11, N0,
      ST_TET, 4, P5, P6, P1, N0,
      ST_PYR, 5, P6, E11, E01, P1, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_PYR, 5, P5, P0, E08, E04, N0,
      ST_PYR, 5, P5, E04, E06, P6, N0,
      /* case 149 */ 8,
      ST_PNT, 5, E04, E06, E00, E01, E11,
      ST_TET, 4, P3, E10, E03, E02,
      ST_PYR, 5, P6, P5, E04, E06, N0,
      ST_PYR, 5, P5, P1, E00, E04, N0,
      ST_TET, 4, P1, E01, E00, N0,
      ST_TET, 4, P5, P6, P1, N0,
      ST_PYR, 5, P1, P6, E11, E01, N0,
      ST_TET, 4, P6, E06, E11, N0,
      /* case 150 */ 2,
      ST_WDG, 6, P3, E10, E02, P0, E08, E00,
      ST_WDG, 6, P6, E11, E06, P5, E09, E04,
      /* case 151 */ 2,
      ST_WDG, 6, E09, P5, E04, E11, P6, E06,
      ST_TET, 4, E03, P3, E10, E02,
      /* case 152 */ 9,
      ST_PNT, 5, E02, E06, E03, E08, E04,
      ST_PYR, 5, P2, P1, P5, P6, N0,
      ST_TET, 4, P5, P0, N0, P1,
      ST_PYR, 5, P6, P5, E04, E06, N0,
      ST_PYR, 5, E02, P2, P6, E06, N0,
      ST_TET, 4, P0, P1, P2, N0,
      ST_PYR, 5, P0, P2, E02, E03, N0,
      ST_TET, 4, P0, E03, E08, N0,
      ST_PYR, 5, E04, P5, P0, E08, N0,
      /* case 153 */ 1,
      ST_HEX, 8, P1, P5, P6, P2, E00, E04, E06, E02,
      /* case 154 */ 8,
      ST_PNT, 5, E06, E04, E02, E01, E09,
      ST_TET, 4, P0, E03, E08, E00,
      ST_PYR, 5, P5, E04, E06, P6, N0,
      ST_PYR, 5, P6, E06, E02, P2, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_TET, 4, P6, P2, P5, N0,
      ST_PYR, 5, P2, E01, E09, P5, N0,
      ST_TET, 4, P5, E09, E04, N0,
      /* case 155 */ 7,
      ST_PNT, 5, E04, E06, E02, E09, E01,
      ST_PYR, 5, P6, E06, E02, P2, N0,
      ST_TET, 4, P2, E02, E01, N0,
      ST_PYR, 5, P5, E04, E06, P6, N0,
      ST_TET, 4, E09, E04, P5, N0,
      ST_PYR, 5, E01, E09, P5, P2, N0,
      ST_TET, 4, P2, P5, P6, N0,
      /* case 156 */ 9,
      ST_PNT, 6, E03, E08, E04, E06, E11, E01,
      ST_TET, 4, P5, P6, P1, N0,
      ST_PYR, 5, P5, E04, E06, P6, N0,
      ST_TET, 4, P0, P5, P1, N0,
      ST_PYR, 5, P0, E08, E04, P5, N0,
      ST_TET, 4, E03, E08, P0, N0,
      ST_PYR, 5, P0, P1, E01, E03, N0,
      ST_PYR, 5, E01, P1, P6, E11, N0,
      ST_TET, 4, P6, E06, E11, N0,
      /* case 157 */ 7,
      ST_PNT, 5, E06, E04, E00, E11, E01,
      ST_PYR, 5, P5, P1, E00, E04, N0,
      ST_TET, 4, P1, E01, E00, N0,
      ST_PYR, 5, P6, P5, E04, E06, N0,
      ST_TET, 4, E11, P6, E06, N0,
      ST_PYR, 5, E01, P1, P6, E11, N0,
      ST_TET, 4, P1, P5, P6, N0,
      /* case 158 */ 2,
      ST_WDG, 6, E09, P5, E04, E11, P6, E06,
      ST_TET, 4, E03, E08, P0, E00,
      /* case 159 */ 1,
      ST_WDG, 6, P5, E04, E09, P6, E06, E11,
      /* case 160 */ 12,
      ST_PNT, 2, E10, E09,
      ST_PYR, 5, P3, P0, P1, P2, N0,
      ST_TET, 4, P2, P1, P6, N0,
      ST_TET, 4, P3, P2, P6, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_TET, 4, P1, P0, P4, N0,
      ST_PYR, 5, P1, P4, E04, E09, N0,
      ST_PYR, 5, P6, P1, E09, E05, N0,
      ST_TET, 4, P6, E05, E06, N0,
      ST_PYR, 5, P3, P6, E06, E10, N0,
      ST_PYR, 5, P3, E10, E07, P4, N0,
      ST_TET, 4, P4, E07, E04, N0,
      /* case 161 */ 6,
      ST_TET, 4, P4, E04, E08, E07,
      ST_TET, 4, P6, P3, P2, P1,
      ST_PYR, 5, E00, P1, P3, E03, E10,
      ST_PYR, 5, E09, E05, P6, P1, E06,
      ST_PYR, 5, P3, P6, E06, E10, P1,
      ST_PYR, 5, E10, E06, E09, E00, P1,
      /* case 162 */ 11,
      ST_PNT, 4, E01, E00, E10, E10,
      ST_PYR, 5, P3, E10, E07, P4, N0,
      ST_TET, 4, E07, E04, P4, N0,
      ST_PYR, 5, P0, P4, E04, E00, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_TET, 4, P6, E05, E06, N0,
      ST_TET, 4, P2, P6, P3, N0,
      ST_PYR, 5, P6, E06, E10, P3, N0,
      ST_TET, 4, P2, P3, P0, N0,
      ST_PYR, 5, P2, P0, E00, E01, N0,
      ST_PYR, 5, P2, E01, E05, P6, N0,
      /* case 163 */ 8,
      ST_PNT, 5, E01, E05, E03, E10, E06,
      ST_TET, 4, P4, E04, E08, E07,
      ST_PYR, 5, P6, P2, E01, E05, N0,
      ST_PYR, 5, P2, P3, E03, E01, N0,
      ST_TET, 4, P3, E10, E03, N0,
      ST_TET, 4, P2, P6, P3, N0,
      ST_PYR, 5, P3, P6, E06, E10, N0,
      ST_TET, 4, P6, E05, E06, N0,
      /* case 164 */ 6,
      ST_TET, 4, P6, E05, E06, E11,
      ST_TET, 4, P3, P0, P1, P4,
      ST_PYR, 5, E04, E09, P1, P4, E01,
      ST_PYR, 5, E07, P4, P3, E10, E02,
      ST_PYR, 5, P1, E01, E02, P3, P4,
      ST_PYR, 5, E01, E04, E07, E02, P4,
      /* case 165 */ 4,
      ST_TET, 4, E07, E08, E04, P4,
      ST_TET, 4, E10, E02, E03, P3,
      ST_TET, 4, E06, E05, E11, P6,
      ST_TET, 4, E01, E09, E00, P1,
      /* case 166 */ 8,
      ST_PNT, 5, E00, E02, E04, E07, E10,
      ST_TET, 4, P6, E11, E05, E06,
      ST_PYR, 5, P3, P0, E00, E02, N0,
      ST_PYR, 5, P0, P4, E04, E00, N0,
      ST_TET, 4, P4, E07, E04, N0,
      ST_TET, 4, P0, P3, P4, N0,
      ST_PYR, 5, P4, P3, E10, E07, N0,
      ST_TET, 4, P3, E02, E10, N0,
      /* case 167 */ 3,
      ST_TET, 4, E11, E06, E05, P6,
      ST_TET, 4, E02, P3, E03, E10,
      ST_TET, 4, E04, E08, P4, E07,
      /* case 168 */ 11,
      ST_PNT, 4, E02, E03, E09, E09,
      ST_PYR, 5, P1, P4, E04, E09, N0,
      ST_TET, 4, E04, P4, E07, N0,
      ST_PYR, 5, P0, E03, E07, P4, N0,
      ST_TET, 4, P1, P0, P4, N0,
      ST_TET, 4, P6, E05, E06, N0,
      ST_TET, 4, P2, P1, P6, N0,
      ST_PYR, 5, P6, P1, E09, E05, N0,
      ST_TET, 4, P2, P0, P1, N0,
      ST_PYR, 5, P2, E02, E03, P0, N0,
      ST_PYR, 5, P2, P6, E06, E02, N0,
      /* case 169 */ 8,
      ST_PNT, 5, E02, E06, E00, E09, E05,
      ST_TET, 4, P4, E08, E07, E04,
      ST_PYR, 5, P6, E06, E02, P2, N0,
      ST_PYR, 5, P2, E02, E00, P1, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_TET, 4, P2, P1, P6, N0,
      ST_PYR, 5, P1, E09, E05, P6, N0,
      ST_TET, 4, P6, E05, E06, N0,
      /* case 170 */ 2,
      ST_WDG, 6, P0, E03, E00, P4, E07, E04,
      ST_WDG, 6, P2, E01, E02, P6, E05, E06,
      /* case 171 */ 2,
      ST_WDG, 6, E05, P6, E06, E01, P2, E02,
      ST_TET, 4, E08, E07, P4, E04,
      /* case 172 */ 8,
      ST_PNT, 5, E03, E07, E01, E09, E04,
      ST_TET, 4, P6, E06, E11, E05,
      ST_PYR, 5, P4, P0, E03, E07, N0,
      ST_PYR, 5, P0, P1, E01, E03, N0,
      ST_TET, 4, P1, E09, E01, N0,
      ST_TET, 4, P0, P4, P1, N0,
      ST_PYR, 5, P1, P4, E04, E09, N0,
      ST_TET, 4, P4, E07, E04, N0,
      /* case 173 */ 3,
      ST_TET, 4, E06, E05, E11, P6,
      ST_TET, 4, E07, P4, E08, E04,
      ST_TET, 4, E01, E00, P1, E09,
      /* case 174 */ 2,
      ST_WDG, 6, E07, P4, E04, E03, P0, E00,
      ST_TET, 4, E11, E05, P6, E06,
      /* case 175 */ 2,
      ST_TET, 4, E04, E07, E08, P4,
      ST_TET, 4, E05, E11, E06, P6,
      /* case 176 */ 9,
      ST_PNT, 5, E09, E08, E05, E06, E10,
      ST_PYR, 5, P1, P2, P3, P0, N0,
      ST_TET, 4, P3, P6, N0, P2,
      ST_PYR, 5, P0, P3, E10, E08, N0,
      ST_PYR, 5, E09, P1, P0, E08, N0,
      ST_TET, 4, P6, P2, P1, N0,
      ST_PYR, 5, P6, P1, E09, E05, N0,
      ST_TET, 4, P6, E05, E06, N0,
      ST_PYR, 5, E10, P3, P6, E06, N0,
      /* case 177 */ 5,
      ST_TET, 4, P6, P3, P2, P1,
      ST_PYR, 5, E00, P1, P3, E03, E10,
      ST_PYR, 5, E09, E05, P6, P1, E06,
      ST_PYR, 5, P3, P6, E06, E10, P1,
      ST_PYR, 5, E10, E06, E09, E00, P1,
      /* case 178 */ 9,
      ST_PNT, 6, E08, E00, E01, E05, E06, E10,
      ST_TET, 4, P2, P6, P3, N0,
      ST_PYR, 5, P2, E01, E05, P6, N0,
      ST_TET, 4, P0, P2, P3, N0,
      ST_PYR, 5, P0, E00, E01, P2, N0,
      ST_TET, 4, E08, E00, P0, N0,
      ST_PYR, 5, P0, P3, E10, E08, N0,
      ST_PYR, 5, E10, P3, P6, E06, N0,
      ST_TET, 4, P6, E05, E06, N0,
      /* case 179 */ 7,
      ST_PNT, 5, E05, E01, E03, E06, E10,
      ST_PYR, 5, P2, P3, E03, E01, N0,
      ST_TET, 4, P3, E10, E03, N0,
      ST_PYR, 5, P6, P2, E01, E05, N0,
      ST_TET, 4, E06, P6, E05, N0,
      ST_PYR, 5, E10, P3, P6, E06, N0,
      ST_TET, 4, P3, P2, P6, N0,
      /* case 180 */ 8,
      ST_PNT, 5, E08, E10, E09, E01, E02,
      ST_TET, 4, P6, E05, E06, E11,
      ST_PYR, 5, P3, E10, E08, P0, N0,
      ST_PYR, 5, P0, E08, E09, P1, N0,
      ST_TET, 4, P1, E09, E01, N0,
      ST_TET, 4, P0, P1, P3, N0,
      ST_PYR, 5, P1, E01, E02, P3, N0,
      ST_TET, 4, P3, E02, E10, N0,
      /* case 181 */ 3,
      ST_TET, 4, E06, E05, E11, P6,
      ST_TET, 4, E10, E03, P3, E02,
      ST_TET, 4, E09, P1, E00, E01,
      /* case 182 */ 2,
      ST_WDG, 6, E08, P0, E00, E10, P3, E02,
      ST_TET, 4, E05, P6, E11, E06,
      /* case 183 */ 2,
      ST_TET, 4, E02, E03, E10, P3,
      ST_TET, 4, E11, E06, E05, P6,
      /* case 184 */ 9,
      ST_PNT, 6, E08, E03, E02, E06, E05, E09,
      ST_TET, 4, P2, P1, P6, N0,
      ST_PYR, 5, P2, P6, E06, E02, N0,
      ST_TET, 4, P0, P1, P2, N0,
      ST_PYR, 5, P0, P2, E02, E03, N0,
      ST_TET, 4, E08, P0, E03, N0,
      ST_PYR, 5, P0, E08, E09, P1, N0,
      ST_PYR, 5, E09, E05, P6, P1, N0,
      ST_TET, 4, P6, E05, E06, N0,
      /* case 185 */ 7,
      ST_PNT, 5, E06, E02, E00, E05, E09,
      ST_PYR, 5, P2, E02, E00, P1, N0,
      ST_TET, 4, P1, E00, E09, N0,
      ST_PYR, 5, P6, E06, E02, P2, N0,
      ST_TET, 4, E05, E06, P6, N0,
      ST_PYR, 5, E09, E05, P6, P1, N0,
      ST_TET, 4, P1, P6, P2, N0,
      /* case 186 */ 2,
      ST_WDG, 6, E05, P6, E06, E01, P2, E02,
      ST_TET, 4, E08, P0, E03, E00,
      /* case 187 */ 1,
      ST_WDG, 6, P6, E06, E05, P2, E02, E01,
      /* case 188 */ 2,
      ST_WDG, 6, E03, P0, E08, E01, P1, E09,
      ST_TET, 4, E06, P6, E05, E11,
      /* case 189 */ 2,
      ST_TET, 4, E09, E00, E01, P1,
      ST_TET, 4, E05, E11, E06, P6,
      /* case 190 */ 2,
      ST_TET, 4, E00, E08, E03, P0,
      ST_TET, 4, E11, E06, E05, P6,
      /* case 191 */ 1,
      ST_TET, 4, E05, E11, E06, P6,
      /* case 192 */ 2,
      ST_HEX, 8, E05, P5, P4, E07, E11, P2, P3, E10,
      ST_WDG, 6, P5, P1, P2, P4, P0, P3,
      /* case 193 */ 11,
      ST_PNT, 4, E11, E05, E00, E00,
      ST_PYR, 5, P1, E00, E08, P4, N0,
      ST_TET, 4, E08, E07, P4, N0,
      ST_PYR, 5, P5, P4, E07, E05, N0,
      ST_TET, 4, P1, P4, P5, N0,
      ST_TET, 4, P3, E10, E03, N0,
      ST_TET, 4, P2, P3, P1, N0,
      ST_PYR, 5, P3, E03, E00, P1, N0,
      ST_TET, 4, P2, P1, P5, N0,
      ST_PYR, 5, P2, P5, E05, E11, N0,
      ST_PYR, 5, P2, E11, E10, P3, N0,
      /* case 194 */ 11,
      ST_PNT, 4, E10, E07, E00, E00,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_TET, 4, E09, P5, E05, N0,
      ST_PYR, 5, P4, E07, E05, P5, N0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_TET, 4, P2, E01, E11, N0,
      ST_TET, 4, P3, P0, P2, N0,
      ST_PYR, 5, P2, P0, E00, E01, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_PYR, 5, P3, E10, E07, P4, N0,
      ST_PYR, 5, P3, P2, E11, E10, N0,
      /* case 195 */ 2,
      ST_WDG, 6, P3, E03, E10, P2, E01, E11,
      ST_WDG, 6, P4, E07, E08, P5, E05, E09,
      /* case 196 */ 9,
      ST_PNT, 5, E07, E05, E10, E02, E01,
      ST_PYR, 5, P4, P5, P1, P0, N0,
      ST_TET, 4, P1, N0, P3, P0,
      ST_PYR, 5, P5, E05, E01, P1, N0,
      ST_PYR, 5, E07, E05, P5, P4, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_PYR, 5, P3, E10, E07, P4, N0,
      ST_TET, 4, P3, E02, E10, N0,
      ST_PYR, 5, E01, E02, P3, P1, N0,
      /* case 197 */ 8,
      ST_PNT, 5, E05, E01, E07, E08, E00,
      ST_TET, 4, P3, E02, E10, E03,
      ST_PYR, 5, P1, P5, E05, E01, N0,
      ST_PYR, 5, P5, P4, E07, E05, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_TET, 4, P5, P1, P4, N0,
      ST_PYR, 5, P4, P1, E00, E08, N0,
      ST_TET, 4, P1, E01, E00, N0,
      /* case 198 */ 9,
      ST_PNT, 6, E02, E10, E07, E05, E09, E00,
      ST_TET, 4, P4, P5, P0, N0,
      ST_PYR, 5, P4, E07, E05, P5, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_PYR, 5, P3, E10, E07, P4, N0,
      ST_TET, 4, E02, E10, P3, N0,
      ST_PYR, 5, P3, P0, E00, E02, N0,
      ST_PYR, 5, E00, P0, P5, E09, N0,
      ST_TET, 4, P5, E05, E09, N0,
      /* case 199 */ 2,
      ST_WDG, 6, E05, P5, E09, E07, P4, E08,
      ST_TET, 4, E02, P3, E03, E10,
      /* case 200 */ 9,
      ST_PNT, 5, E03, E07, E02, E11, E05,
      ST_PYR, 5, P0, P4, P5, P1, N0,
      ST_TET, 4, P5, N0, P2, P1,
      ST_PYR, 5, P4, E07, E05, P5, N0,
      ST_PYR, 5, E03, E07, P4, P0, N0,
      ST_TET, 4, P2, P0, P1, N0,
      ST_PYR, 5, P2, E02, E03, P0, N0,
      ST_TET, 4, P2, E11, E02, N0,
      ST_PYR, 5, E05, E11, P2, P5, N0,
      /* case 201 */ 9,
      ST_PNT, 6, E02, E11, E05, E07, E08, E00,
      ST_TET, 4, P5, P1, P4, N0,
      ST_PYR, 5, P5, P4, E07, E05, N0,
      ST_TET, 4, P2, P1, P5, N0,
      ST_PYR, 5, P2, P5, E05, E11, N0,
      ST_TET, 4, E02, P2, E11, N0,
      ST_PYR, 5, P2, E02, E00, P1, N0,
      ST_PYR, 5, E00, E08, P4, P1, N0,
      ST_TET, 4, P4, E08, E07, N0,
      /* case 202 */ 8,
      ST_PNT, 5, E07, E05, E03, E00, E09,
      ST_TET, 4, P2, E11, E02, E01,
      ST_PYR, 5, P5, P4, E07, E05, N0,
      ST_PYR, 5, P4, P0, E03, E07, N0,
      ST_TET, 4, P0, E00, E03, N0,
      ST_TET, 4, P4, P5, P0, N0,
      ST_PYR, 5, P0, P5, E09, E00, N0,
      ST_TET, 4, P5, E05, E09, N0,
      /* case 203 */ 2,
      ST_WDG, 6, E05, P5, E09, E07, P4, E08,
      ST_TET, 4, E02, E01, P2, E11,
      /* case 204 */ 1,
      ST_HEX, 8, E03, E01, E05, E07, P0, P1, P5, P4,
      /* case 205 */ 7,
      ST_PNT, 5, E01, E05, E07, E00, E08,
      ST_PYR, 5, P5, P4, E07, E05, N0,
      ST_TET, 4, P4, E08, E07, N0,
      ST_PYR, 5, P1, P5, E05, E01, N0,
      ST_TET, 4, E00, P1, E01, N0,
      ST_PYR, 5, E08, P4, P1, E00, N0,
      ST_TET, 4, P4, P5, P1, N0,
      /* case 206 */ 7,
      ST_PNT, 5, E05, E07, E03, E09, E00,
      ST_PYR, 5, P4, P0, E03, E07, N0,
      ST_TET, 4, P0, E00, E03, N0,
      ST_PYR, 5, P5, P4, E07, E05, N0,
      ST_TET, 4, E09, P5, E05, N0,
      ST_PYR, 5, E00, P0, P5, E09, N0,
      ST_TET, 4, P0, P4, P5, N0,
      /* case 207 */ 1,
      ST_WDG, 6, P5, E09, E05, P4, E08, E07,
      /* case 208 */ 9,
      ST_PNT, 5, E08, E10, E04, E05, E11,
      ST_PYR, 5, P0, P1, P2, P3, N0,
      ST_TET, 4, P2, P5, N0, P1,
      ST_PYR, 5, P3, P2, E11, E10, N0,
      ST_PYR, 5, E08, P0, P3, E10, N0,
      ST_TET, 4, P5, P1, P0, N0,
      ST_PYR, 5, P5, P0, E08, E04, N0,
      ST_TET, 4, P5, E04, E05, N0,
      ST_PYR, 5, E11, P2, P5, E05, N0,
      /* case 209 */ 9,
      ST_PNT, 6, E04, E05, E11, E10, E03, E00,
      ST_TET, 4, P2, P3, P1, N0,
      ST_PYR, 5, P2, E11, E10, P3, N0,
      ST_TET, 4, P5, P2, P1, N0,
      ST_PYR, 5, P5, E05, E11, P2, N0,
      ST_TET, 4, E04, E05, P5, N0,
      ST_PYR, 5, P5, P1, E00, E04, N0,
      ST_PYR, 5, E00, P1, P3, E03, N0,
      ST_TET, 4, P3, E10, E03, N0,
      /* case 210 */ 8,
      ST_PNT, 5, E10, E11, E08, E00, E01,
      ST_TET, 4, P5, E04, E05, E09,
      ST_PYR, 5, P2, E11, E10, P3, N0,
      ST_PYR, 5, P3, E10, E08, P0, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_TET, 4, P3, P0, P2, N0,
      ST_PYR, 5, P0, E00, E01, P2, N0,
      ST_TET, 4, P2, E01, E11, N0,
      /* case 211 */ 2,
      ST_WDG, 6, E10, P3, E03, E11, P2, E01,
      ST_TET, 4, E04, P5, E09, E05,
      /* case 212 */ 9,
      ST_PNT, 6, E10, E02, E01, E05, E04, E08,
      ST_TET, 4, P1, P0, P5, N0,
      ST_PYR, 5, P1, P5, E05, E01, N0,
      ST_TET, 4, P3, P0, P1, N0,
      ST_PYR, 5, P3, P1, E01, E02, N0,
      ST_TET, 4, E10, P3, E02, N0,
      ST_PYR, 5, P3, E10, E08, P0, N0,
      ST_PYR, 5, E08, E04, P5, P0, N0,
      ST_TET, 4, P5, E04, E05, N0,
      /* case 213 */ 2,
      ST_WDG, 6, E01, P1, E00, E05, P5, E04,
      ST_TET, 4, E10, E03, P3, E02,
      /* case 214 */ 2,
      ST_WDG, 6, E08, P0, E00, E10, P3, E02,
      ST_TET, 4, E05, E09, P5, E04,
      /* case 215 */ 2,
      ST_TET, 4, E05, E04, E09, P5,
      ST_TET, 4, E02, E03, E10, P3,
      /* case 216 */ 5,
      ST_TET, 4, P2, P0, P1, P5,
      ST_PYR, 5, E04, P5, P0, E08, E03,
      ST_PYR, 5, E05, E11, P2, P5, E02,
      ST_PYR, 5, P0, P2, E02, E03, P5,
      ST_PYR, 5, E03, E02, E05, E04, P5,
      /* case 217 */ 7,
      ST_PNT, 5, E02, E00, E04, E11, E05,
      ST_PYR, 5, P1, E00, E04, P5, N0,
      ST_TET, 4, P5, E04, E05, N0,
      ST_PYR, 5, P2, E02, E00, P1, N0,
      ST_TET, 4, E11, E02, P2, N0,
      ST_PYR, 5, E05, E11, P2, P5, N0,
      ST_TET, 4, P5, P2, P1, N0,
      /* case 218 */ 3,
      ST_TET, 4, E02, E11, E01, P2,
      ST_TET, 4, E03, E08, P0, E00,
      ST_TET, 4, E05, P5, E04, E09,
      /* case 219 */ 2,
      ST_TET, 4, E11, E01, E02, P2,
      ST_TET, 4, E05, E04, E09, P5,
      /* case 220 */ 7,
      ST_PNT, 5, E05, E01, E03, E04, E08,
      ST_PYR, 5, P1, E01, E03, P0, N0,
      ST_TET, 4, P0, E03, E08, N0,
      ST_PYR, 5, P5, E05, E01, P1, N0,
      ST_TET, 4, E04, E05, P5, N0,
      ST_PYR, 5, E08, E04, P5, P0, N0,
      ST_TET, 4, P0, P5, P1, N0,
      /* case 221 */ 1,
      ST_WDG, 6, P1, E00, E01, P5, E04, E05,
      /* case 222 */ 2,
      ST_TET, 4, E00, E08, E03, P0,
      ST_TET, 4, E09, E05, E04, P5,
      /* case 223 */ 1,
      ST_TET, 4, E09, E05, E04, P5,
      /* case 224 */ 9,
      ST_PNT, 5, E10, E11, E07, E04, E09,
      ST_PYR, 5, P3, P0, P1, P2, N0,
      ST_TET, 4, P1, P4, N0, P0,
      ST_PYR, 5, P2, P1, E09, E11, N0,
      ST_PYR, 5, E10, P3, P2, E11, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_PYR, 5, P4, P3, E10, E07, N0,
      ST_TET, 4, P4, E07, E04, N0,
      ST_PYR, 5, E09, P1, P4, E04, N0,
      /* case 225 */ 8,
      ST_PNT, 5, E11, E09, E10, E03, E00,
      ST_TET, 4, P4, E07, E04, E08,
      ST_PYR, 5, P1, E09, E11, P2, N0,
      ST_PYR, 5, P2, E11, E10, P3, N0,
      ST_TET, 4, P3, E10, E03, N0,
      ST_TET, 4, P2, P3, P1, N0,
      ST_PYR, 5, P3, E03, E00, P1, N0,
      ST_TET, 4, P1, E00, E09, N0,
      /* case 226 */ 9,
      ST_PNT, 6, E04, E07, E10, E11, E01, E00,
      ST_TET, 4, P3, P0, P2, N0,
      ST_PYR, 5, P3, P2, E11, E10, N0,
      ST_TET, 4, P4, P0, P3, N0,
      ST_PYR, 5, P4, P3, E10, E07, N0,
      ST_TET, 4, E04, P4, E07, N0,
      ST_PYR, 5, P4, E04, E00, P0, N0,
      ST_PYR, 5, E00, E01, P2, P0, N0,
      ST_TET, 4, P2, E01, E11, N0,
      /* case 227 */ 2,
      ST_WDG, 6, E10, P3, E03, E11, P2, E01,
      ST_TET, 4, E04, E08, P4, E07,
      /* case 228 */ 5,
      ST_TET, 4, P3, P0, P1, P4,
      ST_PYR, 5, E04, E09, P1, P4, E01,
      ST_PYR, 5, E07, P4, P3, E10, E02,
      ST_PYR, 5, P1, E01, E02, P3, P4,
      ST_PYR, 5, E01, E04, E07, E02, P4,
      /* case 229 */ 3,
      ST_TET, 4, E02, E03, E10, P3,
      ST_TET, 4, E01, P1, E09, E00,
      ST_TET, 4, E07, E04, P4, E08,
      /* case 230 */ 7,
      ST_PNT, 5, E02, E00, E04, E10, E07,
      ST_PYR, 5, P0, P4, E04, E00, N0,
      ST_TET, 4, P4, E07, E04, N0,
      ST_PYR, 5, P3, P0, E00, E02, N0,
      ST_TET, 4, E10, P3, E02, N0,
      ST_PYR, 5, E07, P4, P3, E10, N0,
      ST_TET, 4, P4, P0, P3, N0,
      /* case 231 */ 2,
      ST_TET, 4, E10, E02, E03, P3,
      ST_TET, 4, E07, E08, E04, P4,
      /* case 232 */ 9,
      ST_PNT, 6, E07, E04, E09, E11, E02, E03,
      ST_TET, 4, P1, P2, P0, N0,
      ST_PYR, 5, P1, E09, E11, P2, N0,
      ST_TET, 4, P4, P1, P0, N0,
      ST_PYR, 5, P4, E04, E09, P1, N0,
      ST_TET, 4, E07, E04, P4, N0,
      ST_PYR, 5, P4, P0, E03, E07, N0,
      ST_PYR, 5, E03, P0, P2, E02, N0,
      ST_TET, 4, P2, E11, E02, N0,
      /* case 233 */ 2,
      ST_WDG, 6, E11, P2, E02, E09, P1, E00,
      ST_TET, 4, E07, P4, E08, E04,
      /* case 234 */ 2,
      ST_WDG, 6, E07, P4, E04, E03, P0, E00,
      ST_TET, 4, E11, P2, E01, E02,
      /* case 235 */ 2,
      ST_TET, 4, E11, E01, E02, P2,
      ST_TET, 4, E04, E07, E08, P4,
      /* case 236 */ 7,
      ST_PNT, 5, E07, E03, E01, E04, E09,
      ST_PYR, 5, P0, P1, E01, E03, N0,
      ST_TET, 4, P1, E09, E01, N0,
      ST_PYR, 5, P4, P0, E03, E07, N0,
      ST_TET, 4, E04, P4, E07, N0,
      ST_PYR, 5, E09, P1, P4, E04, N0,
      ST_TET, 4, P1, P0, P4, N0,
      /* case 237 */ 2,
      ST_TET, 4, E04, E07, E08, P4,
      ST_TET, 4, E09, E00, E01, P1,
      /* case 238 */ 1,
      ST_WDG, 6, P4, E04, E07, P0, E00, E03,
      /* case 239 */ 1,
      ST_TET, 4, E07, E08, E04, P4,
      /* case 240 */ 1,
      ST_HEX, 8, P0, P1, P2, P3, E08, E09, E11, E10,
      /* case 241 */ 7,
      ST_PNT, 5, E09, E11, E10, E00, E03,
      ST_PYR, 5, P2, E11, E10, P3, N0,
      ST_TET, 4, P3, E10, E03, N0,
      ST_PYR, 5, P1, E09, E11, P2, N0,
      ST_TET, 4, E00, E09, P1, N0,
      ST_PYR, 5, E03, E00, P1, P3, N0,
      ST_TET, 4, P3, P1, P2, N0,
      /* case 242 */ 7,
      ST_PNT, 5, E11, E10, E08, E01, E00,
      ST_PYR, 5, P3, E10, E08, P0, N0,
      ST_TET, 4, P0, E08, E00, N0,
      ST_PYR, 5, P2, E11, E10, P3, N0,
      ST_TET, 4, E01, E11, P2, N0,
      ST_PYR, 5, E00, E01, P2, P0, N0,
      ST_TET, 4, P0, P2, P3, N0,
      /* case 243 */ 1,
      ST_WDG, 6, P3, E03, E10, P2, E01, E11,
      /* case 244 */ 7,
      ST_PNT, 5, E10, E08, E09, E02, E01,
      ST_PYR, 5, P0, E08, E09, P1, N0,
      ST_TET, 4, P1, E09, E01, N0,
      ST_PYR, 5, P3, E10, E08, P0, N0,
      ST_TET, 4, E02, E10, P3, N0,
      ST_PYR, 5, E01, E02, P3, P1, N0,
      ST_TET, 4, P1, P3, P0, N0,
      /* case 245 */ 2,
      ST_TET, 4, E02, E03, E10, P3,
      ST_TET, 4, E01, E09, E00, P1,
      /* case 246 */ 1,
      ST_WDG, 6, P0, E00, E08, P3, E02, E10,
      /* case 247 */ 1,
      ST_TET, 4, E10, E02, E03, P3,
      /* case 248 */ 7,
      ST_PNT, 5, E11, E09, E08, E02, E03,
      ST_PYR, 5, P1, P0, E08, E09, N0,
      ST_TET, 4, P0, E03, E08, N0,
      ST_PYR, 5, P2, P1, E09, E11, N0,
      ST_TET, 4, E02, P2, E11, N0,
      ST_PYR, 5, E03, P0, P2, E02, N0,
      ST_TET, 4, P0, P1, P2, N0,
      /* case 249 */ 1,
      ST_WDG, 6, P2, E02, E11, P1, E00, E09,
      /* case 250 */ 2,
      ST_TET, 4, E00, E08, E03, P0,
      ST_TET, 4, E01, E02, E11, P2,
      /* case 251 */ 1,
      ST_TET, 4, E01, E02, E11, P2,
      /* case 252 */ 1,
      ST_WDG, 6, P0, E08, E03, P1, E09, E01,
      /* case 253 */ 1,
      ST_TET, 4, E01, E09, E00, P1,
      /* case 254 */ 1,
      ST_TET, 4, E03, E00, E08, P0,
      /* case 255 */ 0,
      // vtkm::CELL_SHAPE_WEDGE
      /* case 0 */ 1,
      ST_WDG, 6, P0, P1, P2, P3, P4, P5,
      /* case 1 */ 2,
      ST_WDG, 6, E00, E02, E06, P1, P2, P3,
      ST_PYR, 5, P1, P2, P5, P4, P3,
      /* case 2 */ 2,
      ST_WDG, 6, E01, E00, E07, P2, P0, P4,
      ST_PYR, 5, P2, P0, P3, P5, P4,
      /* case 3 */ 8,
      ST_PNT, 7, P2, P3, P4, E01, E02, E06, E07,
      ST_TET, 4, P4, P5, P3, P2,
      ST_TET, 4, P2, P3, P4, N0,
      ST_PYR, 5, E06, E07, P4, P3, N0,
      ST_PYR, 5, E01, E07, E06, E02, N0,
      ST_TET, 4, P2, E01, E02, N0,
      ST_PYR, 5, P2, E02, E06, P3, N0,
      ST_PYR, 5, E07, E01, P2, P4, N0,
      /* case 4 */ 2,
      ST_WDG, 6, E02, E01, E08, P0, P1, P5,
      ST_PYR, 5, P0, P1, P4, P3, P5,
      /* case 5 */ 8,
      ST_PNT, 7, P1, P5, P3, E00, E01, E08, E06,
      ST_TET, 4, P3, P4, P5, P1,
      ST_TET, 4, P1, P5, P3, N0,
      ST_PYR, 5, E08, E06, P3, P5, N0,
      ST_PYR, 5, E00, E06, E08, E01, N0,
      ST_TET, 4, P1, E00, E01, N0,
      ST_PYR, 5, P1, E01, E08, P5, N0,
      ST_PYR, 5, E06, E00, P1, P3, N0,
      /* case 6 */ 8,
      ST_PNT, 7, P0, P4, P5, E02, E00, E07, E08,
      ST_TET, 4, P5, P3, P4, P0,
      ST_TET, 4, P0, P4, P5, N0,
      ST_PYR, 5, E07, E08, P5, P4, N0,
      ST_PYR, 5, E02, E08, E07, E00, N0,
      ST_TET, 4, P0, E02, E00, N0,
      ST_PYR, 5, P0, E00, E07, P4, N0,
      ST_PYR, 5, E08, E02, P0, P5, N0,
      /* case 7 */ 1,
      ST_WDG, 6, E06, E07, E08, P3, P4, P5,
      /* case 8 */ 2,
      ST_WDG, 6, P4, P5, P0, E03, E05, E06,
      ST_PYR, 5, P4, P1, P2, P5, P0,
      /* case 9 */ 1,
      ST_HEX, 8, P1, P2, P5, P4, E00, E02, E05, E03,
      /* case 10 */ 9,
      ST_PNT, 6, E00, E01, E07, E03, E05, E06,
      ST_PYR, 5, P5, P0, E06, E05, N0,
      ST_TET, 4, P0, E00, E06, N0,
      ST_PYR, 5, P0, P2, E01, E00, N0,
      ST_TET, 4, P5, P2, P0, N0,
      ST_PYR, 5, P4, E07, E01, P2, N0,
      ST_TET, 4, P5, P4, P2, N0,
      ST_PYR, 5, E05, E03, P4, P5, N0,
      ST_TET, 4, E03, E07, P4, N0,
      /* case 11 */ 7,
      ST_PNT, 5, E01, E02, E05, E03, E07,
      ST_PYR, 5, P4, P5, E05, E03, N0,
      ST_TET, 4, E03, E07, P4, N0,
      ST_PYR, 5, E02, E05, P5, P2, N0,
      ST_PYR, 5, E01, P2, P4, E07, N0,
      ST_TET, 4, P4, P2, P5, N0,
      ST_TET, 4, P2, E01, E02, N0,
      /* case 12 */ 9,
      ST_PNT, 6, E05, E03, E06, E02, E01, E08,
      ST_PYR, 5, P1, E01, E08, P5, N0,
      ST_TET, 4, P5, E08, E05, N0,
      ST_PYR, 5, P5, E05, E03, P4, N0,
      ST_TET, 4, P1, P5, P4, N0,
      ST_PYR, 5, P0, P4, E03, E06, N0,
      ST_TET, 4, P1, P4, P0, N0,
      ST_PYR, 5, E01, P1, P0, E02, N0,
      ST_TET, 4, E02, P0, E06, N0,
      /* case 13 */ 7,
      ST_PNT, 5, E01, E00, E03, E05, E08,
      ST_PYR, 5, P5, E05, E03, P4, N0,
      ST_TET, 4, E05, P5, E08, N0,
      ST_PYR, 5, E00, P1, P4, E03, N0,
      ST_PYR, 5, E01, E08, P5, P1, N0,
      ST_TET, 4, P5, P4, P1, N0,
      ST_TET, 4, P1, E00, E01, N0,
      /* case 14 */ 2,
      ST_TET, 4, P0, E02, E00, E06,
      ST_WDG, 6, E05, E08, P5, E03, E07, P4,
      /* case 15 */ 1,
      ST_WDG, 6, E03, P4, E07, E05, P5, E08,
      /* case 16 */ 2,
      ST_WDG, 6, P5, P3, P1, E04, E03, E07,
      ST_PYR, 5, P5, P2, P0, P3, P1,
      /* case 17 */ 9,
      ST_PNT, 6, E03, E04, E07, E00, E02, E06,
      ST_PYR, 5, P2, E02, E06, P3, N0,
      ST_TET, 4, P3, E06, E03, N0,
      ST_PYR, 5, P3, E03, E04, P5, N0,
      ST_TET, 4, P2, P3, P5, N0,
      ST_PYR, 5, P1, P5, E04, E07, N0,
      ST_TET, 4, P2, P5, P1, N0,
      ST_PYR, 5, E02, P2, P1, E00, N0,
      ST_TET, 4, E00, P1, E07, N0,
      /* case 18 */ 1,
      ST_HEX, 8, P2, P0, P3, P5, E01, E00, E03, E04,
      /* case 19 */ 7,
      ST_PNT, 5, E02, E01, E04, E03, E06,
      ST_PYR, 5, P3, E03, E04, P5, N0,
      ST_TET, 4, E03, P3, E06, N0,
      ST_PYR, 5, E01, P2, P5, E04, N0,
      ST_PYR, 5, E02, E06, P3, P2, N0,
      ST_TET, 4, P3, P5, P2, N0,
      ST_TET, 4, P2, E01, E02, N0,
      /* case 20 */ 9,
      ST_PNT, 6, E01, E02, E08, E04, E03, E07,
      ST_PYR, 5, P3, P1, E07, E03, N0,
      ST_TET, 4, P1, E01, E07, N0,
      ST_PYR, 5, P1, P0, E02, E01, N0,
      ST_TET, 4, P3, P0, P1, N0,
      ST_PYR, 5, P5, E08, E02, P0, N0,
      ST_TET, 4, P3, P5, P0, N0,
      ST_PYR, 5, E03, E04, P5, P3, N0,
      ST_TET, 4, E04, E08, P5, N0,
      /* case 21 */ 2,
      ST_TET, 4, P1, E00, E01, E07,
      ST_WDG, 6, E03, E06, P3, E04, E08, P5,
      /* case 22 */ 7,
      ST_PNT, 5, E02, E00, E03, E04, E08,
      ST_PYR, 5, P5, P3, E03, E04, N0,
      ST_TET, 4, E04, E08, P5, N0,
      ST_PYR, 5, E00, E03, P3, P0, N0,
      ST_PYR, 5, E02, P0, P5, E08, N0,
      ST_TET, 4, P5, P0, P3, N0,
      ST_TET, 4, P0, E02, E00, N0,
      /* case 23 */ 1,
      ST_WDG, 6, E04, P5, E08, E03, P3, E06,
      /* case 24 */ 8,
      ST_PNT, 7, P5, P0, P1, E04, E05, E06, E07,
      ST_TET, 4, P1, P0, P2, P5,
      ST_TET, 4, P5, P1, P0, N0,
      ST_PYR, 5, E06, P0, P1, E07, N0,
      ST_PYR, 5, E04, E05, E06, E07, N0,
      ST_TET, 4, P5, E05, E04, N0,
      ST_PYR, 5, P5, P0, E06, E05, N0,
      ST_PYR, 5, E07, P1, P5, E04, N0,
      /* case 25 */ 7,
      ST_PNT, 5, E04, E05, E02, E00, E07,
      ST_PYR, 5, P1, E00, E02, P2, N0,
      ST_TET, 4, E00, P1, E07, N0,
      ST_PYR, 5, E05, P5, P2, E02, N0,
      ST_PYR, 5, E04, E07, P1, P5, N0,
      ST_TET, 4, P1, P2, P5, N0,
      ST_TET, 4, P5, E05, E04, N0,
      /* case 26 */ 7,
      ST_PNT, 5, E05, E04, E01, E00, E06,
      ST_PYR, 5, P0, P2, E01, E00, N0,
      ST_TET, 4, E00, E06, P0, N0,
      ST_PYR, 5, E04, E01, P2, P5, N0,
      ST_PYR, 5, E05, P5, P0, E06, N0,
      ST_TET, 4, P0, P5, P2, N0,
      ST_TET, 4, P5, E05, E04, N0,
      /* case 27 */ 1,
      ST_WDG, 6, E05, P5, E04, E02, P2, E01,
      /* case 28 */ 2,
      ST_TET, 4, P5, E05, E04, E08,
      ST_WDG, 6, E02, E06, P0, E01, E07, P1,
      /* case 29 */ 2,
      ST_TET, 4, P1, E00, E01, E07,
      ST_TET, 4, E05, E04, P5, E08,
      /* case 30 */ 2,
      ST_TET, 4, P5, E05, E04, E08,
      ST_TET, 4, E00, P0, E02, E06,
      /* case 31 */ 1,
      ST_TET, 4, E05, E08, E04, P5,
      /* case 32 */ 2,
      ST_WDG, 6, P3, P4, P2, E05, E04, E08,
      ST_PYR, 5, P3, P0, P1, P4, P2,
      /* case 33 */ 9,
      ST_PNT, 6, E02, E00, E06, E05, E04, E08,
      ST_PYR, 5, P4, P2, E08, E04, N0,
      ST_TET, 4, P2, E02, E08, N0,
      ST_PYR, 5, P2, P1, E00, E02, N0,
      ST_TET, 4, P4, P1, P2, N0,
      ST_PYR, 5, P3, E06, E00, P1, N0,
      ST_TET, 4, P4, P3, P1, N0,
      ST_PYR, 5, E04, E05, P3, P4, N0,
      ST_TET, 4, E05, E06, P3, N0,
      /* case 34 */ 9,
      ST_PNT, 6, E04, E05, E08, E01, E00, E07,
      ST_PYR, 5, P0, E00, E07, P4, N0,
      ST_TET, 4, P4, E07, E04, N0,
      ST_PYR, 5, P4, E04, E05, P3, N0,
      ST_TET, 4, P0, P4, P3, N0,
      ST_PYR, 5, P2, P3, E05, E08, N0,
      ST_TET, 4, P0, P3, P2, N0,
      ST_PYR, 5, E00, P0, P2, E01, N0,
      ST_TET, 4, E01, P2, E08, N0,
      /* case 35 */ 2,
      ST_TET, 4, P2, E01, E02, E08,
      ST_WDG, 6, E04, E07, P4, E05, E06, P3,
      /* case 36 */ 1,
      ST_HEX, 8, P0, P1, P4, P3, E02, E01, E04, E05,
      /* case 37 */ 7,
      ST_PNT, 5, E00, E01, E04, E05, E06,
      ST_PYR, 5, P3, P4, E04, E05, N0,
      ST_TET, 4, E05, E06, P3, N0,
      ST_PYR, 5, E01, E04, P4, P1, N0,
      ST_PYR, 5, E00, P1, P3, E06, N0,
      ST_TET, 4, P3, P1, P4, N0,
      ST_TET, 4, P1, E00, E01, N0,
      /* case 38 */ 7,
      ST_PNT, 5, E00, E02, E05, E04, E07,
      ST_PYR, 5, P4, E04, E05, P3, N0,
      ST_TET, 4, E04, P4, E07, N0,
      ST_PYR, 5, E02, P0, P3, E05, N0,
      ST_PYR, 5, E00, E07, P4, P0, N0,
      ST_TET, 4, P4, P3, P0, N0,
      ST_TET, 4, P0, E02, E00, N0,
      /* case 39 */ 1,
      ST_WDG, 6, E05, P3, E06, E04, P4, E07,
      /* case 40 */ 8,
      ST_PNT, 7, P4, P2, P0, E03, E04, E08, E06,
      ST_TET, 4, P0, P2, P1, P4,
      ST_TET, 4, P4, P0, P2, N0,
      ST_PYR, 5, E08, P2, P0, E06, N0,
      ST_PYR, 5, E03, E04, E08, E06, N0,
      ST_TET, 4, P4, E04, E03, N0,
      ST_PYR, 5, P4, P2, E08, E04, N0,
      ST_PYR, 5, E06, P0, P4, E03, N0,
      /* case 41 */ 7,
      ST_PNT, 5, E04, E03, E00, E02, E08,
      ST_PYR, 5, P2, P1, E00, E02, N0,
      ST_TET, 4, E02, E08, P2, N0,
      ST_PYR, 5, E03, E00, P1, P4, N0,
      ST_PYR, 5, E04, P4, P2, E08, N0,
      ST_TET, 4, P2, P4, P1, N0,
      ST_TET, 4, P4, E04, E03, N0,
      /* case 42 */ 2,
      ST_TET, 4, P4, E04, E03, E07,
      ST_WDG, 6, E01, E08, P2, E00, E06, P0,
      /* case 43 */ 2,
      ST_TET, 4, P4, E04, E03, E07,
      ST_TET, 4, E02, P2, E01, E08,
      /* case 44 */ 7,
      ST_PNT, 5, E03, E04, E01, E02, E06,
      ST_PYR, 5, P0, E02, E01, P1, N0,
      ST_TET, 4, E02, P0, E06, N0,
      ST_PYR, 5, E04, P4, P1, E01, N0,
      ST_PYR, 5, E03, E06, P0, P4, N0,
      ST_TET, 4, P0, P1, P4, N0,
      ST_TET, 4, P4, E04, E03, N0,
      /* case 45 */ 1,
      ST_WDG, 6, E04, P4, E03, E01, P1, E00,
      /* case 46 */ 2,
      ST_TET, 4, P0, E02, E00, E06,
      ST_TET, 4, E04, E03, P4, E07,
      /* case 47 */ 1,
      ST_TET, 4, E04, E07, E03, P4,
      /* case 48 */ 8,
      ST_PNT, 7, P3, P1, P2, E05, E03, E07, E08,
      ST_TET, 4, P2, P1, P0, P3,
      ST_TET, 4, P3, P2, P1, N0,
      ST_PYR, 5, E07, P1, P2, E08, N0,
      ST_PYR, 5, E05, E03, E07, E08, N0,
      ST_TET, 4, P3, E03, E05, N0,
      ST_PYR, 5, P3, P1, E07, E03, N0,
      ST_PYR, 5, E08, P2, P3, E05, N0,
      /* case 49 */ 2,
      ST_TET, 4, P3, E03, E05, E06,
      ST_WDG, 6, E00, E07, P1, E02, E08, P2,
      /* case 50 */ 7,
      ST_PNT, 5, E05, E03, E00, E01, E08,
      ST_PYR, 5, P2, E01, E00, P0, N0,
      ST_TET, 4, E01, P2, E08, N0,
      ST_PYR, 5, E03, P3, P0, E00, N0,
      ST_PYR, 5, E05, E08, P2, P3, N0,
      ST_TET, 4, P2, P0, P3, N0,
      ST_TET, 4, P3, E03, E05, N0,
      /* case 51 */ 2,
      ST_TET, 4, P2, E01, E02, E08,
      ST_TET, 4, E03, E05, P3, E06,
      /* case 52 */ 7,
      ST_PNT, 5, E03, E05, E02, E01, E07,
      ST_PYR, 5, P1, P0, E02, E01, N0,
      ST_TET, 4, E01, E07, P1, N0,
      ST_PYR, 5, E05, E02, P0, P3, N0,
      ST_PYR, 5, E03, P3, P1, E07, N0,
      ST_TET, 4, P1, P3, P0, N0,
      ST_TET, 4, P3, E03, E05, N0,
      /* case 53 */ 2,
      ST_TET, 4, P3, E03, E05, E06,
      ST_TET, 4, E01, P1, E00, E07,
      /* case 54 */ 1,
      ST_WDG, 6, E03, P3, E05, E00, P0, E02,
      /* case 55 */ 1,
      ST_TET, 4, E03, E06, E05, P3,
      /* case 56 */ 1,
      ST_WDG, 6, P0, P1, P2, E06, E07, E08,
      /* case 57 */ 1,
      ST_WDG, 6, E02, P2, E08, E00, P1, E07,
      /* case 58 */ 1,
      ST_WDG, 6, E00, P0, E06, E01, P2, E08,
      /* case 59 */ 1,
      ST_TET, 4, E02, E01, E08, P2,
      /* case 60 */ 1,
      ST_WDG, 6, E01, P1, E07, E02, P0, E06,
      /* case 61 */ 1,
      ST_TET, 4, E01, E00, E07, P1,
      /* case 62 */ 1,
      ST_TET, 4, E00, E02, E06, P0,
      /* case 63 */ 0,
      // vtkm::CELL_SHAPE_PYRAMID
      /* case 0 */ 1,
      ST_PYR, 5, P0, P1, P2, P3, P4,
      /* case 1 */ 2,
      ST_WDG, 6, E00, E04, E03, P1, P4, P3,
      ST_TET, 4, P1, P2, P3, P4,
      /* case 2 */ 2,
      ST_WDG, 6, E01, E05, E00, P2, P4, P0,
      ST_TET, 4, P2, P3, P0, P4,
      /* case 3 */ 7,
      ST_PNT, 7, P4, E05, E04, E01, E03, P2, P3,
      ST_TET, 4, E04, P4, E05, N0,
      ST_PYR, 5, E01, E03, E04, E05, N0,
      ST_PYR, 5, E01, E05, P4, P2, N0,
      ST_TET, 4, P2, P4, P3, N0,
      ST_PYR, 5, P3, P4, E04, E03, N0,
      ST_PYR, 5, P2, P3, E03, E01, N0,
      /* case 4 */ 2,
      ST_WDG, 6, E02, E06, E01, P3, P4, P1,
      ST_TET, 4, P3, P0, P1, P4,
      /* case 5 */ 2,
      ST_WDG, 6, E04, P4, E06, E00, P1, E01,
      ST_WDG, 6, P4, E04, E06, P3, E03, E02,
      /* case 6 */ 7,
      ST_PNT, 7, P4, E06, E05, E02, E00, P3, P0,
      ST_TET, 4, E05, P4, E06, N0,
      ST_PYR, 5, E02, E00, E05, E06, N0,
      ST_PYR, 5, E02, E06, P4, P3, N0,
      ST_TET, 4, P3, P4, P0, N0,
      ST_PYR, 5, P0, P4, E05, E00, N0,
      ST_PYR, 5, P3, P0, E00, E02, N0,
      /* case 7 */ 2,
      ST_TET, 4, E04, E05, E06, P4,
      ST_WDG, 6, E02, E03, P3, E06, E04, P4,
      /* case 8 */ 2,
      ST_WDG, 6, E03, E07, E02, P0, P4, P2,
      ST_TET, 4, P0, P1, P2, P4,
      /* case 9 */ 7,
      ST_PNT, 7, P4, E04, E07, E00, E02, P1, P2,
      ST_TET, 4, E07, P4, E04, N0,
      ST_PYR, 5, E00, E02, E07, E04, N0,
      ST_PYR, 5, E00, E04, P4, P1, N0,
      ST_TET, 4, P1, P4, P2, N0,
      ST_PYR, 5, P2, P4, E07, E02, N0,
      ST_PYR, 5, P1, P2, E02, E00, N0,
      /* case 10 */ 2,
      ST_WDG, 6, E07, P4, E05, E03, P0, E00,
      ST_WDG, 6, P4, E07, E05, P2, E02, E01,
      /* case 11 */ 2,
      ST_TET, 4, E07, E04, E05, P4,
      ST_WDG, 6, E01, E02, P2, E05, E07, P4,
      /* case 12 */ 7,
      ST_PNT, 7, P4, E07, E06, E03, E01, P0, P1,
      ST_TET, 4, E06, P4, E07, N0,
      ST_PYR, 5, E03, E01, E06, E07, N0,
      ST_PYR, 5, E03, E07, P4, P0, N0,
      ST_TET, 4, P0, P4, P1, N0,
      ST_PYR, 5, P1, P4, E06, E01, N0,
      ST_PYR, 5, P0, P1, E01, E03, N0,
      /* case 13 */ 2,
      ST_TET, 4, E06, E07, E04, P4,
      ST_WDG, 6, E00, E01, P1, E04, E06, P4,
      /* case 14 */ 2,
      ST_TET, 4, E05, E06, E07, P4,
      ST_WDG, 6, E03, E00, P0, E07, E05, P4,
      /* case 15 */ 1,
      ST_PYR, 5, E04, E05, E06, E07, P4,
      /* case 16 */ 1,
      ST_HEX, 8, P0, P1, P2, P3, E04, E05, E06, E07,
      /* case 17 */ 2,
      ST_WDG, 6, E03, E07, P3, E00, E05, P1,
      ST_WDG, 6, E05, E06, E07, P1, P2, P3,
      /* case 18 */ 2,
      ST_WDG, 6, E00, E04, P0, E01, E06, P2,
      ST_WDG, 6, E06, E07, E04, P2, P3, P0,
      /* case 19 */ 1,
      ST_WDG, 6, E03, E07, P3, E01, E06, P2,
      /* case 20 */ 2,
      ST_WDG, 6, E01, E05, P1, E02, E07, P3,
      ST_WDG, 6, E07, E04, E05, P3, P0, P1,
      /* case 21 */ 2,
      ST_TET, 4, E00, P1, E01, E05,
      ST_TET, 4, P3, E03, E02, E07,
      /* case 22 */ 1,
      ST_WDG, 6, E00, E04, P0, E02, E07, P3,
      /* case 23 */ 1,
      ST_TET, 4, P3, E03, E02, E07,
      /* case 24 */ 2,
      ST_WDG, 6, E02, E06, P2, E03, E04, P0,
      ST_WDG, 6, E04, E05, E06, P0, P1, P2,
      /* case 25 */ 1,
      ST_WDG, 6, E02, E06, P2, E00, E05, P1,
      /* case 26 */ 2,
      ST_TET, 4, E03, P0, E00, E04,
      ST_TET, 4, P2, E02, E01, E06,
      /* case 27 */ 1,
      ST_TET, 4, P2, E02, E01, E06,
      /* case 28 */ 1,
      ST_WDG, 6, E01, E05, P1, E03, E04, P0,
      /* case 29 */ 1,
      ST_TET, 4, P1, E01, E00, E05,
      /* case 30 */ 1,
      ST_TET, 4, P0, E00, E03, E04,
      /* case 31 */ 0,
    };
    // clang-format on
    return ClipTablesData[idx];
  }

  /**
   * Get the the case index for the given shape and caseId.
   */
  VTKM_EXEC inline static vtkm::Id GetCaseIndex(vtkm::UInt8 shape, vtkm::Id caseId)
  {
    // clang-format off
    // Index into ClipTablesData for each shape and configuration
    VTKM_STATIC_CONSTEXPR_ARRAY vtkm::UInt16 ClipTablesIndices[] = {
      // vtkm::CELL_SHAPE_VERTEX
      0, 4, // case 0 - 1
      // vtkm::CELL_SHAPE_LINE
      5, 10, 15, 20, // case 0 - 3
      // vtkm::CELL_SHAPE_TRIANGLE
      21, 27, 34, 41, 47, 54, 60, 66, // case 0 - 7
      // vtkm::CELL_SHAPE_QUAD
      67, 74, 86, 98, 105, 117, 128, 135, // case 0 - 7
      141, 153, 160, 171, 177, 184, 190, 196, // case 8 - 15
      // vtkm::CELL_SHAPE_TETRA
      197, 204, 213, 222, 231, 240, 249, 258, // case 0 - 7
      265, 274, 283, 292, 299, 308, 315, 322, // case 8 - 15
      // vtkm::CELL_SHAPE_HEXAHEDRON
      323, 334, 397, 460, 479, 542, 618, 637, // case 0 - 7
      698, 761, 780, 856, 917, 936, 997, 1058, // case 8 - 15
      1069, 1132, 1151, 1227, 1288, 1329, 1401, 1473, // case 16 - 23
      1534, 1610, 1671, 1712, 1747, 1819, 1880, 1933, // case 24 - 31
      1980, 2043, 2119, 2138, 2199, 2275, 2316, 2377, // case 32 - 39
      2412, 2453, 2525, 2597, 2658, 2730, 2783, 2844, // case 40 - 47
      2891, 2910, 2971, 3032, 3043, 3115, 3168, 3229, // case 48 - 55
      3276, 3348, 3409, 3462, 3509, 3526, 3541, 3556, // case 56 - 63
      3565, 3628, 3669, 3745, 3817, 3836, 3908, 3969, // case 64 - 71
      4030, 4106, 4178, 4219, 4272, 4333, 4394, 4429, // case 72 - 79
      4476, 4552, 4624, 4665, 4718, 4790, 4807, 4860, // case 80 - 87
      4875, 4916, 4969, 4994, 5013, 5066, 5081, 5100, // case 88 - 95
      5113, 5132, 5204, 5265, 5326, 5387, 5440, 5451, // case 96 - 103
      5498, 5570, 5587, 5640, 5655, 5716, 5731, 5778, // case 104 - 111
      5787, 5848, 5909, 5944, 5991, 6052, 6067, 6114, // case 112 - 119
      6123, 6176, 6191, 6210, 6223, 6238, 6251, 6264, // case 120 - 127
      6271, 6334, 6410, 6451, 6523, 6599, 6640, 6712, // case 128 - 135
      6765, 6784, 6845, 6917, 6978, 7039, 7074, 7135, // case 136 - 143
      7182, 7201, 7262, 7334, 7395, 7467, 7520, 7537, // case 144 - 151
      7552, 7613, 7624, 7677, 7724, 7785, 7832, 7847, // case 152 - 159
      7856, 7932, 7973, 8045, 8098, 8139, 8164, 8217, // case 160 - 167
      8236, 8308, 8361, 8378, 8393, 8446, 8465, 8480, // case 168 - 175
      8493, 8554, 8589, 8650, 8697, 8750, 8769, 8784, // case 176 - 183
      8797, 8858, 8905, 8920, 8929, 8944, 8957, 8970, // case 184 - 191
      8977, 8996, 9068, 9140, 9157, 9218, 9271, 9332, // case 192 - 199
      9347, 9408, 9469, 9522, 9537, 9548, 9595, 9642, // case 200 - 207
      9651, 9712, 9773, 9826, 9841, 9902, 9917, 9932, // case 208 - 215
      9945, 9980, 10027, 10046, 10059, 10106, 10115, 10128, // case 216 - 223
      10135, 10196, 10249, 10310, 10325, 10360, 10379, 10426, // case 224 - 231
      10439, 10500, 10515, 10530, 10543, 10590, 10603, 10612, // case 232 - 239
      10619, 10630, 10677, 10724, 10733, 10780, 10793, 10802, // case 240 - 247
      10809, 10856, 10865, 10878, 10885, 10894, 10901, 10908, // case 248 - 255
      // vtkm::CELL_SHAPE_WEDGE
      10909, 10918, 10934, 10950, 11006, 11022, 11078, 11134, // case 0 - 7
      11143, 11159, 11170, 11231, 11278, 11339, 11386, 11401, // case 8 - 15
      11410, 11426, 11487, 11498, 11545, 11606, 11621, 11668, // case 16 - 23
      11677, 11733, 11780, 11827, 11836, 11851, 11864, 11877, // case 24 - 31
      11884, 11900, 11961, 12022, 12037, 12048, 12095, 12142, // case 32 - 39
      12151, 12207, 12254, 12269, 12282, 12329, 12338, 12351, // case 40 - 47
      12358, 12414, 12429, 12476, 12489, 12536, 12549, 12558, // case 48 - 55
      12565, 12574, 12583, 12592, 12599, 12608, 12615, 12622, // case 56 - 63
      // vtkm::CELL_SHAPE_PYRAMID
      12623, 12631, 12646, 12661, 12711, 12726, 12743, 12793, // case 0 - 7
      12808, 12823, 12873, 12890, 12905, 12955, 12970, 12985, // case 8 - 15
      12993, 13004, 13021, 13038, 13047, 13064, 13077, 13086, // case 16 - 23
      13093, 13110, 13119, 13132, 13139, 13148, 13155, 13162, // case 24 - 31
    };
    // clang-format on
    return ClipTablesIndices[ClipTablesBase::GetCellClipTableIndex(shape) + caseId];
  }

  /**
   * Given the number of points and a case index, return if the cell is discarded.
   */
  VTKM_EXEC inline static bool IsCellDiscarded(vtkm::IdComponent numberOfPoints,
                                               vtkm::UInt8 caseIndex)
  {
    return caseIndex == ClipTablesBase::GetMaxCellCase(numberOfPoints);
  }

  /**
   * Given the number of points and a case index, return if the cell is kept.
   */
  VTKM_EXEC inline static constexpr bool IsCellKept(vtkm::IdComponent vtkmNotUsed(numberOfPoints),
                                                    vtkm::UInt8 caseIndex)
  {
    return caseIndex == 0;
  }

  /**
   * Get the discarded cell case index.
   */
  VTKM_EXEC inline static constexpr vtkm::UInt8 GetDiscardedCellCase() { return 255; }

  /**
   * Get the kept cell case index.
   */
  VTKM_EXEC inline static constexpr vtkm::UInt8 GetKeptCellCase() { return 0; }
};
}
}
} // namespace vtkm::worklet::internal

#endif // vtk_m_ClipTables_h
