! $Id$
!
! Earth System Modeling Framework
! Copyright (c) 2002-2025, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!
!==============================================================================
!
program ESMF_AlarmTest

!------------------------------------------------------------------------------
!

#include "ESMF.h"
 
!==============================================================================
!BOP
! !PROGRAM: ESMF_AlarmTest - Test Alarm functionalities
!
! !DESCRIPTION:
!
! The code in this file drives F90 Alarm unit tests.
!
!EOP
!-----------------------------------------------------------------------------
! !USES:
      use ESMF_TestMod      ! test methods
      use ESMF
      implicit none

!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
      character(*), parameter :: version = &
      '$Id$'
!------------------------------------------------------------------------------

      ! cumulative result: count failures; no failures equals "all pass"
      integer :: result = 0

      ! individual test result code
      integer :: rc,localrc

      ! individual test name
      character(ESMF_MAXSTR) :: name

      ! individual test failure message
      character(ESMF_MAXSTR) :: failMsg

      ! instantiate a clock 
      type(ESMF_Clock) :: clock1
      type(ESMF_Alarm) :: alarm1

      ! instantiate a calendar
      type(ESMF_Calendar) :: gregorianCalendar, julianCalendar, &
                             no_leapCalendar, esmf_360dayCalendar

      ! instantiate timestep, start and stop times
      type(ESMF_TimeInterval) :: timeStep, repeatDuration
      type(ESMF_Time) :: startTime, stopTime
      type(ESMF_Time) :: alarmTime, ringTime

      logical :: isCreated

#ifdef ESMF_TESTEXHAUSTIVE
      logical :: bool, correct

      ! instantiate a clock 
      type(ESMF_Clock) :: clock, clock2, clock3, domainClock, CLOCK_ATM, repeatClock
      logical :: alarmCountPass, isringing, sticky, enabled
      integer(ESMF_KIND_I8) :: forwardCount
      logical :: willRingNext, testPass, alarmsNotEqual, alarmsEqual
      type(ESMF_Direction_Flag) :: reverseDirection, forwardDirection
      type(ESMF_Alarm) :: alarm, alarm2, afterAlarm, beforeAlarm
      type(ESMF_Alarm) :: alarmList(201), alarmListOne(1)
      type(ESMF_Alarm) :: alarm6, alarm7, alarm8
      type(ESMF_Alarm) :: alarm5(200), alarm3, alarm4
      type(ESMF_Alarm) :: ALARM_HISTORY

      integer(ESMF_KIND_I8) :: reverseCount, iteration
      integer :: dd, nclock, ringCount, expectedCount, i, yy
      integer :: mm, m, h, sstep, nstep, nring, alarmCount

      ! instantiate timestep, start and stop times
      type(ESMF_TimeInterval) :: TIMEINTERVAL_HISTORY, alarmStep, alarmStep2
      type(ESMF_TimeInterval) :: runDuration, ringDuration, ringInterval
      type(ESMF_Time) :: currentTime, currTime, afterAlarmTime, beforeAlarmTime
      type(ESMF_Time) :: alarmStopTime, nextTime, prevTime, currentTime2, time1

      character(ESMF_MAXSTR) :: aName
#endif


      ! initialize ESMF framework
      call ESMF_TestStart(ESMF_SRCLINE, rc=rc)
      if (rc /= ESMF_SUCCESS) call ESMF_Finalize(endflag=ESMF_END_ABORT)

      ! initialize one calendar to be Gregorian type
      gregorianCalendar = ESMF_CalendarCreate(ESMF_CALKIND_GREGORIAN, &
        name="Gregorian", rc=rc)

      ! initialize secand calendar to be Julian type
      julianCalendar = ESMF_CalendarCreate(ESMF_CALKIND_JULIANDAY, &
        name="Julian", rc=rc)

      ! initialize third calendar to be No Leap type
      no_leapCalendar = ESMF_CalendarCreate(ESMF_CALKIND_NOLEAP, &
        name="NoLeap", rc=rc)

      ! initialize third calendar to be 360 day type
      esmf_360dayCalendar = ESMF_CalendarCreate(ESMF_CALKIND_360DAY, &
        name="360Day", rc=rc)

!-------------------------------------------------------------------------------
!    The unit tests are divided into Sanity and Exhaustive. The Sanity tests are
!    always run. When the environment variable, EXHAUSTIVE, is set to ON then
!    the EXHAUSTIVE and sanity tests both run. If the EXHAUSTIVE variable is set
!    to OFF, then only the sanity unit tests.
!    Special strings (Non-exhaustive and exhaustive) have been
!    added to allow a script to count the number and types of unit tests.
!-------------------------------------------------------------------------------


      ! initialize clock time intervals and instants


      ! ----------------------------------------------------------------------------

      !NEX_UTest
      ! Test Setting Time Step
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Set Time Interval Initialization Test"
      call ESMF_TimeIntervalSet(timeStep, h=1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !NEX_UTest
      ! Test Setting the Start Time
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Set Start Time Initialization Test"
      call ESMF_TimeSet(startTime, yy=2003, mm=3, dd=13, &
                                   h=18, m=45, s=27, &
                                   calendar=gregorianCalendar, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !NEX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Stop Time Initialization Test"
      call ESMF_TimeSet(stopTime, yy=2003, mm=3, dd=24, &
                                  calendar=gregorianCalendar, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------
      !NEX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Alarm Time Initialization Test"
      call ESMF_TimeSet(alarmTime, yy=2003, mm=9, dd=15, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !NEX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Clock Initialization Test"
      clock1 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                               name="Clock 1", rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !NEX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Create Alarm Test"
      alarm1 = ESMF_AlarmCreate(name="WAKEUP", clock=clock1, ringTime=alarmTime, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------
      !NEX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Destroy Alarm Test"
      call ESMF_AlarmDestroy(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)


  !------------------------------------------------------------------------
  !NEX_UTest
  write(name, *) "Testing Alarm IsCreated for uncreated object"
  write(failMsg, *) "Did not return .false."
  isCreated = ESMF_AlarmIsCreated(alarm1)
  call ESMF_Test((isCreated .eqv. .false.), name, failMsg, result, ESMF_SRCLINE)
  !------------------------------------------------------------------------

  !------------------------------------------------------------------------
  !NEX_UTest
  write(name, *) "Testing Alarm IsCreated for uncreated object"
  write(failMsg, *) "Did not return ESMF_SUCCESS"
  isCreated = ESMF_AlarmIsCreated(alarm1, rc=rc)
  call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
  !------------------------------------------------------------------------

  !------------------------------------------------------------------------
  !NEX_UTest
  write(name, *) "Create test Alarm for IsCreated"
  write(failMsg, *) "Did not return ESMF_SUCCESS"
  alarm1 = ESMF_AlarmCreate(name="WAKEUP", clock=clock1, ringTime=alarmTime, rc=rc)
  call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
  !------------------------------------------------------------------------

  !------------------------------------------------------------------------
  !NEX_UTest
  write(name, *) "Testing Alarm IsCreated for created object"
  write(failMsg, *) "Did not return .true."
  isCreated = ESMF_AlarmIsCreated(alarm1)
  call ESMF_Test((isCreated .eqv. .true.), name, failMsg, result, ESMF_SRCLINE)
  !------------------------------------------------------------------------

  !------------------------------------------------------------------------
  !NEX_UTest
  write(name, *) "Testing Alarm IsCreated for created object"
  write(failMsg, *) "Did not return ESMF_SUCCESS"
  isCreated = ESMF_AlarmIsCreated(alarm1, rc=rc)
  call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
  !------------------------------------------------------------------------

  !------------------------------------------------------------------------
  !NEX_UTest
  write(name, *) "Destroy test Alarm for IsCreated"
  write(failMsg, *) "Did not return ESMF_SUCCESS"
  call ESMF_AlarmDestroy(alarm1, rc=rc)
  call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
  !------------------------------------------------------------------------

  !------------------------------------------------------------------------
  !NEX_UTest
  write(name, *) "Testing Alarm IsCreated for destroyed object"
  write(failMsg, *) "Did not return .false."
  isCreated = ESMF_AlarmIsCreated(alarm1)
  call ESMF_Test((isCreated .eqv. .false.), name, failMsg, result, ESMF_SRCLINE)
  !------------------------------------------------------------------------

  !------------------------------------------------------------------------
  !NEX_UTest
  write(name, *) "Testing Alarm IsCreated for destroyed object"
  write(failMsg, *) "Did not return ESMF_SUCCESS"
  isCreated = ESMF_AlarmIsCreated(alarm1, rc=rc)
  call ESMF_Test((rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
  !------------------------------------------------------------------------




#ifdef ESMF_TESTEXHAUSTIVE
      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS and alarmCount = 0"
      write(name, *) "Get number of alarms after destroyed Alarm Test"
      call ESMF_ClockGet(clock1, alarmCount=alarmCount, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS.and.alarmCount.eq.0), &
                      name, failMsg, result, ESMF_SRCLINE)
      !print *, "alarmCount = ", alarmCount

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_FAILURE"
      write(name, *) "Get a destroyed Alarm Test"
      call ESMF_ClockGetAlarm(clock1, "WAKEUP", alarm=alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_FAILURE), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Advance clock after destroyed alarm test"
      call ESMF_ClockAdvance(clock1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Destroy a destroyed Alarm Test"
      call ESMF_AlarmDestroy(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Destroy a non-created Alarm Test"
      call ESMF_AlarmDestroy(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Check if destroyed Alarm is sticky Test"
      sticky =  ESMF_AlarmIsSticky(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Check if non-created Alarm is sticky Test"
      sticky =  ESMF_AlarmIsSticky(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Validate destroyed Alarm Test"
      call ESMF_AlarmValidate(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Validate non-created Alarm Test"
      call ESMF_AlarmValidate(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Check if destroyed Alarm is enabled Test"
      enabled =  ESMF_AlarmIsEnabled(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Check if non-created Alarm is enabled Test"
      enabled =  ESMF_AlarmIsEnabled(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------
      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Check if destroyed Alarm is not ringing Test"
      isringing = ESMF_AlarmIsRinging(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------
      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Check if non-created Alarm is not ringing Test"
      isringing = ESMF_AlarmIsRinging(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Get name of destroyed Alarm Test"
      call  ESMF_AlarmGet(alarm1, name=aName, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Get name of destroyed Alarm Test"
      call  ESMF_AlarmGet(alarm6, name=aName, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Set name of destroyed Alarm Test"
      call  ESMF_AlarmSet(alarm1, name="ALARM1", rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Set name of non-created Alarm Test"
      call  ESMF_AlarmSet(alarm6, name="ALARM1", rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Turn on Ringing on destroyed Alarm "
      call ESMF_AlarmRingerOn(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Turn on Ringing on non-created Alarm "
      call ESMF_AlarmRingerOn(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Turn off Ringing on destroyed Alarm "
      call ESMF_AlarmRingerOff(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Turn off Ringing on non-created Alarm "
      call ESMF_AlarmRingerOff(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------
      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Enable destroyed Alarm Test"
      call ESMF_AlarmEnable(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Enable non-create Alarm Test"
      call ESMF_AlarmEnable(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Check if destroyed Alarm is enabled Test"
      enabled =  ESMF_AlarmIsEnabled(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Check if non-created Alarm is enabled Test"
      enabled =  ESMF_AlarmIsEnabled(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Disable destroyed Alarm Test"
      call ESMF_AlarmDisable(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Disable non-created Alarm Test"
      call ESMF_AlarmDisable(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test if destroyed Alarm Previously ringing
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Destroyed Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test if non-created Alarm Previously ringing
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Non-created Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Turn off Ringing on destroyed Alarm "
      call ESMF_AlarmRingerOff(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Turn off Ringing on non-created Alarm "
      call ESMF_AlarmRingerOff(alarm6, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test destroyed Alarm will ring next
      write(failMsg, *) " Did not return ESMF_RC_OBJ_DELETED"
      write(name, *) "Destroyed Alarm will ring next Test"
      willRingNext = ESMF_AlarmWillRingNext(alarm1, timeStep=timeStep, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_DELETED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test non-created Alarm will ring next
      write(failMsg, *) " Did not return ESMF_RC_OBJ_NOT_CREATED"
      write(name, *) "Non-created Alarm will ring next Test"
      willRingNext = ESMF_AlarmWillRingNext(alarm6, timeStep=timeStep, rc=rc)
      call ESMF_Test((rc.eq.ESMF_RC_OBJ_NOT_CREATED), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Create Alarm Test"
      alarm1 = ESMF_AlarmCreate(name="WAKEUP1", clock=clock1, ringTime=alarmTime, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Create Alarm Test"
      alarm3 = ESMF_AlarmCreate(name="WAKEUP3", clock=clock1, ringTime=alarmTime, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Create Alarm Copy Test"
      alarm2 = ESMF_AlarmCreate(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Clock Get Alarm List Test 1 - optional args missing"
      call ESMF_ClockGetAlarmList(clock1, ESMF_ALARMLIST_ALL, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Clock Get Alarm List Test 2 - only alarmList specified"
      call ESMF_ClockGetAlarmList(clock1, ESMF_ALARMLIST_ALL, &
                                  alarmList=alarmList, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS and alarmCount=3"
      write(name, *) "Clock Get Alarm List Test 3 - only alarmCount specified"
      call ESMF_ClockGetAlarmList(clock1, ESMF_ALARMLIST_ALL, &
                                  alarmCount=alarmCount, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS.and.alarmCount.eq.3), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS and alarmCount=1"
      write(name, *) "Clock Get Alarm List Test 4 - single-element length alarmList specified"
      call ESMF_AlarmRingerOn(alarm1, rc=rc)
      call ESMF_ClockGetAlarmList(clock1, ESMF_ALARMLIST_RINGING, &
                                  alarmList=alarmListOne, &
                                  alarmCount=alarmCount, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS.and.alarmCount.eq.1), &
                      name, failMsg, result, ESMF_SRCLINE)
      call ESMF_AlarmRingerOff(alarm1, rc=rc)
      !call ESMF_AlarmPrint(alarmListOne(1), options="name", rc=rc)
      !print *, "alarmCount = ", alarmCount

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS and alarmCount = 3"
      write(name, *) "Clock Get Alarm List Test 5"
      call ESMF_ClockGetAlarmList(clock1, ESMF_ALARMLIST_ALL, &
                                  alarmList=alarmList, alarmCount=alarmCount, &
                                  rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS.and.alarmCount.eq.3), &
                      name, failMsg, result, ESMF_SRCLINE)
      !call ESMF_AlarmPrint(alarmList(1), options="name", rc=rc)
      !call ESMF_AlarmPrint(alarmList(2), options="name", rc=rc)
      !call ESMF_AlarmPrint(alarmList(3), options="name", rc=rc)
      !print *, "alarmCount = ", alarmCount

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmAssignment(=)(alarm,alarm)
      write(failMsg, *) "Returned not equal"
      write(name, *) "Alarm Assignment Test"
      alarm8 = alarm1  ! exercise default F90 Alarm = assignment
      call ESMF_AlarmGet(alarm8, name=aName, clock=clock3, &
                         ringTime=time1, rc=rc)
      call ESMF_Test((alarm8==alarm1 .and. aName=="WAKEUP1" .and. &
                      clock3==clock1 .and. time1==alarmTime), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(==)(alarm1,alarm2)
      write(failMsg, *) "Returned not equal"
      write(name, *) "Alarms Equal Test"
      alarmsEqual = (alarm1 == alarm2)  ! exercise Alarm == operator
      call ESMF_Test((alarmsEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(==)(alarm1,alarm2)
      write(failMsg, *) "Returned equal"
      write(name, *) "Alarms Not Equal Test"
      alarmsEqual = (alarm1 == alarm3)  ! exercise Alarm == operator
      call ESMF_Test((.not.alarmsEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest 
      ! Testing ESMF_AlarmOperator(/=)(alarm1,alarm2)
      write(failMsg, *) "Returned not equal"
      write(name, *) "Alarms Equal Test"
      alarmsNotEqual = (alarm1 /= alarm2)  ! exercise Alarm /= operator
      call ESMF_Test((.not.alarmsNotEqual), &
                      name, failMsg, result, ESMF_SRCLINE)
      call ESMF_AlarmDestroy(alarm2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(/=)(alarm1,alarm2)
      write(failMsg, *) "Returned equal"
      write(name, *) "Alarms Not Equal Test"
      alarmsNotEqual = (alarm1 /= alarm3)  ! exercise Alarm /= operator
      call ESMF_Test((alarmsNotEqual), &
                      name, failMsg, result, ESMF_SRCLINE)
      call ESMF_AlarmDestroy(alarm3, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(==)(alarm,alarm1)
      write(failMsg, *) "Returned equal"
      write(name, *) "Deleted Alarm Not Equal Created Alarm Test 1"
      alarmsEqual = (alarm2 == alarm1)  ! exercise Alarm == operator
      call ESMF_Test((.not.alarmsEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(==)(alarm,alarm1)
      write(failMsg, *) "Returned equal"
      write(name, *) "Deleted Alarm Not Equal Uncreated Alarm Test 1"
      alarmsEqual = (alarm3 == alarm7)  ! exercise Alarm == operator
      call ESMF_Test((.not.alarmsEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(==)(alarm,alarm1)
      write(failMsg, *) "Returned equal"
      write(name, *) "Uncreated Alarm Not Equal Created Alarm Test 1"
      alarmsEqual = (alarm7 == alarm1)  ! exercise Alarm == operator
      call ESMF_Test((.not.alarmsEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(==)(alarm,alarm1)
      write(failMsg, *) "Returned not equal"
      write(name, *) "Deleted Alarm Equal Deleted Alarm Test 1"
      alarmsEqual = (alarm2 == alarm3)  ! exercise Alarm == operator
      call ESMF_Test((alarmsEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(==)(alarm,alarm1)
      write(failMsg, *) "Returned not equal"
      write(name, *) "Uncreated Alarm Equal Uncreated Alarm Test 1"
      alarmsEqual = (alarm7 == alarm7)  ! exercise Alarm == operator
      call ESMF_Test((alarmsEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(/=)(alarm,alarm1)
      write(failMsg, *) "Returned equal"
      write(name, *) "Deleted Alarm Not Equal Created Alarm Test 2"
      alarmsNotEqual = (alarm2 /= alarm1)  ! exercise Alarm /= operator
      call ESMF_Test((alarmsNotEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(/=)(alarm,alarm1)
      write(failMsg, *) "Returned equal"
      write(name, *) "Deleted Alarm Not Equal Uncreated Alarm Test 2"
      alarmsNotEqual = (alarm3 /= alarm7)  ! exercise Alarm /= operator
      call ESMF_Test((alarmsNotEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(/=)(alarm,alarm1)
      write(failMsg, *) "Returned equal"
      write(name, *) "Uncreated Alarm Not Equal Created Alarm Test 2"
      alarmsNotEqual = (alarm7 /= alarm1)  ! exercise Alarm /= operator
      call ESMF_Test((alarmsNotEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(/=)(alarm,alarm1)
      write(failMsg, *) "Returned not equal"
      write(name, *) "Deleted Alarm Equal Deleted Alarm Test 2"
      alarmsNotEqual = (alarm2 /= alarm3)  ! exercise Alarm /= operator
      call ESMF_Test((.not.alarmsNotEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Testing ESMF_AlarmOperator(/=)(alarm,alarm1)
      write(failMsg, *) "Returned equal"
      write(name, *) "Uncreated Alarm Equal Uncreated Alarm Test 2"
      alarmsNotEqual = (alarm7 /= alarm7)  ! exercise Alarm /= operator
      call ESMF_Test((.not.alarmsNotEqual), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Validate Alarm Test"
      call ESMF_AlarmValidate(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned not sticky"
      write(name, *) "Check if Alarm is sticky Test"
      sticky =  ESMF_AlarmIsSticky(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(sticky), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned not enabled"
      write(name, *) "Check if Alarm is enabled Test"
      enabled =  ESMF_AlarmIsEnabled(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(enabled), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------
      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Set Alarm not sticky Test"
      call  ESMF_AlarmNotSticky(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or name is not correct"
      write(name, *) "Get Alarm name Test"
      call  ESMF_AlarmGet(alarm1, name=aName, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(aName.eq."WAKEUP1"), &
                      name, failMsg, result, ESMF_SRCLINE)
      !print *, "Alarm name is ", aName

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or name is not correct"
      write(name, *) "Set Alarm name Test"
      call  ESMF_AlarmSet(alarm1, name="ALARM1", rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or name is not correct"
      write(name, *) "Get Alarm name Test"
      call  ESMF_AlarmGet(alarm1, name=aName, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(aName.eq."ALARM1"), &
                      name, failMsg, result, ESMF_SRCLINE)
      !print *, "Alarm name is ", aName

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned sticky"
      write(name, *) "Check if Alarm is not sticky Test"
      sticky =  ESMF_AlarmIsSticky(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.sticky), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Set Alarm sticky Test"
      call  ESMF_AlarmSticky(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned not sticky"
      write(name, *) "Check if Alarm is sticky Test"
      sticky =  ESMF_AlarmIsSticky(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(sticky), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or is ringing"
      write(name, *) "Check if Alarm is not ringing Test"
      isringing = ESMF_AlarmIsRinging(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.isringing), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Turn on Ringing Alarm "
      call ESMF_AlarmRingerOn(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or its not ringing"
      write(name, *) "Check if Alarm is ringing Test"
      isringing = ESMF_AlarmIsRinging(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(isringing), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Turn off Ringing Alarm "
      call ESMF_AlarmRingerOff(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or its ringing"
      write(name, *) "Check if Alarm is not ringing Test"
      isringing = ESMF_AlarmIsRinging(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.isringing), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Enable Alarm Test"
      call ESMF_AlarmEnable(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned enabled"
      write(name, *) "Check if Alarm is enabled Test"
      enabled =  ESMF_AlarmIsEnabled(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(enabled), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Alarm Print Test"
      call  ESMF_AlarmPrint(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Disable Alarm Test"
      call ESMF_AlarmDisable(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned enabled"
      write(name, *) "Check if Alarm is disabled Test"
      enabled =  ESMF_AlarmIsEnabled(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.enabled), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Destroy Alarm Test"
      call ESMF_AlarmDestroy(alarm1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Destroy Clock Test"
      call ESMF_ClockDestroy(clock1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Test Setting Time Step
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Set Time Interval Initialization Test"
      call ESMF_TimeIntervalSet(timeStep, h=1, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Test Setting the Start Time
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Set Start Time Initialization Test"
      call ESMF_TimeSet(startTime, yy=2003, mm=3, dd=13, &
                                   h=1, m=45, s=27, &
                                   calkindflag=ESMF_CALKIND_GREGORIAN, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Test Setting the Stop Time
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Set Stop Time Initialization Test"
      call ESMF_TimeSet(stopTime, yy=2003, mm=3, dd=13, &
                                   h=18, m=45, s=27, &
                                   calkindflag=ESMF_CALKIND_GREGORIAN, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Test Setting the Alarm Time
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Set Alarm Time Initialization Test"
      call ESMF_TimeSet(alarmTime, yy=2003, mm=3, dd=13, h=5, &
                                   calkindflag=ESMF_CALKIND_GREGORIAN, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      ! Initialize clock 
      !EX_UTest
       write(name, *) "Clock Initialization Test"
       write(failMsg, *) " Did not return ESMF_SUCCESS"
       clock = ESMF_ClockCreate(timeStep, startTime, &
                                stopTime=stopTime, name="Clock 1", rc=rc)
       call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                       name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Sticky Alarm Time Initialization Test"
      alarm =  ESMF_AlarmCreate(name="alarm1", clock=clock, ringTime=alarmTime, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Get Alarm Test"
      call ESMF_ClockGetAlarm(clock, alarmname="alarm1", alarm=alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm will ring next
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm will ring next Test"
      willRingNext = ESMF_AlarmWillRingNext(alarm, timeStep=timeStep, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.willRingNext), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Get Clock Next Time
      write(failMsg, *) " Did not return ESMF_SUCCESS)"
      write(name, *) "Get Clock Next Time Test"
      call ESMF_ClockGetNextTime(clock, nextTime, timeStep=timeStep, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm will ring next
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm will ring next Test"
      willRingNext = ESMF_AlarmWillRingNext(alarm, timeStep=timeStep, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(willRingNext), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Get Time from clock
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Get Clock Current Time Test"
      call ESMF_ClockGet(clock, currTime=currentTime, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test that Clock Get Next Time Passed
      !Testing ESMF_TimeOperator(==)(time1,time2)
      write(failMsg, *) " Next Time not equal to current Time"
      write(name, *) "Get Clock Next Time Test"
      bool = (nextTime == currentTime)  ! exercising Time == operator
      call ESMF_Test((bool), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm is ringing Test"
      bool =  ESMF_AlarmIsRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm still ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm is still ringing Test"
      bool =  ESMF_AlarmIsRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm Previously ringing
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)

      !print *, "bool is ", bool

   
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm Previously ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm still ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm is still ringing Test"
      bool =  ESMF_AlarmIsRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Turn off Ringing Alarm "
      call ESMF_AlarmRingerOff(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS or its ringing"
      write(name, *) "Check if Alarm is not ringing Test"
      isringing = ESMF_AlarmIsRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.isringing), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm Previously ringing
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm Previously ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm Previously ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.bool), &
                      name, failMsg, result, ESMF_SRCLINE)
      call ESMF_AlarmDestroy (alarm, rc=rc)
      call ESMF_ClockDestroy (clock, rc=rc)

      ! ----------------------------------------------------------------------------

      ! Initialize clock 
      !EX_UTest
       write(name, *) "Clock Initialization Test"
       write(failMsg, *) " Did not return ESMF_SUCCESS"
       clock = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="Clock 1", rc=rc)
       call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                       name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Non-Sticky Alarm Time Initialization Test"
      alarm =  ESMF_AlarmCreate(name="alarm1", clock=clock, &
          ringTime=alarmTime, sticky=.FALSE., rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------
      !EX_UTest
      !Test Alarm will ring next
      call ESMF_ClockAdvance(clock, rc=rc)
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm will ring next Test"
      willRingNext = ESMF_AlarmWillRingNext(alarm, timeStep=timeStep, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.willRingNext), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm will ring next
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm will ring next Test"
      willRingNext = ESMF_AlarmWillRingNext(alarm, timeStep=timeStep, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(willRingNext), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm is ringing Test"
      bool =  ESMF_AlarmIsRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm still ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm is still ringing Test"
      bool =  ESMF_AlarmIsRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.bool), &
                      name, failMsg, result, ESMF_SRCLINE)
      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm Previously ringing
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(bool), &
                      name, failMsg, result, ESMF_SRCLINE)

      !print *, "bool is ", bool

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm Previously ringing
      call ESMF_ClockAdvance(clock, rc=rc)
      write(failMsg, *) " Did not return ESMF_SUCCESS or returned wrong state"
      write(name, *) "Alarm Was Previously ringing Test"
      bool =  ESMF_AlarmWasPrevRinging(alarm, rc=rc)
      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(.not.bool), &
                      name, failMsg, result, ESMF_SRCLINE)


      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test destroying alarm
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Destroying alarm test"
      call ESMF_AlarmDestroy (alarm, rc=rc)
      call ESMF_Test(rc == ESMF_SUCCESS, name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test destroying clock after alarm
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Destroying clock after alarm test"
      call ESMF_ClockDestroy (clock, rc=rc)
      call ESMF_Test(rc == ESMF_SUCCESS, name, failMsg, result, ESMF_SRCLINE)
      

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky ringTime Alarm with repeating clock test."

      ! Init correct
      correct=.true.      

      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=rc)

      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=rc)
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=rc)
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=rc)

      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=11, s=30, &
                                calendar=gregorianCalendar, rc=rc)
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           sticky=.false., rc=rc)

      ! Advance clock to 5:14
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.


      ! Advance clock to 6:10 (e.g. 5:10) checking that it's not ringing
      do i=1,4
         
         ! Advance
         call ESMF_ClockAdvance(repeatClock, rc=rc)

         ! Make sure not ringing
         if (ESMF_AlarmIsRinging(alarm)) correct=.false.

         ! DEBUG OUTPUT
         ! write(*,*) i,"Alarm is ringing=",ESMF_AlarmIsRinging(alarm)

      enddo

      ! Advance clock to 5:24
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing (we passed the ringTime again
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.
      
      ! DEBUG OUTPUT
      !write(*,*) "Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)
            
      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=rc)
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=rc)
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)

      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky ringTime ESMF_AlarmWillRingNext() with repeating clock test."

      ! Init correct
      correct=.true.      

      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=rc)

      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=rc)
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=rc)
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=rc)

      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=11, s=30, &
                                calendar=gregorianCalendar, rc=rc)
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           sticky=.false., rc=rc)

      ! Make sure is ringing next
      if (.not. ESMF_AlarmWillRingNext(alarm)) correct=.false.

      ! Advance clock to 5:14
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure NOT ringing next
      if (ESMF_AlarmWillRingNext(alarm)) correct=.false.

      ! Advance clock to 5:56 checking that it's not ringing
      do i=1,3
         
         ! Advance
         call ESMF_ClockAdvance(repeatClock, rc=rc)

         ! Make sure not ringing next
         if (ESMF_AlarmWillRingNext(alarm)) correct=.false.

         ! DEBUG OUTPUT
         !write(*,*) i,"Alarm is ringing next=",ESMF_AlarmIsRinging(alarm)

      enddo

      ! Advance clock to 5:10
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure is ringing next because that will pass the ringTime again
      if (.not. ESMF_AlarmWillRingNext(alarm)) correct=.false.
      
      ! DEBUG OUTPUT
      !write(*,*) "Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)
            
      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=rc)
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=rc)
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky ringTime with non-default ringTimeStepCount Alarm with repeating clock test."

      ! Init correct
      correct=.true.      

      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=rc)

      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=rc)
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=rc)
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=rc)

      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=11, s=30, &
                                calendar=gregorianCalendar, rc=rc)
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           ringTimeStepCount=3, sticky=.false., rc=rc)

      ! Advance clock to 5:14 (step 1)
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.


      ! Advance clock to 5:28 (step 2)
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! Advance clock to 5:42 (step 3)
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.


      ! Advance clock to 6:10 (e.g. 5:10) checking that it's not ringing
      do i=1,2
         
         ! Advance
         call ESMF_ClockAdvance(repeatClock, rc=rc)

         ! Make sure not ringing
         if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      enddo

      ! Advance clock to 5:24 (step 1)
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing (we passed the ringTime again
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.


      ! Advance clock to 5:38 (step 2)
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing (we passed the ringTime again
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! Advance clock to 5:52 (step 3)
      call ESMF_ClockAdvance(repeatClock, rc=rc)
      
      ! Make sure ringing (we passed the ringTime again
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! Advance clock to 6:06 (i.e. 5:06)
      call ESMF_ClockAdvance(repeatClock, rc=rc)
      
      ! Make sure not ringing (we're outside the number of steps again)
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.
            
      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=rc)
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=rc)
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)
      

      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky ringTime Alarm with ringDuration with repeating clock test."

      ! Init correct
      correct=.true.      

      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=rc)

      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=rc)
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=rc)
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=rc)

      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=11, s=0, &
                                calendar=gregorianCalendar, rc=rc)

      ! Set alarm Duration
      call ESMF_TimeIntervalSet(ringDuration, m=30, rc=rc)
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           ringDuration=ringDuration, sticky=.false., rc=rc)

      
      ! Advance clock to 5:14
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:14 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)
      
      ! Advance clock to 5:28
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:28 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)

      ! Advance clock to 5:42
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:42 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)

      ! Advance clock to 5:56
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure has stopped ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:56 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)
            
      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=rc)
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=rc)
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)


      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky ringTime Alarm with wrapping ringDuration with repeating clock test."

      ! Init correct
      correct=.true.      

      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=rc)

      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=rc)
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=rc)
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=rc)

      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=41, s=0, &
                                calendar=gregorianCalendar, rc=rc)

      ! Set alarm Duration
      call ESMF_TimeIntervalSet(ringDuration, m=30, rc=rc)
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           ringDuration=ringDuration, sticky=.false., rc=rc)

      
      ! Advance clock to 5:14
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure not ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:14 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)
      
      ! Advance clock to 5:28
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure not ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:28 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)

      ! Advance clock to 5:42
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:42 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)

      ! Advance clock to 5:56
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure alarm is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:56 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)

      ! Advance clock to 5:10
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure alarm is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:10 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)

      ! Advance clock to 5:24
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Make sure alarm is not ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "6:24 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)
           
      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=rc)
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=rc)
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)



      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky Alarm with ringInterval with repeating clock test."

      ! Init correct
      correct=.true.      
      rc=ESMF_SUCCESS
      
      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=11, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set alarm Interval
      call ESMF_TimeIntervalSet(ringInterval, m=20, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           ringInterval=ringInterval, sticky=.false., rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Advance clock to 5:14
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Make sure ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:14 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)
      
      ! Advance clock to 5:28
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure not ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:28 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:42
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:42 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)
      
      ! Advance clock to 5:56
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:56 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:10
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is not ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:10 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:24
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:24 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)



      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky Alarm with ringInterval and stopTime with repeating clock test."

      ! Init correct
      correct=.true.      
      rc=ESMF_SUCCESS
      
      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=11, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Set stopTime
      call ESMF_TimeSet(stopTime, yy=2024, mm=3, dd=14, h=5, m=50, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set alarm Interval
      call ESMF_TimeIntervalSet(ringInterval, m=20, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           ringInterval=ringInterval, stopTime=stopTime, sticky=.false., rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Advance clock to 5:14
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Make sure ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:14 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)
      
      ! Advance clock to 5:28
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure not ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:28 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:42
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:42 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)
      
      ! Advance clock to 5:56 (PAST stopTime)
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is NOT ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:56 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:10 (PAST stopTime)
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is not ringing
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:10 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:24 (passing ringTime again...) 
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:24 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)


      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky Alarm with ringInterval and wrapped stopTime with repeating clock test."

      ! Init correct
      correct=.true.      
      rc=ESMF_SUCCESS
      
      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=31, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Set stopTime
      call ESMF_TimeSet(stopTime, yy=2024, mm=3, dd=14, h=5, m=10, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set alarm Interval
      call ESMF_TimeIntervalSet(ringInterval, m=20, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           ringInterval=ringInterval, stopTime=stopTime, sticky=.false., rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Advance clock to 5:14
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Make sure NOT ringing (it's not in 5:31 - 5:10)
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:14 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)
      
      ! Advance clock to 5:28
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure NOT ringing (it's not in 5:31 - 5:10)
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:28 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:42
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:42 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)
      
      ! Advance clock to 5:56
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:56 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:10
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure is not ringing 
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:10 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:24
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure NOT ringing (it's not in 5:31 - 5:10)
      if (ESMF_AlarmIsRinging(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:24 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)


      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Not sticky Alarm with ringInterval and ESMF_AlarmWillRingNext() with repeating clock test."

      ! Init correct
      correct=.true.      
      rc=ESMF_SUCCESS
      
      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=14, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=11, s=0, &
                                calendar=gregorianCalendar, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Set alarm Interval
      call ESMF_TimeIntervalSet(ringInterval, m=20, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Create ringTime Alarm 
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           ringInterval=ringInterval, sticky=.false., rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure ringing
      if (.not. ESMF_AlarmWillRingNext(alarm)) correct=.false.
      
      ! Advance clock to 5:14
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Make sure NOT ringing
      if (ESMF_AlarmWillRingNext(alarm)) correct=.false.
      
      ! DEBUG OUTPUT
      !write(*,*) "5:14 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)
      
      ! Advance clock to 5:28
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure ringing
      if (.not. ESMF_AlarmWillRingNext(alarm)) correct=.false.
      
      ! DEBUG OUTPUT
      !write(*,*) "5:28 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:42
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! DEBUG OUTPUT
      !write(*,*) "5:42 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Make sure ringing
      if (.not. ESMF_AlarmWillRingNext(alarm)) correct=.false.
      
      ! Advance clock to 5:56
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure NOT ringing
      if (ESMF_AlarmWillRingNext(alarm)) correct=.false.

      ! DEBUG OUTPUT
      !write(*,*) "5:56 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Advance clock to 5:10
      call ESMF_ClockAdvance(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE

      ! Make sure ringing
      if (.not. ESMF_AlarmWillRingNext(alarm)) correct=.false.
      
      ! DEBUG OUTPUT
      !write(*,*) "5:10 Is ringing=",ESMF_AlarmIsRinging(alarm,rc=localrc)

      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=localrc)
      if (localrc .ne. ESMF_SUCCESS) rc=ESMF_FAILURE
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)

      
      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Returned ESMF_FAILURE"
      write(name, *) "Sticky ringTime Alarm with repeating clock test."

      ! Simple test of repeating within one hour
      
      ! Set start time 
      call ESMF_TimeSet(startTime, yy=2024, mm=3, dd=14, h=5, m=0, s=0, &
                                calendar=gregorianCalendar, rc=rc)

      ! Set timeStep to one minute
      call ESMF_TimeIntervalSet(timeStep, m=1, rc=rc)
      
      ! Set repeat duration to one hour
      call ESMF_TimeIntervalSet(repeatDuration, h=1, rc=rc)
      
      ! Create Clock
      repeatClock = ESMF_ClockCreate(timeStep, startTime, &
           repeatDuration=repeatDuration, rc=rc)

      ! Set ringTime
      call ESMF_TimeSet(ringTime, yy=2024, mm=3, dd=14, h=5, m=0, s=30, &
                                calendar=gregorianCalendar, rc=rc)
      
      ! Create Alarm
      alarm = ESMF_AlarmCreate(clock=repeatClock, ringTime=ringTime, &
           sticky=.true., rc=rc)

      ! Advance clock (to alarm time)
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Advance clock (to time after alarmTime to check stickiness)
      call ESMF_ClockAdvance(repeatClock, rc=rc)

      ! Init correct
      correct=.true.
      
      ! Check if ringing
      if (.not. ESMF_AlarmIsRinging(alarm)) correct=.false.
      
      ! DEBUG OUTPUT
      !write(*,*) "Is ringing=",ESMF_AlarmIsRinging(alarm,rc=rc)
      
      
      ! Free Alarm
      call ESMF_AlarmDestroy(alarm, rc=rc)
      
      ! Free Clock
      call ESMF_ClockDestroy(repeatClock, rc=rc)
      
      call ESMF_Test(((rc.eq.ESMF_SUCCESS) .and. correct), &
                      name, failMsg, result, ESMF_SRCLINE)

      
      ! ----------------------------------------------------------------------------



      
      !EX_UTest
      !Test Non-Sticky Alarms 1
      !  from Chris Hill via support issue 988241, bug 996229
      write(failMsg, *) " Did not return nstep=48, sstep=73, i=144, and ESMF_SUCCESS"
      write(name, *) "Non-Sticky Alarm Test 1"
      call ESMF_TimeIntervalSet(timeStep, s=100, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, s=150, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, h=3, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                              name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, sticky=.false., rc=rc)
      ! number of clock time steps alarm rings for
      nstep = 0

      ! starting time step number for first alarm ring
      sstep = 0

      ! total clock time steps
      i = 0

      ! run the clock
      do while (.not. ESMF_ClockIsStopTime(clock2, rc=rc))
        i = i + 1
        call ESMF_ClockGetAlarmList(clock2, ESMF_ALARMLIST_RINGING, &
                                    alarmList=alarmList, &
                                    alarmCount=alarmCount, rc=rc)
        if (alarmCount .gt. 0) then
          if (sstep .eq. 0) sstep = i
          nstep = nstep + 1
        endif
        call ESMF_ClockAdvance(clock2, rc=rc)
      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS) &
                      .and.(nstep.eq.48).and.(sstep.eq.73).and.(i.eq.144), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------
      !EX_UTest
      !Test Non-Sticky Alarms 2
      !  Test ringDuration and stopTime
      write(failMsg, *) " Did not return nstep=12 and ESMF_SUCCESS"
      write(name, *) "Non-Sticky Alarm Test 2"
      call ESMF_TimeIntervalSet(timeStep, m=15, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      clock2 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, h=2, rc=rc)
      call ESMF_TimeIntervalSet(ringDuration, m=30, rc=rc)
      call ESMF_TimeSet(alarmStopTime, yy=2000, mm=1, dd=1, h=13, &
                        calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, &
                                ringDuration=ringDuration, &
                                stopTime=alarmStopTime, sticky=.false., rc=rc)
      ! number of clock time steps alarm rings for
      nstep = 0

      ! run the clock
      do while (.not. ESMF_ClockIsStopTime(clock2, rc=rc))
        if (ESMF_AlarmIsRinging(alarm4)) then
          nstep = nstep + 1
        endif
        call ESMF_ClockAdvance(clock2, rc=rc)
      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nstep.eq.12), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Non-Sticky Alarms 3
      !  Test ringTimeStepCount and stopTime
      write(failMsg, *) " Did not return nstep=18 and ESMF_SUCCESS"
      write(name, *) "Non-Sticky Alarm Test 3"
      call ESMF_TimeIntervalSet(timeStep, m=15, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                              name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, h=2, rc=rc)
      call ESMF_TimeSet(alarmStopTime, yy=2000, mm=1, dd=1, h=13, &
                        calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, &
                                ringTimeStepCount=3, &
                                stopTime=alarmStopTime, sticky=.false., rc=rc)
      ! number of clock time steps alarm rings for
      nstep = 0

      ! run the clock
      do while (.not. ESMF_ClockIsStopTime(clock2, rc=rc))
        if (ESMF_AlarmIsRinging(alarm4)) then
          nstep = nstep + 1
        endif
        call ESMF_ClockAdvance(clock2, rc=rc)
      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nstep.eq.18), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Non-Sticky Alarms 4
      !  Test ringTimeStepCount precedence over ringDuration
      write(failMsg, *) " Did not return nstep=18 and ESMF_SUCCESS"
      write(name, *) "Non-Sticky Alarm Test 4"
      call ESMF_TimeIntervalSet(timeStep, m=15, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                              name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, h=2, rc=rc)

      call ESMF_TimeIntervalSet(ringDuration, m=30, rc=rc)

      call ESMF_TimeSet(alarmStopTime, yy=2000, mm=1, dd=1, h=13, &
                        calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, &
                                ringTimeStepCount=3, &
                                ringDuration=ringDuration, &
                                stopTime=alarmStopTime, sticky=.false., rc=rc)
      ! number of clock time steps alarm rings for
      nstep = 0

      ! run the clock
      do while (.not. ESMF_ClockIsStopTime(clock2, rc=rc))
        if (ESMF_AlarmIsRinging(alarm4)) then
          nstep = nstep + 1
        endif
        call ESMF_ClockAdvance(clock2, rc=rc)
      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nstep.eq.18), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Non-Sticky Alarms 5
      !  Test reverse non-sticky alarm 
      write(failMsg, *) "Did not return nstep=24, forwardCount=96, reverseCount=0, etc., and ESMF_SUCCESS"
      write(name, *) "Non-Sticky Alarm Test 5"
      call ESMF_TimeIntervalSet(timeStep, m=15, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      clock2 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, h=2, rc=rc)
      call ESMF_TimeIntervalSet(ringDuration, m=30, rc=rc)
      call ESMF_TimeSet(alarmStopTime, yy=2000, mm=1, dd=1, h=13, &
                        calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, &
                                ringDuration=ringDuration, &
                                stopTime=alarmStopTime, sticky=.false., rc=rc)
      ! number of clock time steps alarm rings for
      nstep = 0

      ! number of times the clock has been run
      nclock = 0

      do while (nclock < 2)
        ! run the clock
        do while (.not. ESMF_ClockIsDone(clock2, rc=rc))
          !call ESMF_ClockGet(clock2, currTime=currentTime, rc=rc)
          !call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, m=m, rc=rc)
          !print *, mm, "/", dd, "/", yy, " ", h, ":", m
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          if (ESMF_AlarmIsRinging(alarm4)) then
            nstep = nstep + 1
            !print *, "on"
            !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          else
            !print *, "off"
          endif
          !call ESMF_AlarmPrint(alarm4, options="ringtime string", rc=rc)
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_ClockAdvance(clock2, rc=rc)
        enddo

        if (nclock.eq.0) then
          call ESMF_ClockGet(clock2, direction=forwardDirection, &
                                     advanceCount=forwardCount, rc=rc)
          !print *, "forwardCount = ", forwardCount
        else
          call ESMF_ClockGet(clock2, direction=reverseDirection, &
                             advanceCount=reverseCount, rc=rc)
          !print *, "reverseCount = ", reverseCount
        endif

        call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_REVERSE, rc=rc)
        nclock = nclock + 1

      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nstep.eq.24).and. &
                     (forwardCount.eq.96).and.(reverseCount.eq.0).and. &
                     (forwardDirection.eq.ESMF_DIRECTION_FORWARD).and. &
                     (reverseDirection.eq.ESMF_DIRECTION_REVERSE).and. &
                     ESMF_ClockIsReverse(clock2), &
                      name, failMsg, result, ESMF_SRCLINE)

      !print *, "nstep = ", nstep, " nclock = ", nclock
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Non-Sticky Alarms 6
      !  Test reverse non-sticky alarm with negative timestep
      write(failMsg, *) "Did not return nring=48, forwardCount=96, reverseCount=0, etc., and ESMF_SUCCESS"
      write(name, *) "Non-Sticky Alarm Test 6"
      call ESMF_TimeIntervalSet(timeStep, m=-15, rc=rc)
      call ESMF_TimeSet(startTime, yy=2000, mm=1, dd=1, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=1999, mm=12, dd=31, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      clock2 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, h=-2, rc=rc)
      call ESMF_TimeIntervalSet(ringDuration, m=-30, rc=rc)
      call ESMF_TimeSet(alarmStopTime, yy=1999, mm=12, dd=31, h=21, &
                        calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, &
                                ringDuration=ringDuration, &
                                stopTime=alarmStopTime, &
                                sticky=.false., rc=rc)
      ! number of clock time steps alarm rings for
      nring = 0

      ! number of times the clock has been run
      nclock = 0

      do while (nclock < 2)
        ! run the clock
        do while (.not. ESMF_ClockIsDone(clock2, rc=rc))
          call ESMF_ClockGet(clock2, currTime=currentTime, rc=rc)
          call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, m=m, rc=rc)
          !print *, mm, "/", dd, "/", yy, " ", h, ":", m
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          if (ESMF_AlarmIsRinging(alarm4)) then
            nring = nring + 1
            !print *, "ringing!"
            !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          endif
          !call ESMF_AlarmPrint(alarm4, options="ringbegin string", rc=rc)
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          !call ESMF_AlarmPrint(alarm4, options="ringtime string", rc=rc)
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_ClockAdvance(clock2, rc=rc)
        enddo

        if (nclock.eq.0) then
          call ESMF_ClockGet(clock2, direction=forwardDirection, &
                                     advanceCount=forwardCount, rc=rc)
          !print *, "forwardCount = ", forwardCount
        else
          call ESMF_ClockGet(clock2, direction=reverseDirection, &
                             advanceCount=reverseCount, rc=rc)
          !print *, "reverseCount = ", reverseCount
        endif

        !print *, "Going in REVERSE ..."
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_REVERSE, rc=rc)
        nclock = nclock + 1

      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nring.eq.48).and. &
                     (forwardCount.eq.96).and.(reverseCount.eq.0).and. &
                     (forwardDirection.eq.ESMF_DIRECTION_FORWARD).and. &
                     (reverseDirection.eq.ESMF_DIRECTION_REVERSE).and. &
                     ESMF_ClockIsReverse(clock2), &
                      name, failMsg, result, ESMF_SRCLINE)

      !print *, "nring = ", nring, " nclock = ", nclock
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Sticky Alarms 1
      !  Test reverse sticky alarm 
      write(failMsg, *) " Did not return nstep=2 and ESMF_SUCCESS"
      write(name, *) "Sticky Alarm Test 1"
      call ESMF_TimeIntervalSet(timeStep, m=15, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      clock2 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, rc=rc)
      ! number of clock time steps alarm rings for
      nstep = 0

      ! number of times the clock has been run
      nclock = 0

      do while (nclock < 2)
        ! run the clock
        do while (.not. ESMF_ClockIsDone(clock2, rc=rc))
          call ESMF_ClockGet(clock2, currTime=currentTime, rc=rc)
          call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, m=m, rc=rc)
          !print *, mm, "/", dd, "/", yy, " ", h, ":", m
          if (ESMF_AlarmIsRinging(alarm4)) then
            nstep = nstep + 1
            !print *, "on"
            call ESMF_AlarmRingerOff(alarm4, rc=rc)
          else
            !print *, "off"
          endif
          call ESMF_ClockAdvance(clock2, rc=rc)
        enddo

        call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_REVERSE, rc=rc)
        nclock = nclock + 1
      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nstep.eq.2), &
                      name, failMsg, result, ESMF_SRCLINE)

      !print *, "nstep = ", nstep, " nclock = ", nclock

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Sticky Alarms 2
      !  Test reverse interval sticky alarm 
      write(failMsg, *) " Did not return nstep=23 and ESMF_SUCCESS"
      write(name, *) "Sticky Alarm Test 2"
      call ESMF_TimeIntervalSet(timeStep, m=15, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, h=23, &
                        calendar=gregorianCalendar, rc=rc)
      clock2 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, h=2, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, rc=rc)
      ! number of clock time steps alarm rings for
      nstep = 0

      ! number of times the clock has been run
      nclock = 0

      do while (nclock < 2)
        ! run the clock
        do while (.not. ESMF_ClockIsDone(clock2, rc=rc))
          call ESMF_ClockGet(clock2, currTime=currentTime, rc=rc)
          call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, m=m, rc=rc)
          !print *, mm, "/", dd, "/", yy, " ", h, ":", m
          if (ESMF_AlarmIsRinging(alarm4)) then
            nstep = nstep + 1
            !print *, "on"
            call ESMF_AlarmRingerOff(alarm4, rc=rc)
          else
            !print *, "off"
          endif
          call ESMF_ClockAdvance(clock2, rc=rc)
        enddo

        call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_REVERSE, rc=rc)
        nclock = nclock + 1
      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nstep.eq.23), &
                      name, failMsg, result, ESMF_SRCLINE)

      !print *, "nstep = ", nstep, " nclock = ", nclock

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Sticky Alarms 3
      !  Test reverse one-shot sticky alarms per WRF use case
      !  From Tom Henderson/WRF

      write(name, *) "Sticky Alarm Test 3"
      write(failMsg, *) " Alarms did not turn on/off at the correct time/iteration or did not return ESMF_SUCCESS"

      call ESMF_TimeIntervalSet(timeStep, h=1, rc=rc)
      call ESMF_TimeSet(startTime, yy=2005, mm=6, dd=15, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2005, mm=6, dd=15, h=5, &
                        calendar=gregorianCalendar, rc=rc)
      domainClock = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                     name="WRF Clock", rc=rc)

      call ESMF_TimeSet(beforeAlarmTime, yy=2005, mm=6, dd=15, h=2, &
                        calendar=gregorianCalendar, rc=rc)
      beforeAlarm = ESMF_AlarmCreate(clock=domainClock, &
                                     ringTime=beforeAlarmTime, rc=rc)

      call ESMF_TimeSet(afterAlarmTime, yy=2005, mm=6, dd=15, h=3, &
                        calendar=gregorianCalendar, rc=rc)
      afterAlarm  = ESMF_AlarmCreate(clock=domainClock, &
                                     ringTime=afterAlarmTime, rc=rc)

      ! any single failure will cause the whole test to fail
      testPass = .true.

      ! track loop iterations
      iteration = 0

      call ESMF_ClockGet(domainClock, currTime=currentTime, rc=rc)
      call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, rc=rc)
      !print *, "Begin"
      !print *, mm, "/", dd, "/", yy, " ", h, ": 0"

      ! run the clock forward
      do while (.not. ESMF_ClockIsDone(domainClock, rc=rc))
        iteration = iteration + 1
        !print *, "Iteration = ", iteration

        if (ESMF_AlarmIsRinging(beforeAlarm, rc=rc)) then
          if (iteration .ne. 2 .or. h .ne. 2) then
            testPass = .false.
          endif
          !print *, "  beforeAlarm on"
          call ESMF_AlarmRingerOff(beforeAlarm, rc=rc)
        else
          !print *, "  beforeAlarm off"
        endif

        ! would call WRF solver here, before clock advance
        !print *, "  Solve"

        call ESMF_ClockAdvance(domainClock, rc=rc)
        call ESMF_ClockGet(domainClock, currTime=currentTime, rc=rc)
        call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, rc=rc)
        !print *, " ClockAdvance()"
        !print *, mm, "/", dd, "/", yy, " ", h, ": 0"

        if (ESMF_AlarmIsRinging(afterAlarm, rc=rc)) then
          if (iteration .ne. 2 .or. h .ne. 3) then
            testPass = .false.
          endif
          !print *, "  afterAlarm on"
          call ESMF_AlarmRingerOff(afterAlarm, rc=rc)
        else
          !print *, "  afterAlarm off"
        endif
      enddo

      ! run the clock backwards
      call ESMF_ClockSet(domainClock, direction=ESMF_DIRECTION_REVERSE, rc=rc)
      !print *
      !print *, "domainClock set in reverse"
      !print *

      do while (.not. ESMF_ClockIsDone(domainClock, rc=rc))
        !print *, "Iteration = ", iteration

        if (ESMF_AlarmIsRinging(afterAlarm, rc=rc)) then
          if (iteration .ne. 2 .or. h .ne. 3) then
            testPass = .false.
          endif
          !print *, "  afterAlarm on"
          call ESMF_AlarmRingerOff(afterAlarm, rc=rc)
        else
          !print *, "  afterAlarm off"
        endif

        ! advance clock
        call ESMF_ClockAdvance(domainClock, rc=rc)
        call ESMF_ClockGet(domainClock, currTime=currentTime, rc=rc)
        call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, rc=rc)
        !print *, " ClockAdvance()"
        !print *, mm, "/", dd, "/", yy, " ", h, ": 0"

        ! would call WRF solver here, after clock advance
        !print *, "  Solve"

        if (ESMF_AlarmIsRinging(beforeAlarm, rc=rc)) then
          if (iteration .ne. 2 .or. h .ne. 2) then
            testPass = .false.
          endif
          !print *, "  beforeAlarm on"
          call ESMF_AlarmRingerOff(beforeAlarm, rc=rc)
        else
          !print *, "  beforeAlarm off"
        endif

        iteration = iteration - 1
      enddo

      call ESMF_Test(testPass.and.(rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(afterAlarm, rc=rc)
      call ESMF_AlarmDestroy(beforeAlarm, rc=rc)
      call ESMF_ClockDestroy(domainClock, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Sticky Alarms 4
      !  Test reverse *interval* sticky alarms; variation on WRF use case above

      write(name, *) "Sticky Alarm Test 4"
      write(failMsg, *) " Alarms did not turn on/off at the correct time/iteration or did not return ESMF_SUCCESS"

      call ESMF_TimeIntervalSet(timeStep, h=1, rc=rc)
      call ESMF_TimeSet(startTime, yy=2005, mm=6, dd=15, h=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2005, mm=6, dd=15, h=10, &
                        calendar=gregorianCalendar, rc=rc)
      domainClock = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                     name="WRF Clock", rc=rc)

      call ESMF_TimeSet(beforeAlarmTime, yy=2005, mm=6, dd=15, h=2, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, h=3, rc=rc)
      beforeAlarm = ESMF_AlarmCreate(clock=domainClock, &
                                     ringTime=beforeAlarmTime, &
                                     ringInterval=alarmStep, rc=rc)

      call ESMF_TimeSet(afterAlarmTime, yy=2005, mm=6, dd=15, h=3, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep2, h=2, rc=rc)
      afterAlarm  = ESMF_AlarmCreate(clock=domainClock, &
                                     ringTime=afterAlarmTime, &
                                     ringInterval=alarmStep2, rc=rc)

      ! any single failure will cause the whole test to fail
      testPass = .true.

      ! track loop iterations
      iteration = 0

      call ESMF_ClockGet(domainClock, currTime=currentTime, rc=rc)
      call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, rc=rc)
      !print *, "Begin"
      !print *, mm, "/", dd, "/", yy, " ", h, ": 0"

      ! run the clock forward
      do while (.not. ESMF_ClockIsDone(domainClock, rc=rc))
        iteration = iteration + 1
        !print *, "Iteration = ", iteration

        if (ESMF_AlarmIsRinging(beforeAlarm, rc=rc)) then
          if ((iteration .ne. 2 .or. h .ne. 2).and. &
              (iteration .ne. 5 .or. h .ne. 5).and. &
              (iteration .ne. 8 .or. h .ne. 8)) then
            testPass = .false.
          endif
          !print *, "  beforeAlarm on"
          call ESMF_AlarmRingerOff(beforeAlarm, rc=rc)
        else
          !print *, "  beforeAlarm off"
        endif

        ! would call WRF solver here, before clock advance
        !print *, "  Solve"

        call ESMF_ClockAdvance(domainClock, rc=rc)
        call ESMF_ClockGet(domainClock, currTime=currentTime, rc=rc)
        call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, rc=rc)
        !print *, " ClockAdvance()"
        !print *, mm, "/", dd, "/", yy, " ", h, ": 0"

        if (ESMF_AlarmIsRinging(afterAlarm, rc=rc)) then
          if ((iteration .ne. 2 .or. h .ne. 3).and. &
              (iteration .ne. 4 .or. h .ne. 5).and. &
              (iteration .ne. 6 .or. h .ne. 7).and. &
              (iteration .ne. 8 .or. h .ne. 9)) then
            testPass = .false.
          endif
          !print *, "  afterAlarm on"
          call ESMF_AlarmRingerOff(afterAlarm, rc=rc)
        else
          !print *, "  afterAlarm off"
        endif
      enddo

      ! run the clock backwards
      call ESMF_ClockSet(domainClock, direction=ESMF_DIRECTION_REVERSE, rc=rc)
      !print *
      !print *, "domainClock set in reverse"
      !print *

      do while (.not. ESMF_ClockIsDone(domainClock, rc=rc))
        !print *, "Iteration = ", iteration

        if (ESMF_AlarmIsRinging(afterAlarm, rc=rc)) then
          if ((iteration .ne. 2 .or. h .ne. 3).and. &
              (iteration .ne. 4 .or. h .ne. 5).and. &
              (iteration .ne. 6 .or. h .ne. 7).and. &
              (iteration .ne. 8 .or. h .ne. 9)) then
            testPass = .false.
          endif
          !print *, "  afterAlarm on"
          call ESMF_AlarmRingerOff(afterAlarm, rc=rc)
        else
          !print *, "  afterAlarm off"
        endif

        ! advance clock
        call ESMF_ClockAdvance(domainClock, rc=rc)
        call ESMF_ClockGet(domainClock, currTime=currentTime, rc=rc)
        call ESMF_TimeGet(currentTime, yy=yy, mm=mm, dd=dd, h=h, rc=rc)
        !print *, " ClockAdvance()"
        !print *, mm, "/", dd, "/", yy, " ", h, ": 0"

        ! would call WRF solver here, after clock advance
        !print *, "  Solve"

        if (ESMF_AlarmIsRinging(beforeAlarm, rc=rc)) then
          if ((iteration .ne. 2 .or. h .ne. 2).and. &
              (iteration .ne. 5 .or. h .ne. 5).and. &
              (iteration .ne. 8 .or. h .ne. 8)) then
            testPass = .false.
          endif
          !print *, "  beforeAlarm on"
          call ESMF_AlarmRingerOff(beforeAlarm, rc=rc)
        else
          !print *, "  beforeAlarm off"
        endif

        iteration = iteration - 1
      enddo

      call ESMF_Test(testPass.and.(rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(afterAlarm, rc=rc)
      call ESMF_AlarmDestroy(beforeAlarm, rc=rc)
      call ESMF_ClockDestroy(domainClock, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Sticky Alarms 5
      !  Test delay in checking for ringing sticky alarms,
      !  from James Geiger in #2825456, case #2 in his alarm_test.f90 reproducer
      write(name, *) "Sticky Alarm Test 5"
      write(failMsg, *) " Alarm did not ring at the correct times or did not return ESMF_SUCCESS"
      call ESMF_TimeIntervalSet(timeStep, s=3600, rc=rc)
      call ESMF_TimeSet(startTime, yy=2000, mm=1, dd=1, &
                                   calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=3, &
                                   calendar=gregorianCalendar, rc=rc)
      clock2 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="Clock", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2000, mm=1, dd=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, h=3, rc=rc)
      alarmTime = alarmTime + alarmStep
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, rc=rc)
      testPass = .false.
      iteration = 0
      nring = 0
      do while (.not.ESMF_ClockIsStopTime(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (iteration .eq. 10) then
          ! waiting period is over, clear any alarm which may have occurred
          if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
            call ESMF_AlarmRingerOff(alarm4, rc=rc)
          endif
        endif
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)

        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (iteration .gt. 10) then
          ! process any alarm occuring after initial 10 hour waiting period
          if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
            nring = nring + 1
            !print *, "alarm4 is ringing!"
            !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
            call ESMF_AlarmRingerOff(alarm4, rc=rc)
          endif
        endif
      end do

      ! test in REVERSE mode (not from James)
      call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_REVERSE, rc=rc)

      do while (.not.ESMF_ClockIsDone(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
          nring = nring + 1
          !print *, "alarm4 is ringing!"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmRingerOff(alarm4, rc=rc)
        endif
      end do

      ! test double back in FORWARD mode (not from James)
      call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_FORWARD, rc=rc)

      do while (.not.ESMF_ClockIsDone(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
          nring = nring + 1
          !print *, "alarm4 is ringing!"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmRingerOff(alarm4, rc=rc)
        endif
      end do

      !print *, "alarm4 rang ", nring, " times, clock stepped ", iteration, " times."
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
      if (nring.eq.44 .and. iteration.eq.144) testPass = .true.
      call ESMF_Test(testPass.and.(rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Sticky Alarms 6
      !  Test timeStep == ringInterval, from James Geiger in #2910203
      write(name, *) "Sticky Alarm Test 6"
      write(failMsg, *) " Alarm did not ring at the correct times or did not return ESMF_SUCCESS"
      call ESMF_TimeIntervalSet(timeStep, s=900, rc=rc)
      call ESMF_TimeSet(startTime, yy=2003, mm=9, dd=1, &
                                   calendar=gregorianCalendar, rc=rc)
      ! dd=3, instead of original 30 in Jims report, to shorten test run time
      call ESMF_TimeSet(stopTime, yy=2003, mm=9, dd=3, &
                                   calendar=gregorianCalendar, rc=rc)
      clock2 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="The Clock", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2003, mm=9, dd=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, s=900, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, rc=rc)
      testPass = .false.
      iteration = 0
      nring = 0
      do while (.not.ESMF_ClockIsStopTime(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
          nring = nring + 1
          !print *, "alarm4 is ringing!"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmRingerOff(alarm4, rc=rc)
        endif
      end do

      ! test in REVERSE mode (not from James)
      call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_REVERSE, rc=rc)

      do while (.not.ESMF_ClockIsDone(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
          nring = nring + 1
          !print *, "alarm4 is ringing!"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmRingerOff(alarm4, rc=rc)
        endif
      end do

      ! test double back in FORWARD mode (not from James),
      call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_FORWARD, rc=rc)

      do while (.not.ESMF_ClockIsDone(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
          nring = nring + 1
          !print *, "alarm4 is ringing!"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmRingerOff(alarm4, rc=rc)
        endif
      end do

      !print *, "alarm4 rang ", nring, " times, clock stepped ", iteration, " times."
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
      if (nring.eq.576 .and. iteration.eq.576) testPass = .true.
      call ESMF_Test(testPass.and.(rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Sticky Alarms 7
      !  Test timeStep > ringInterval, inspired by James Geiger in #2910203
      write(name, *) "Sticky Alarm Test 7"
      write(failMsg, *) " Alarm did not ring at the correct times or did not return ESMF_SUCCESS"
      call ESMF_TimeIntervalSet(timeStep, s=1800, rc=rc)
      call ESMF_TimeSet(startTime, yy=2003, mm=9, dd=1, &
                                   calendar=gregorianCalendar, rc=rc)
      ! dd=3, instead of original 30 in Jims report, to save test run time
      call ESMF_TimeSet(stopTime, yy=2003, mm=9, dd=3, &
                                   calendar=gregorianCalendar, rc=rc)
      clock2 = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                                name="The Clock", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=2003, mm=9, dd=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, s=900, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, rc=rc)
      testPass = .false.
      iteration = 0
      nring = 0
      do while (.not.ESMF_ClockIsStopTime(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
          nring = nring + 1
          !print *, "alarm4 is ringing!"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmRingerOff(alarm4, rc=rc)
        endif
        !call ESMF_AlarmPrint(alarm4, options="ringtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
      end do
      !print *, "At end of 1st forward run, nring = ", nring
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      ! TODO: test in REVERSE mode (not from James)
      !       currently doesn't ring in REVERSE
      call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_REVERSE, rc=rc)

      do while (.not.ESMF_ClockIsDone(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
          nring = nring + 1
          !print *, "alarm4 is ringing!"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmRingerOff(alarm4, rc=rc)
        endif
        !call ESMF_AlarmPrint(alarm4, options="ringtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
      end do
      !print *, "At end of reverse run, nring = ", nring
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      ! TODO: test double back in FORWARD mode (not from James)
      !       currently doesn't work until back into day 2  
      call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_FORWARD, rc=rc)

      do while (.not.ESMF_ClockIsDone(clock2, rc=rc))
        iteration = iteration + 1
        !print *, "clock2 timestep ", iteration
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock2, rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock2, options="currTime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm4, rc=rc)) then
          nring = nring + 1
          !print *, "alarm4 is ringing!"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmRingerOff(alarm4, rc=rc)
        endif
        !call ESMF_AlarmPrint(alarm4, options="ringtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
      end do
      !print *, "At end of 2nd forward run, nring = ", nring
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      !print *, "alarm4 rang ", nring, " times, clock stepped ", iteration, " times."
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
      if (nring.eq.288 .and. iteration.eq.288) testPass = .true.
      call ESMF_Test(testPass.and.(rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Fractional Time Alarms 1
      write(failMsg, *) " Did not return nstep=8, and ESMF_SUCCESS"
      write(name, *) "Fractional Time Alarm Test 1"
      call ESMF_TimeIntervalSet(timeStep, ms=10, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, ms=100, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, m=59, s=59, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                              name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=1999, mm=12, dd=31, h=23, m=59, s=59, &
                        ms=200, calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, rc=rc)

      ! number of clock time steps alarm rings for
      nstep = 0

      ! run the clock
      do while (.not. ESMF_ClockIsStopTime(clock2, rc=rc))
        if (ESMF_AlarmIsRinging(alarm4)) then
          nstep = nstep + 1
          call ESMF_AlarmRingerOff(alarm4)
        endif
        call ESMF_ClockAdvance(clock2, rc=rc)
      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nstep.eq.8), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Fractional Time Alarms 2
      write(failMsg, *) " Did not return nstep=4, and ESMF_SUCCESS"
      write(name, *) "Fractional Time Alarm Test 2"
      call ESMF_TimeIntervalSet(timeStep, sN=-22, sD=7, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, sN=-44, sD=7, rc=rc)
      call ESMF_TimeSet(startTime, yy=2000, mm=1, dd=1, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=1999, mm=12, dd=31, h=23, m=59, s=28, &
                        sN=4, sD=7, calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                              name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=1999, mm=12, dd=31, h=23, m=59, s=53, &
                        sN=2, sD=7, calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                ringInterval=alarmStep, rc=rc)

      !print *, "stopTime = "
      !call ESMF_TimePrint(stopTime)
      !print *, "alarmTime = "
      !call ESMF_TimePrint(alarmTime)
      !print *, "timeStep = "
      !call ESMF_TimeIntervalPrint(timeStep)
      !print *, "alarmStep = "
      !call ESMF_TimeIntervalPrint(alarmStep)

      ! number of clock time steps alarm rings for
      nstep = 0

      ! run the clock
      do while (.not. ESMF_ClockIsStopTime(clock2, rc=rc))
        if (ESMF_AlarmIsRinging(alarm4)) then
          nstep = nstep + 1
          call ESMF_AlarmRingerOff(alarm4)
        endif
        call ESMF_ClockAdvance(clock2, rc=rc)
      enddo

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nstep.eq.4), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringTime = clock startTime => should ring immediately
      !  upon alarm creation.
      write(failMsg, *) " Did not return alarm ringing and ESMF_SUCCESS"
      write(name, *) "Test Alarm ringTime = Clock startTime"
      call ESMF_TimeIntervalSet(timeStep, ms=10, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, m=59, s=59, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                              name="Clock 2", rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=startTime, rc=rc)

      bool = ESMF_AlarmIsRinging(alarm4)

      call ESMF_Test((bool.and.rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringTime = clock startTime *and* ringInterval specified
      !  upon alarm creation => should ring immediately.  Test 1
      !  From Tom Black in Support ticket 1989990.
      write(failMsg, *) " Did not return alarm ringing and ESMF_SUCCESS"
      write(name, *) "Alarm ringTime = Clock startTime with ringInterval, test 1"
      call ESMF_TimeIntervalSet(timeStep, m=2, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, s=60, rc=rc)
      call ESMF_TimeSet(startTime, yy=2007, mm=9, dd=18, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2007, mm=9, dd=19, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                              name="Clock 2", rc=rc)
      alarm4 = ESMF_AlarmCreate(clock=clock2, ringTime=startTime, &
                                ringInterval=alarmStep, rc=rc)

      call ESMF_AlarmGet(alarm4, ringTime=alarmTime, rc=rc)
      bool = ESMF_AlarmIsRinging(alarm4)

      call ESMF_Test(((alarmTime==startTime+alarmStep).and.bool.and. &
                       rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringTime = clock startTime *and* ringInterval specified
      !  upon alarm creation => should ring immediately.  Test 2
      !  From Tom Black in Support ticket 1989990.
      write(failMsg, *) " Did not return alarm ringing 21 times and ESMF_SUCCESS"
      write(name, *) "Alarm ringTime = Clock startTime with ringInterval, test 2"

      ! any single failure will cause the whole test to fail
      testPass = .true.

      call ESMF_TimeIntervalSet(timeStep, s=60, rc=rc)
      call ESMF_TimeIntervalSet(runDuration, s=3600, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, s=180, rc=rc)
      call ESMF_TimeSet(startTime, yy=2007, mm=9, dd=18, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, &
                              runDuration=runDuration, name="Clock 2", rc=rc)

      ! Tom Black's AlarmCreate() ...
      alarm4=ESMF_AlarmCreate(       &
         name             ='ALARM Recv from Parent'  &  !<-- Name of Alarm
        ,clock            =clock2    &  !<-- Each domain's ATM Driver Clock
        ,ringTime         =startTime &  !<-- First time the Alarm rings (ESMF)
        ,ringInterval     =alarmStep &  !<-- Recv from my parent at this
                                        !    frequency (ESMF)
        ,ringTimeStepCount=1       &  !<-- The Alarm rings for this many
                                      !    timesteps
        ,sticky           =.false. &  !<-- Alarm does not ring until turned off
        ,rc               =rc)

      ! run the clock
      ringCount = 0
      do while (.not. ESMF_ClockIsStopTime(clock2, rc=rc))
        call ESMF_ClockGet(clock2, advanceCount=forwardCount, rc=rc)        
        !print *, "At clock timestep #", forwardCount

        call ESMF_AlarmGet(alarm4, clock=clock1, rc=rc)
        call ESMF_ClockGet(clock1, currTime=currentTime2, rc=rc)
        !print *, "Alarm's clock's currTime ..."
        !call ESMF_TimePrint(currentTime2, options="string", rc=rc)
        if (clock2 /= clock1) then
           testPass = .false.
           !print *, "Alarm's clock and domain clock are *not* the same"
        endif

        call ESMF_ClockGet(clock2, currTime=currentTime, prevTime=prevTime, &
                           rc=rc)
        !print *, "Clock's currTime and prevTime ..."
        !call ESMF_TimePrint(currentTime, options="string", rc=rc)
        !call ESMF_TimePrint(prevTime, options="string", rc=rc)

        if (currentTime /= currentTime2) then
           testPass = .false.
           !print *, "Alarm's clock's currTime is *not* the same as the domain clock's currTime"
        endif

        call ESMF_AlarmGet(alarm4, ringTime=alarmTime, rc=rc)
        !print *, "Alarm's ringTime ..."
        !call ESMF_TimePrint(alarmTime, options="string", rc=rc)

        if (ESMF_AlarmIsRinging(alarm=alarm4, rc=rc)) then
            ringCount = ringCount + 1
            !print *, " Alarm is ringing"
            if (alarmTime /= currentTime+alarmStep) then
               testPass = .false.
               !print *, "  Alarm ringTime *not* equal to clock currTime + alarmStep"
            endif
        endif
        call ESMF_ClockAdvance(clock2, rc=rc)
      enddo

      call ESMF_ClockGet(clock2, advanceCount=forwardCount, rc=rc)        
      !print *, "End of clock run: At clock timestep #", forwardCount
      if (ESMF_AlarmIsRinging(alarm=alarm4, rc=rc)) then
          !print *, " Alarm is ringing"
          ringCount = ringCount + 1
      endif

      ! Alarm should have rung 21 times, including from timestep 0 (upon alarm
      ! creation) through timestep 60 (end of clock run) inclusive.
      ! Final ringTime should be one alarmStep past the clock end time.
      call ESMF_AlarmGet(alarm4, ringTime=alarmTime, rc=rc)
      call ESMF_Test(((alarmTime==startTime+runDuration+alarmStep).and. &
                       ringCount==21.and.testPass.and. &
                       rc.eq.ESMF_SUCCESS), name, failMsg, result, ESMF_SRCLINE)
      !print *, "Alarm rang ", ringCount, " times."
      !print *, "Alarm's final ringTime, after final clockAdvance() ..."
      !call ESMF_TimePrint(alarmTime, options="string", rc=rc)

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringTime = clock startTime *and* ringInterval specified
      !  upon alarm creation => should ring immediately.  Test 3, test REVERSE
      !  back to clock starttime.  Robustness test for solution to
      !  Tom Black problem in Support ticket 1989990.
      write(failMsg, *) " Did not return alarm ringing 41 times and ESMF_SUCCESS"
      write(name, *) "Alarm ringTime = Clock startTime with ringInterval, test 3 (reverse)"

      call ESMF_TimeIntervalSet(timeStep, s=60, rc=rc)
      call ESMF_TimeIntervalSet(runDuration, s=3600, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, s=180, rc=rc)
      call ESMF_TimeSet(startTime, yy=2007, mm=9, dd=18, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, &
                              runDuration=runDuration, name="Clock 2", rc=rc)

      ! like Tom Black's AlarmCreate(), but sticky
      alarm4=ESMF_AlarmCreate(       &
         name             ='ALARM Recv from Parent'  &  !<-- Name of Alarm
        ,clock            =clock2    &  !<-- Each domain's ATM Driver Clock
        ,ringTime         =startTime &  !<-- First time the Alarm rings (ESMF)
        ,ringInterval     =alarmStep &  !<-- Recv from my parent at this
                                        !    frequency (ESMF)
        ,rc               =rc)

      ! number of clock time steps alarm rings for
      nring = 0

      ! number of times the clock has been run
      nclock = 0

      do while (nclock < 2)
        ! run the clock
        do while (.not. ESMF_ClockIsDone(clock2, rc=rc))
          if (ESMF_AlarmIsRinging(alarm4)) then
            nring = nring + 1
            !print *, "on"
            call ESMF_AlarmRingerOff(alarm4, rc=rc)
          else
            !print *, "off"
          endif
          call ESMF_ClockAdvance(clock2, rc=rc)
        enddo

        if (nclock.eq.0) then
          call ESMF_ClockGet(clock2, direction=forwardDirection, &
                                     advanceCount=forwardCount, rc=rc)
          !print *, "forwardCount = ", forwardCount
        else
          call ESMF_ClockGet(clock2, direction=reverseDirection, &
                             advanceCount=reverseCount, rc=rc)
          !print *, "reverseCount = ", reverseCount
        endif

        call ESMF_ClockSet(clock2, direction=ESMF_DIRECTION_REVERSE, rc=rc)
        nclock = nclock + 1
      enddo

      ! count ring at the end point
      if (ESMF_AlarmIsRinging(alarm4)) then
        nring = nring + 1
        !print *, "on"
        call ESMF_AlarmRingerOff(alarm4, rc=rc)
      else
        !print *, "off"
      endif

      call ESMF_Test((rc.eq.ESMF_SUCCESS).and.(nring.eq.41).and. &
                     (forwardCount.eq.60).and.(reverseCount.eq.0).and. &
                     (forwardDirection.eq.ESMF_DIRECTION_FORWARD).and. &
                     (reverseDirection.eq.ESMF_DIRECTION_REVERSE).and. &
                     ESMF_ClockIsReverse(clock2), &
                      name, failMsg, result, ESMF_SRCLINE)

      !print *, "nring = ", nring, " nclock = ", nclock

      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringTime = clock startTime *and* ringInterval specified
      !  upon alarm creation *and* after clock restored from restart state --
      !  with 2 calls to ESMF_ClockSet() -- should ring immediately.  
      !  This approach to restart will become obsolete with the implementation
      !  of WriteRestart()/ReadStart(), but it should remain technically valid,
      !  along with this unit test.  From Ratko Vasic in ticket #2685243.

      write(failMsg, *) " Did not return alarm ringing and ESMF_SUCCESS"
      write(name, *) "Alarm ringTime = Clock startTime with ringInterval, after clock restore"

      call ESMF_TimeIntervalSet(timeStep, m=3, rc=rc)
      call ESMF_TimeIntervalSet(TIMEINTERVAL_HISTORY, h=3, rc=rc)
      call ESMF_TimeSet(startTime, mm=9, dd=18, yy=2007, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, mm=9, dd=19, yy=2007, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(currTime, mm=9, dd=18, yy=2007, h=12, &
                        calendar=gregorianCalendar, rc=rc)

      CLOCK_ATM=ESMF_ClockCreate( &
        name       ='CLOCK_ATM'   &  !<-- The ATM Clock's name
       ,timeStep   =timeStep      &  !<-- The fundamental timestep in
                                     !     this component
       ,startTime  =startTime     &  !<-- Start time of simulation
       ,stopTime   =stopTime      &  !<-- Stop time of simulation
       ,rc         =rc)

      ! 1st ClockSet() to reset currTime and advanceCount for restart
      call ESMF_ClockSet(             &
        clock       =CLOCK_ATM        &  !<-- The ATM Component's Clock
       ,currtime    =currTime         &  !<-- Current time of simulation
       ,advanceCount=240_ESMF_KIND_I8 &  !<-- Timestep at this current time
       ,rc          =rc)

      ! 2nd ClockSet() with same values that CLOCK_ATM already has; should
      ! have no affect
      call ESMF_ClockSet(clock    =CLOCK_ATM &
                        ,currtime =currTime  &
                        ,starttime=startTime &
                        ,rc       =rc)

      !call ESMF_ClockPrint(CLOCK_ATM, rc=rc)

      ALARM_HISTORY=ESMF_AlarmCreate(          &
        name             ='ALARM_HISTORY'      &
       ,clock            =CLOCK_ATM            &  ! <-- ATM Clock
       ,ringTime         =currTime             &  ! <-- Forecast/Restart start
                                                  !      time (ESMF)
       ,ringInterval     =TIMEINTERVAL_HISTORY &  ! <-- Time interval between
       ,ringTimeStepCount=1                    &  ! <-- The Alarm rings for
                                                  !      this many timesteps
       ,sticky           =.false.              &  ! <-- Alarm does not ring
                                                  !      until turned off
       ,rc               =rc) 

      !call ESMF_AlarmPrint(ALARM_HISTORY, rc=rc)

      call ESMF_AlarmGet(ALARM_HISTORY, ringTime=alarmTime, rc=rc)
      bool = ESMF_AlarmIsRinging(ALARM_HISTORY)

      call ESMF_Test(((alarmTime==currTime+TIMEINTERVAL_HISTORY).and.bool &
                       .and. rc.eq.ESMF_SUCCESS), &
                       name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy(ALARM_HISTORY, rc=rc)
      call ESMF_ClockDestroy(CLOCK_ATM, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm list re-allocation within clock, part 1
      write(failMsg, *) " Did not return ESMF_SUCCESS"
      write(name, *) "Alarm list reallocation Test 1"
      call ESMF_TimeIntervalSet(timeStep, ms=10, rc=rc)
      call ESMF_TimeIntervalSet(alarmStep, ms=100, rc=rc)
      call ESMF_TimeSet(startTime, yy=1999, mm=12, dd=31, h=23, m=59, s=59, &
                        calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet(stopTime, yy=2000, mm=1, dd=1, &
                        calendar=gregorianCalendar, rc=rc)
      clock2=ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                              name="Clock 2", rc=rc)
      call ESMF_TimeSet(alarmTime, yy=1999, mm=12, dd=31, h=23, m=59, s=59, &
                        ms=200, calendar=gregorianCalendar, rc=rc)

      ! fill up clock's alarmList
      do i=1,200
        alarm5(i) = ESMF_AlarmCreate(clock=clock2, ringTime=alarmTime, &
                                     ringInterval=alarmStep, rc=rc)
      enddo

      ! add one more alarm than there is space for (200), forcing a
      !   reallocation to 400 alarms
      ! also, set 201st alarm to be the first to ring upon the 1st timestep
      call ESMF_TimeSet(alarmTime, yy=1999, mm=12, dd=31, h=23, m=59, s=59, &
                        ms=10, calendar=gregorianCalendar, rc=rc)
      alarm4 = ESMF_AlarmCreate(name="201st Alarm", clock=clock2, &
                                ringTime=alarmTime, ringInterval=alarmStep, &
                                rc=rc)

      ! see if the 201st alarm was successfully added to the clock
      call ESMF_Test((rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm list re-allocation within clock, part 2
      write(failMsg, *) " Did not return 201 alarms and name '201st Alarm'"
      write(name, *) "Alarm list reallocation Test 2"
      call ESMF_ClockGetAlarmList(clock2, ESMF_ALARMLIST_ALL, &
                                  alarmList=alarmList, &
                                  alarmCount=alarmCount, rc=rc)
      write(*,*) "rc=",rc
      call ESMF_AlarmGet(alarmList(alarmCount), name=aName, rc=rc)

      !print *, "alarmCount = ", alarmCount
      !print *, "201st alarm name = ", aName
      write(*,*) "rc=",rc

      ! see if we have 201 alarms and if the 201st alarm has the right name!
      call ESMF_Test((alarmCount.eq.201).and.(aName.eq."201st Alarm") &
                      .and.(rc.eq.ESMF_SUCCESS), &
                      name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm list re-allocation within clock, part 3
      write(failMsg, *) " Did not return 201st alarm ringing"
      write(name, *) "Alarm list reallocation Test 3"
      call ESMF_ClockAdvance(clock2, rc=rc)
      call ESMF_ClockGetAlarmList(clock2, ESMF_ALARMLIST_RINGING, &
                                  alarmList=alarmList, &
                                  alarmCount=alarmCount, rc=rc)
      ! double check ringing with Alarm API call
      isringing = ESMF_AlarmIsRinging(alarm4, rc=rc)

      !print *, "alarmCount = ", alarmCount
      call ESMF_AlarmGet(alarmList(alarmCount), name=aName, rc=rc)
      !print *, "Ringing alarm name = ", trim(aName), ", is ringing = ", &
            !isringing

      ! see if the 201st alarm is the only one ringing
      call ESMF_Test(isringing.and.(alarmCount.eq.1) &
                     .and.(aName.eq."201st Alarm") &
                     .and.(rc.eq.ESMF_SUCCESS), &
                     name, failMsg, result, ESMF_SRCLINE)

      ! cleanup
      do i=1,200
        call ESMF_AlarmDestroy(alarm5(i), rc=rc)
      enddo
      call ESMF_AlarmDestroy(alarm4, rc=rc)
      call ESMF_ClockDestroy(clock2, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      ! Based on reproducer clocktester.F90 from Atanas. See bug #1531948.
      write(failMsg, *) " Did not ring enough times during forward/backward march"
      write(name, *) "Test ESMF_DIRECTION_FORWARD to a non-sticky alarm " // &
                     "point, ESMF_DIRECTION_REVERSE, ESMF_DIRECTION_FORWARD"
      testPass = .true.
      call ESMF_TimeSet (startTime, yy=2009, mm=1, dd=1, &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeSet (stopTime, yy=2009, mm=1, dd=2, h=0, &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (timeStep, s=3600, rc=rc)
      clock = ESMF_ClockCreate(timeStep, startTime, stopTime=stopTime, &
                               name="ApplClock", rc=rc)
      if (rc /= ESMF_SUCCESS) testPass = .false.

      call ESMF_TimeIntervalSet (alarmStep, s=7200, rc=rc)
      !call ESMF_TimeIntervalSet(ringDuration, h=1, rc=rc)
      !call ESMF_TimeSet(alarmStopTime, yy=2009, mm=1, dd=1, h=21, &
      !                  calendar=gregorianCalendar, rc=rc)
      alarm1 = ESMF_AlarmCreate (clock=clock,  &
          ringTime=startTime, ringInterval=alarmStep, &
      !    ringDuration=ringDuration, &
      !    stopTime=alarmStopTime, &
          sticky=.false., name="testAlarm", rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      !call ESMF_AlarmPrint(alarm1, options="ringduration string", rc=rc)
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
      !call ESMF_AlarmPrint(alarm1, options="ringtimestepcount string", rc=rc)
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      alarmCount = 0
      expectedCount = 22 
      do
        !print *, "***********************Top of loop 1 **********************"
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock, options="currtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm1, rc=rc)) then
          alarmCount = alarmCount + 1
          !print *, "alarm1 is ringing"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          !call ESMF_AlarmRingerOff(alarm1, rc=rc)
        endif
        !call ESMF_AlarmPrint(alarm1, options="ringbegin string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_AlarmPrint(alarm1, options="ringtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_AlarmPrint(alarm1, options="ringend string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock, rc=rc)
        !call ESMF_AlarmPrint(alarm1, options="ringend string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_ClockIsStopTime(clock, rc=rc)) exit
      enddo
      !print *, "At end of 1st forward run, alarmCount = ", alarmCount
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      call ESMF_ClockSet(clock, direction=ESMF_DIRECTION_REVERSE, rc=rc)

      i=0
      do
        !print *, "***********************Top of loop 2 **********************"
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock, options="currtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm1, rc=rc)) then
          alarmCount = alarmCount + 1
          !print *, "alarm1 is ringing"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          !call ESMF_AlarmRingerOff(alarm1, rc=rc)
        endif
        !call ESMF_AlarmPrint(alarm1, options="ringbegin string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_AlarmPrint(alarm1, options="ringtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock, rc=rc)
        i = i+1
        if (i == 5) exit
      enddo
      !print *, "At end of reverse run, alarmCount = ", alarmCount
      !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      call ESMF_ClockSet(clock, direction=ESMF_DIRECTION_FORWARD, rc=rc)

      i=0
      do
        !print *, "***********************Top of loop 3 **********************"
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_ClockPrint(clock, options="currtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        if (ESMF_AlarmIsRinging(alarm1, rc=rc)) then
          alarmCount = alarmCount + 1
          !print *, "alarm1 is ringing"
          !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          !call ESMF_AlarmRingerOff(alarm1, rc=rc)
        endif
        !call ESMF_AlarmPrint(alarm1, options="ringbegin string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        !call ESMF_AlarmPrint(alarm1, options="ringtime string", rc=rc)
        !call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
        call ESMF_ClockAdvance(clock, rc=rc)
        i = i+1
        if (i == 15) exit
      enddo

      if (.not. testPass .or. alarmCount /= expectedCount) then
          if (.not. testPass) print *, 'bad return codes discovered'
          write (failMsg,*) trim (failMsg), ', alarmCount = ', alarmCount, ', expected = ', expectedCount
          print *, 'The alarm ringTime may be stuck at:'
          call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
          call ESMF_AlarmPrint (alarm1, options="ringTime string", rc=rc)
          call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)
      end if

      call ESMF_Test (testPass .and. alarmCount == expectedCount, &
                     name, failMsg, result, ESMF_SRCLINE)
      call ESMF_UtilIOUnitFlush(ESMF_UtilIOStdout)

      call ESMF_AlarmDestroy (alarm1, rc=rc)
      call ESMF_ClockDestroy (clock, rc=rc)

      ! ----------------------------------------------------------------------------
      !EX_UTest
      !Test Alarm ringTime increment, first forwards a fixed number of
      !timesteps, stopping at an alarm ringing time step.
      !Using ESMF_DIRECTION_REVERSE, step backwards to some time prior to the
      !clock's startTime.  Then go ESMF_DIRECTION_FORWARD to one step past an
      !alarm ringing time step, and then ESMF_DIRECTION_REVERSE once more. 
      ! Count number of rings.  See bug #1531948.
      write(failMsg, *) " Did not ring enough times during forward/backward march"
      write(name, *) "Test ESMF_DIRECTION_FORWARD to an alarm point, " // &
                     "ESMF_DIRECTION_REVERSE, ESMF_DIRECTION_FORWARD, " // &
                     "ESMF_DIRECTION_REVERSE"

      testPass = .true.
      call ESMF_TimeSet (startTime, yy=2008, mm=1, dd=23, h=0,  &
          calendar=gregorianCalendar, rc=rc)
      !call ESMF_TimePrint (startTime, options="string isofrac", rc=rc)
      call ESMF_TimeIntervalSet (timeStep, h=3, rc=rc)
      clock = ESMF_ClockCreate(startTime = startTime, timeStep=timeStep, &
                              name="clock 1", rc=rc)
      if (rc /= ESMF_SUCCESS) testPass = .false.

      call ESMF_TimeSet (alarmTime, yy=2008, mm=1, dd=23, h=6,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (alarmStep, h=6, rc=rc)
      alarm1 = ESMF_AlarmCreate (  &
          name="Alarm 1", clock=clock,  &
          ringTime=alarmTime, ringInterval=alarmStep, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      alarmCount = 0
      expectedCount = 11
      do, i=1,6
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at forwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at forwards timestep', i
        end if
      end do

      !print *, 'SETTING CLOCK BACKWARDS'
      call ESMF_ClockSet (clock, direction=ESMF_DIRECTION_REVERSE, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      do, i=5, -5, -1
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at backwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
          !call ESMF_AlarmPrint(alarm1, options="ringTime string", rc=rc)
        else
          !print *, 'alarm not ringing at backwards timestep', i
          !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        end if
      end do

      !print *, 'SETTING CLOCK FORWARDS'
      call ESMF_ClockSet (clock, direction=ESMF_DIRECTION_FORWARD, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      do, i=-4,7
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at forwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at forwards timestep', i
        end if
      end do

      !print *, 'SETTING CLOCK BACKWARDS'
      call ESMF_ClockSet (clock, direction=ESMF_DIRECTION_REVERSE, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      do, i=6, -5, -1
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at backwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
          !call ESMF_AlarmPrint(alarm1, options="ringTime string", rc=rc)
        else
          !print *, 'alarm not ringing at backwards timestep', i
          !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        end if
      end do

      if (.not. testPass .or. alarmCount /= expectedCount) then
          if (.not. testPass) print *, 'bad return codes discovered'
          write (failMsg,*) trim (failMsg), ', alarmCount = ', alarmCount, ', expected = ', expectedCount
          print *, 'The alarm ringTime may be stuck at:'
          call ESMF_AlarmPrint (alarm1, options="ringTime string", rc=rc)
      end if

      call ESMF_Test (testPass .and. alarmCount == expectedCount, &
                     name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy (alarm1, rc=rc)
      call ESMF_ClockDestroy (clock, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringTime increment, first forwards a fixed number of
      !timesteps, stopping at an alarm ringing time step.  Using a negative
      !timestemp, step backwards to some time prior to the clock's startTime.
      !Count number of rings.  See bugs #1531948, #1457135.
      write(failMsg, *) " Did not ring enough times during forward/backward march"
      write(name, *) "Test forward to an alarm point, then step backward using a negative timeStep"

      testPass = .true.
      call ESMF_TimeSet (startTime, yy=2008, mm=1, dd=23, h=0,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (timeStep, h=3, rc=rc)
      clock = ESMF_ClockCreate(startTime = startTime, timeStep=timeStep, &
                               name="clock 1", rc=rc)
      if (rc /= ESMF_SUCCESS) testPass = .false.

      call ESMF_TimeSet (alarmTime, yy=2008, mm=1, dd=23, h=6,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (alarmStep, h=6, rc=rc)
      alarm1 = ESMF_AlarmCreate (  &
          name="Alarm 1", clock=clock,  &
          ringTime=alarmTime, ringInterval=alarmStep, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      alarmCount = 0
      expectedCount = 8
      do, i=1,6
        call ESMF_ClockAdvance (clock, rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at forwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at forwards timestep', i
        end if
      end do

      !print *, 'SETTING CLOCK BACKWARDS WITH NEGATIVE TIMESTEP'
      timeStep = -timeStep
      call ESMF_ClockSet (clock, timeStep=timeStep, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      do, i=5, -5, -1
        call ESMF_ClockAdvance (clock, rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at backwards timestep', i, ', time:'
          !call ESMF_ClockPrint (clock, options="currTime string")
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at backwards timestep', i
          !call ESMF_ClockPrint (clock, options="currTime string")
        end if
      end do

      if (.not. testPass .or. alarmCount /= expectedCount) then
          if (.not. testPass) print *, 'bad return codes discovered'
          write (failMsg,*) trim (failMsg), ', alarmCount = ', alarmCount, ', expected = ', expectedCount
          print *, 'The alarm ringTime may be stuck at:'
          call ESMF_AlarmPrint (alarm1, options="ringTime string", rc=rc)
      end if

      call ESMF_Test (testPass .and. alarmCount == expectedCount, &
                     name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy (alarm1, rc=rc)
      call ESMF_ClockDestroy (clock, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringTime increment, first forwards a fixed number of
      !timesteps, stopping at a non-alarm ringing time step.  Using a negative
      !timestemp, step backwards to some time prior to the clock's startTime.
      !Count number of rings.  See bugs #1531948, #1457135.
      write(failMsg, *) " Did not ring enough times during forward/backward march"
      write(name, *) "Test forward to a non-ringing step, then step backward using a negative timeStep"

      testPass = .true.
      call ESMF_TimeSet (startTime, yy=2008, mm=1, dd=23, h=0,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (timeStep, h=3, rc=rc)
      clock = ESMF_ClockCreate(startTime = startTime, timeStep=timeStep, &
                               name="clock 1", rc=rc)
      if (rc /= ESMF_SUCCESS) testPass = .false.

      call ESMF_TimeSet (alarmTime, yy=2008, mm=1, dd=23, h=6,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (alarmStep, h=6, rc=rc)
      alarm1 = ESMF_AlarmCreate (  &
          name="Alarm 1", clock=clock,  &
          ringTime=alarmTime, ringInterval=alarmStep, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      alarmCount = 0
      expectedCount = 9
      do, i=1,7
        call ESMF_ClockAdvance (clock, rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at forwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at forwards timestep', i
        end if
      end do

      !print *, 'SETTING CLOCK BACKWARDS WITH NEGATIVE TIMESTEP'
      timeStep = -timeStep
      call ESMF_ClockSet (clock, timeStep=timeStep, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      do, i=6, -5, -1
        call ESMF_ClockAdvance (clock, rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at backwards timestep', i, ', time:'
          !call ESMF_ClockPrint (clock, options="currTime string")
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at backwards timestep', i
          !call ESMF_ClockPrint (clock, options="currTime string")
        end if
      end do

      if (.not. testPass .or. alarmCount /= expectedCount) then
          if (.not. testPass) print *, 'bad return codes discovered'
          write (failMsg,*) trim (failMsg), ', alarmCount = ', alarmCount, ', expected = ', expectedCount
          print *, 'The alarm ringTime may be stuck at:'
          call ESMF_AlarmPrint (alarm1, options="ringTime string", rc=rc)
      end if

      call ESMF_Test (testPass .and. alarmCount == expectedCount, &
                     name, failMsg, result, ESMF_SRCLINE)

      ! 
      ! ----------------------------------------------------------------------------

      !EX_UTest
      write(failMsg, *) " Did not show null clock in alarm"
      write(name, *) "Test alarm after destroying its associated clock"
      call ESMF_ClockDestroy (clock, rc=rc)
      willRingNext = ESMF_AlarmWillRingNext(alarm1, rc=rc)
      call ESMF_Test (rc==ESMC_RC_PTR_NULL .and. .not.willRingNext, &
                     name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy (alarm1, rc=rc)

      ! ----------------------------------------------------------------------------

      !EX_UTest
      !Test Alarm ringTime increment, arbitrarily forwards and backwards using
      ! positive and negative timesteps.  Alarm ringTime and ringInterval are
      ! changed with each change in timestep direction.  The extent of the run
      ! of each timestep value is different with each change in direction.
      write(failMsg, *) " Did not ring enough times during arbitrary forward/backward march"
      write(name, *) "Test arbitrary positive/negative timeStep runs with differing alarms"

      testPass = .true.
      alarmCountPass = .true.

      call ESMF_TimeSet (startTime, yy=2008, mm=1, dd=23, h=0,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (timeStep, h=1, rc=rc)
      clock = ESMF_ClockCreate(startTime=startTime, timeStep=timeStep, &
                               name="clock 1", rc=rc)
      if (rc /= ESMF_SUCCESS) testPass = .false.

      call ESMF_TimeSet (alarmTime, yy=2008, mm=1, dd=23, h=2,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (alarmStep, h=2, rc=rc)
      alarm1 = ESMF_AlarmCreate (  &
          name="Alarm 1", clock=clock,  &
          ringTime=alarmTime, ringInterval=alarmStep, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      alarmCount = 0
      expectedCount = 15

      do, i=1,7
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at forwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at forwards timestep', i
        end if
      end do
      if (alarmCountPass .and. alarmCount /= 3) then
        alarmCountPass = .false.
        !print *, 'alarmCount = ', alarmCount, ' not 3 at 1st turnaround point'
      end if

      !print *, 'SETTING CLOCK BACKWARDS WITH NEW NEGATIVE TIMESTEP AND RINGINTERVAL'
      timeStep = -2 * timeStep
      alarmStep = -2 * alarmStep
      call ESMF_ClockSet (clock, timeStep=timeStep, rc=rc)
      call ESMF_AlarmSet (alarm1, ringInterval=alarmStep, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      do, i=6, -5, -1
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at backwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at backwards timestep', i
        end if
      end do
      if (alarmCountPass .and. alarmCount /= 9) then
        alarmCountPass = .false.
        !print *, 'alarmCount = ', alarmCount, ' not 9 at 2nd turnaround point'
      end if

      !print *, 'SETTING CLOCK FORWARDS WITH NEW POSITIVE TIMESTEP, RINGINTERVAL, and RINGTIME'
      call ESMF_TimeIntervalSet (timeStep, h=3, rc=rc)
      call ESMF_TimeSet (alarmTime, yy=2008, mm=1, dd=22, h=11,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_TimeIntervalSet (alarmStep, h=6, rc=rc)
      call ESMF_ClockSet (clock, timeStep=timeStep, rc=rc)
      call ESMF_AlarmSet (alarm1, ringTime=alarmTime, ringInterval=alarmStep, &
                          rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      do, i=1,5
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at forwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at forwards timestep', i
        end if
      end do
      if (alarmCountPass .and. alarmCount /= 11) then
        alarmCountPass = .false.
        !print *, 'alarmCount = ', alarmCount, ' not 11 at 3rd turnaround point'
      end if

      !print *, 'SETTING CLOCK BACKWARDS WITH NEW NEGATIVE TIMESTEP AND RINGTIME'
      call ESMF_TimeIntervalSet (timeStep, h=-1, rc=rc)
      call ESMF_TimeSet (alarmTime, yy=2008, mm=1, dd=22, h=21,  &
          calendar=gregorianCalendar, rc=rc)
      call ESMF_ClockSet (clock, timeStep=timeStep, rc=rc)
      call ESMF_AlarmSet (alarm1, ringTime=alarmTime, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.

      do, i=4,2,-1
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing at backwards timestep', i
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing at backwards timestep', i
        end if
      end do
      if (alarmCountPass .and. alarmCount /= 12) then
        alarmCountPass = .false.
        !print *, 'alarmCount = ', alarmCount, ' not 12 at 4th turnaround point'
      end if

      !print *, 'SETTING CLOCK FORWARDS WITH NEW POSITIVE TIMESTEP'
      timeStep = -timeStep
      call ESMF_ClockGet(clock, currTime=startTime, rc=rc)
      call ESMF_ClockSet(clock, timeStep=timeStep, startTime=startTime, &
                         runTimeStepCount=15, rc=rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
   
      do while (.not. ESMF_ClockIsDone(clock, rc=rc))
        call ESMF_ClockAdvance (clock, rc=rc)
        !call ESMF_ClockPrint (clock, options="currTime string", rc=rc)
        if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
        if (ESMF_alarmIsRinging (alarm1)) then
          alarmCount = alarmCount + 1
          !print *, 'alarm IS ringing'
          call ESMF_AlarmRingerOff (alarm1, rc=rc)
        else
          !print *, 'alarm not ringing'
        end if
      end do
      if (alarmCountPass .and. alarmCount /= expectedCount) then
        alarmCountPass = .false.
        !print *, 'alarmCount = ', alarmCount, ', not ', expectedCount, ' at the end'
      end if

      if (.not. alarmCountPass) then
          write(failMsg,*) trim(failMsg), ', alarmCount incorrect at one or more turnaround points'
          !print *, 'Final alarmCount = ', alarmCount, ', expected = ', expectedCount
          !print *, 'Final alarm ringTime:'
          !call ESMF_AlarmPrint (alarm1, "ringTime string", rc=rc)
      end if

      if (.not. testPass) print *, 'bad return codes discovered'

      call ESMF_Test (testPass .and. alarmCountPass, &
                     name, failMsg, result, ESMF_SRCLINE)

      call ESMF_AlarmDestroy (alarm1, rc=rc)
      call ESMF_ClockDestroy (clock, rc=rc)

      ! ----------------------------------------------------------------------------
#if 0
      ! The following tests are from Ben@NASA's support ticket 3614994
      write(failMsg, *) " Alarms did not rewind correct number of times "
      write(name, *) "Test multiple alarms rewind correct number of times "
      rc = ESMF_SUCCESS
      testPass = .true.
      call Test_ReverseAlarms(testPass, rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
      if (.not. testPass) print *, 'bad return codes discovered'
      call ESMF_Test (testPass, name, failMsg, result, ESMF_SRCLINE)

      ! ----------------------------------------------------------------------------

      write(failMsg, *) " Alarms hang... "
      write(name, *) "Test multiple alarms replay without hanging "
      rc = ESMF_SUCCESS
      testPass = .true.
      call Test_AlarmHang(testPass, rc)
      if (testPass .and. rc /= ESMF_SUCCESS) testPass = .false.
      if (.not. testPass) print *, 'bad return codes discovered'
      call ESMF_Test (testPass, name, failMsg, result, ESMF_SRCLINE)
#endif

      ! ----------------------------------------------------------------------------
#endif

      ! destroy calendars
      call ESMF_CalendarDestroy(esmf_360dayCalendar, rc=rc)
      call ESMF_CalendarDestroy(no_leapCalendar, rc=rc)
      call ESMF_CalendarDestroy(julianCalendar, rc=rc)
      call ESMF_CalendarDestroy(gregorianCalendar, rc=rc)

      ! finalize ESMF framework
      call ESMF_TestEnd(ESMF_SRCLINE)

#if 0
  contains

    subroutine test_reverseAlarms(testPass, rc)
      implicit none

      logical, intent(out) :: testPass
      integer :: status,rc

      type(ESMF_TimeInterval) :: dt
      type(ESMF_Time) :: start_time, clock_start, clock_end
      type(ESMF_Clock) :: clock
      character(len=5) :: add_2nd
      integer :: nargs

      type(ESMF_TimeInterval) :: tint
      type(ESMF_Alarm) :: esmfalarm, firstalarm

      integer :: i,nstep

      type(ESMF_Time) :: time
      character(len=10) :: iam='test_clock'
      logical :: esmf_ring
      integer :: nalarms, n_rings=0

      testPass = .false.
      call ESMF_CalendarSetDefault ( ESMF_CALKIND_GREGORIAN, rc=status )
      call verify_(status)
      call ESMF_TimeSet(clock_start,yy=2000,mm=1,dd=1,h=21,m=0,s=0,rc=status)
      call verify_(status)
      call ESMF_TimeSet(clock_end,yy=2000,mm=12,dd=1,h=21,m=0,s=0,rc=status)
      call verify_(status)
      call ESMF_TimeSet(start_time,yy=2000,mm=10,dd=1,h=21,m=0,s=0,rc=status)
      call verify_(status)
      call ESMF_TimeIntervalSet(dt,S=900, sN=0, sD=1,rc=status)
      call verify_(status)
      clock= ESMF_ClockCreate(timeStep=dt,startTime=clock_start,stopTime=clock_end,rc=status)
      call verify_(status)
      call ESMF_ClockSet(clock,currTime=start_time,rc=status)
      call verify_(status)

      call ESMF_ClockGet(clock,currtime=start_time,rc=status)
      if(status /= ESMF_SUCCESS) call ESMF_Finalize()
      call ESMF_TimeIntervalSet(tint,h=2,rc=status)
      if(status /= ESMF_SUCCESS) call ESMF_Finalize()
      firstalarm = ESMF_AlarmCreate(clock=clock,ringInterval=tint,ringTime=start_time,sticky=.false.,name="alarm1",rc=status)
      if(status /= ESMF_SUCCESS) call ESMF_Finalize()

      nstep=47
      call ESMF_ClockGet(clock,currtime=start_time,alarmCount=nalarms,rc=status)
      !write(*,*) "alarms already in clock: ",nalarms
      call verify_(status)
      call ESMF_TimeIntervalSet(tint,h=1,rc=status)
      call verify_(status)
      esmfalarm = ESMF_AlarmCreate(clock=clock,ringInterval=tint,ringTime=start_time,sticky=.false.,name="alarm2",rc=status)
      do i=1,nstep
         call ESMF_ClockGet(clock,currTime=time)
         esmf_ring = ESMF_AlarmIsRinging(esmfalarm,rc=status)
         call verify_(status)
         if ( esmf_ring) then
             !write(*,*)'ringing'
            !call ESMF_TimePrint(time,options='string')
         end if
         call ESMF_ClockAdvance(clock)
      enddo
      call ESMF_ClockSet(clock, direction=ESMF_DIRECTION_REVERSE, rc=status)
      call verify_(status)
      !write(*,*)"*************** start rewind *********************"
      i = 0
      do
         i = i + 1
         !write(*,*) 'Rewind step: ', i
         call ESMF_ClockAdvance(clock,rc=status)
         call verify_(status)
         call ESMF_ClockGet(clock,currTime=time)
         if (ESMF_AlarmIsRinging(esmfalarm)) then
            !write(*,*)'rewinding one step ',ESMF_AlarmIsRinging(esmfalarm)
            n_rings = n_rings + 1
            !call ESMF_TimePrint(time,options='string')
         end if
         
         if (time == start_time) exit
      enddo
      call ESMF_ClockSet(clock, direction=ESMF_DIRECTION_FORWARD, rc=status)
      call verify_(status)
      !write(*,*)"*************** end rewind *********************"
      do i=1,nstep*2
         call ESMF_ClockGet(clock,currTime=time)
         esmf_ring = ESMF_AlarmIsRinging(esmfalarm,rc=status)
         call verify_(status)
         if ( esmf_ring ) then
            !write(*,*)'ringing'
            !call ESMF_TimePrint(time,options='string')
         end if
         call ESMF_ClockAdvance(clock)
      enddo
      if(n_rings == 12) testPass = .true.

       call ESMF_AlarmDestroy(esmfalarm, rc=status)
       call verify_(status)
       call ESMF_AlarmDestroy(firstalarm, rc=status)
       call verify_(status)
       call ESMF_ClockDestroy(clock, rc=status)
       call verify_(status)
           
    end subroutine Test_ReverseAlarms

    subroutine Test_AlarmHang(testPass, rc)
      logical, intent(out) :: testPass
      integer :: status,rc

      type(ESMF_TimeInterval) :: dt
      type(ESMF_Time) :: start_time, clock_start, clock_end
      type(ESMF_Clock) :: clock
      character(len=5) :: add_2nd
      integer :: nargs
           
      integer :: i,nstep

      type(ESMF_TimeInterval) :: tint
      type(ESMF_Time) :: time
      type(ESMF_Alarm) :: esmfalarm
      type(ESMF_Alarm) :: testalarm
      character(len=10) :: iam='test_clock'
      logical :: esmf_ring
      integer :: nalarms

      testPass = .false.

      call ESMF_CalendarSetDefault ( ESMF_CALKIND_GREGORIAN, rc=status )
      call verify_(status)
      call ESMF_TimeSet(clock_start,yy=2000,mm=1,dd=1,h=21,m=0,s=0,rc=status)
      call verify_(status)
      call ESMF_TimeSet(clock_end,yy=2000,mm=12,dd=1,h=21,m=0,s=0,rc=status)
      call verify_(status)
      call ESMF_TimeSet(start_time,yy=2000,mm=10,dd=1,h=21,m=0,s=0,rc=status)
      call verify_(status)
      call ESMF_TimeIntervalSet(dt,S=900, sN=0, sD=1,rc=status)
      call verify_(status)
      clock= ESMF_ClockCreate(timeStep=dt,startTime=clock_start,stopTime=clock_end,rc=status)
      call verify_(status)
      call ESMF_ClockSet(clock,currTime=start_time,rc=status)
      call verify_(status)

      nstep = 12
      call ESMF_ClockGet(clock,currtime=start_time,alarmCount=nalarms,rc=status)
      !write(*,*) "alarms already in clock: ",nalarms
      call verify_(status)
      call ESMF_TimeIntervalSet(tint,h=1,rc=status)
      call verify_(status)
      esmfalarm = ESMF_AlarmCreate(clock=clock,ringInterval=tint,ringTime=start_time,sticky=.false.,name="alarm2",rc=status)
      !call ESMF_TimeIntervalSet(tint,h=3,rc=status)
      !testalarm = ESMF_AlarmCreate(clock=clock,name="alarm3",ringInterval=tint, ringtime=start_time,sticky=.true.,rc=status)
      testalarm = ESMF_AlarmCreate(clock=clock,name="alarm3",ringtime=start_time,sticky=.true.,rc=status)
      call verify_(status)
      call ESMF_AlarmRingerOff(testalarm,rc=status)
      call verify_(status)
      do i=1,nstep
         call ESMF_AlarmRingerOn(testalarm,rc=status)
         call verify_(status)
         call ESMF_ClockGet(clock,currTime=time)
         esmf_ring = ESMF_AlarmIsRinging(esmfalarm,rc=status)
         call verify_(status)
         if ( esmf_ring) then
            !write(*,*)'ringing: esmfalarm'
            call ESMF_TimePrint(time,options='string')
         end if
         call ESMF_ClockAdvance(clock)
      enddo
      call ESMF_ClockSet(clock, direction=ESMF_DIRECTION_REVERSE, rc=status)
      call verify_(status)
      call ESMF_AlarmRingerOff(testalarm,rc=status)
      call verify_(status)
      !write(*,*)"*************** start rewind *********************"
      do
         call ESMF_ClockAdvance(clock,rc=status)
         call verify_(status)
         call ESMF_ClockGet(clock,currTime=time)
         if (ESMF_AlarmIsRinging(esmfalarm)) then
            !write(*,*)'rewinding one step: esmfalarm ',ESMF_AlarmIsRinging(esmfalarm)
            call ESMF_TimePrint(time,options='string')
         end if

         if (ESMF_AlarmIsRinging(esmfalarm)) then
            write(*,*)'rewinding one step: testalarm ',ESMF_AlarmIsRinging(testalarm)
            call ESMF_TimePrint(time,options='string')
         end if
         
         if (time == start_time) exit
      enddo
      call ESMF_ClockSet(clock, direction=ESMF_DIRECTION_FORWARD, rc=status)
      call verify_(status)
      !write(*,*)"*************** end rewind *********************"
      do i=1,nstep*2
         call ESMF_ClockGet(clock,currTime=time)
         esmf_ring = ESMF_AlarmIsRinging(esmfalarm,rc=status)
         call verify_(status)
         if ( esmf_ring ) then
            !write(*,*)'ringing: esmfalarm'
            call ESMF_TimePrint(time,options='string')
         end if
         call ESMF_ClockAdvance(clock)
      enddo

       call ESMF_AlarmDestroy(esmfalarm, rc=status)
       call verify_(status)
       call ESMF_AlarmDestroy(testalarm, rc=status)
       call verify_(status)
       call ESMF_ClockDestroy(clock, rc=status)
       call verify_(status)

       testPass = .true.
      
    end subroutine Test_AlarmHang
#endif

end program ESMF_AlarmTest
