import { jsx as _jsx } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { render } from '../../test-utils/render.js';
import { waitFor } from '../../test-utils/async.js';
import { describe, it, expect, vi, beforeEach } from 'vitest';
import { ApiAuthDialog } from './ApiAuthDialog.js';
import { useKeypress } from '../hooks/useKeypress.js';
import { useTextBuffer, } from '../components/shared/text-buffer.js';
import { clearApiKey } from '@google/gemini-cli-core';
// Mocks
vi.mock('@google/gemini-cli-core', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        clearApiKey: vi.fn().mockResolvedValue(undefined),
    };
});
vi.mock('../hooks/useKeypress.js', () => ({
    useKeypress: vi.fn(),
}));
vi.mock('../components/shared/text-buffer.js', () => ({
    useTextBuffer: vi.fn(),
}));
vi.mock('../contexts/UIStateContext.js', () => ({
    useUIState: vi.fn(() => ({
        terminalWidth: 80,
    })),
}));
const mockedUseKeypress = useKeypress;
const mockedUseTextBuffer = useTextBuffer;
describe('ApiAuthDialog', () => {
    const onSubmit = vi.fn();
    const onCancel = vi.fn();
    let mockBuffer;
    beforeEach(() => {
        vi.clearAllMocks();
        vi.stubEnv('GEMINI_API_KEY', '');
        mockBuffer = {
            text: '',
            lines: [''],
            cursor: [0, 0],
            visualCursor: [0, 0],
            viewportVisualLines: [''],
            handleInput: vi.fn(),
            setText: vi.fn((newText) => {
                mockBuffer.text = newText;
                mockBuffer.viewportVisualLines = [newText];
            }),
        };
        mockedUseTextBuffer.mockReturnValue(mockBuffer);
    });
    it('renders correctly', async () => {
        const { lastFrame, waitUntilReady, unmount } = render(_jsx(ApiAuthDialog, { onSubmit: onSubmit, onCancel: onCancel }));
        await waitUntilReady();
        expect(lastFrame()).toMatchSnapshot();
        unmount();
    });
    it('renders with a defaultValue', async () => {
        const { waitUntilReady, unmount } = render(_jsx(ApiAuthDialog, { onSubmit: onSubmit, onCancel: onCancel, defaultValue: "test-key" }));
        await waitUntilReady();
        expect(mockedUseTextBuffer).toHaveBeenCalledWith(expect.objectContaining({
            initialText: 'test-key',
            viewport: expect.objectContaining({
                height: 4,
            }),
        }));
        unmount();
    });
    it.each([
        {
            keyName: 'return',
            sequence: '\r',
            expectedCall: onSubmit,
            args: ['submitted-key'],
        },
        { keyName: 'escape', sequence: '\u001b', expectedCall: onCancel, args: [] },
    ])('calls $expectedCall.name when $keyName is pressed', async ({ keyName, sequence, expectedCall, args }) => {
        mockBuffer.text = 'submitted-key'; // Set for the onSubmit case
        const { waitUntilReady, unmount } = render(_jsx(ApiAuthDialog, { onSubmit: onSubmit, onCancel: onCancel }));
        await waitUntilReady();
        // calls[0] is the ApiAuthDialog's useKeypress (Ctrl+C handler)
        // calls[1] is the TextInput's useKeypress (typing handler)
        const keypressHandler = mockedUseKeypress.mock.calls[1][0];
        keypressHandler({
            name: keyName,
            shift: false,
            alt: false,
            ctrl: false,
            cmd: false,
            sequence,
        });
        expect(expectedCall).toHaveBeenCalledWith(...args);
        unmount();
    });
    it('displays an error message', async () => {
        const { lastFrame, waitUntilReady, unmount } = render(_jsx(ApiAuthDialog, { onSubmit: onSubmit, onCancel: onCancel, error: "Invalid API Key" }));
        await waitUntilReady();
        expect(lastFrame()).toContain('Invalid API Key');
        unmount();
    });
    it('calls clearApiKey and clears buffer when Ctrl+C is pressed', async () => {
        const { waitUntilReady, unmount } = render(_jsx(ApiAuthDialog, { onSubmit: onSubmit, onCancel: onCancel }));
        await waitUntilReady();
        // Call 0 is ApiAuthDialog (isActive: true)
        // Call 1 is TextInput (isActive: true, priority: true)
        const keypressHandler = mockedUseKeypress.mock.calls[0][0];
        keypressHandler({
            name: 'c',
            shift: false,
            ctrl: true,
            cmd: false,
        });
        await waitFor(() => {
            expect(clearApiKey).toHaveBeenCalled();
            expect(mockBuffer.setText).toHaveBeenCalledWith('');
        });
        unmount();
    });
});
//# sourceMappingURL=ApiAuthDialog.test.js.map