import { jsx as _jsx } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { render } from '../../test-utils/render.js';
import { TerminalProvider, useTerminalContext } from './TerminalContext.js';
import { vi, describe, it, expect } from 'vitest';
import { useEffect, act } from 'react';
import { EventEmitter } from 'node:events';
import { waitFor } from '../../test-utils/async.js';
const mockStdin = new EventEmitter();
// Add required properties for Ink's StdinProps
mockStdin.write = vi.fn();
mockStdin.setEncoding = vi.fn();
mockStdin.setRawMode = vi.fn();
mockStdin.isTTY = true;
// Mock removeListener specifically as it is used in cleanup
mockStdin.removeListener = vi.fn((event, listener) => {
    mockStdin.off(event, listener);
});
vi.mock('ink', async (importOriginal) => {
    const actual = await importOriginal();
    return {
        ...actual,
        useStdin: () => ({
            stdin: mockStdin,
        }),
        useStdout: () => ({
            stdout: {
                write: vi.fn(),
            },
        }),
    };
});
const TestComponent = ({ onColor }) => {
    const { subscribe } = useTerminalContext();
    useEffect(() => {
        subscribe(onColor);
    }, [subscribe, onColor]);
    return null;
};
describe('TerminalContext', () => {
    it('should parse OSC 11 response', async () => {
        const handleColor = vi.fn();
        const { waitUntilReady, unmount } = render(_jsx(TerminalProvider, { children: _jsx(TestComponent, { onColor: handleColor }) }));
        await waitUntilReady();
        await act(async () => {
            mockStdin.emit('data', '\x1b]11;rgb:ffff/ffff/ffff\x1b\\');
        });
        await waitUntilReady();
        await waitFor(() => {
            expect(handleColor).toHaveBeenCalledWith('rgb:ffff/ffff/ffff');
        });
        unmount();
    });
    it('should handle partial chunks', async () => {
        const handleColor = vi.fn();
        const { waitUntilReady, unmount } = render(_jsx(TerminalProvider, { children: _jsx(TestComponent, { onColor: handleColor }) }));
        await waitUntilReady();
        await act(async () => {
            mockStdin.emit('data', '\x1b]11;rgb:0000/');
        });
        await waitUntilReady();
        expect(handleColor).not.toHaveBeenCalled();
        await act(async () => {
            mockStdin.emit('data', '0000/0000\x1b\\');
        });
        await waitUntilReady();
        await waitFor(() => {
            expect(handleColor).toHaveBeenCalledWith('rgb:0000/0000/0000');
        });
        unmount();
    });
});
//# sourceMappingURL=TerminalContext.test.js.map