/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { debugLogger, CoreToolCallStatus, } from '@google/gemini-cli-core';
import {} from '../types.js';
/**
 * Transforms `ToolCall` objects into `HistoryItemToolGroup` objects for UI
 * display. This is a pure projection layer and does not track interaction
 * state.
 */
export function mapToDisplay(toolOrTools, options = {}) {
    const toolCalls = Array.isArray(toolOrTools) ? toolOrTools : [toolOrTools];
    const { borderTop, borderBottom, borderColor, borderDimColor } = options;
    const toolDisplays = toolCalls.map((call) => {
        let description;
        let renderOutputAsMarkdown = false;
        const displayName = call.tool?.displayName ?? call.request.name;
        if (call.status === CoreToolCallStatus.Error) {
            description = JSON.stringify(call.request.args);
        }
        else {
            description = call.invocation.getDescription();
            renderOutputAsMarkdown = call.tool.isOutputMarkdown;
        }
        const baseDisplayProperties = {
            callId: call.request.callId,
            name: displayName,
            description,
            renderOutputAsMarkdown,
        };
        let resultDisplay = undefined;
        let confirmationDetails = undefined;
        let outputFile = undefined;
        let ptyId = undefined;
        let correlationId = undefined;
        let progressMessage = undefined;
        let progress = undefined;
        let progressTotal = undefined;
        switch (call.status) {
            case CoreToolCallStatus.Success:
                resultDisplay = call.response.resultDisplay;
                outputFile = call.response.outputFile;
                break;
            case CoreToolCallStatus.Error:
            case CoreToolCallStatus.Cancelled:
                resultDisplay = call.response.resultDisplay;
                break;
            case CoreToolCallStatus.AwaitingApproval:
                correlationId = call.correlationId;
                // Pass through details. Context handles dispatch (callback vs bus).
                confirmationDetails = call.confirmationDetails;
                break;
            case CoreToolCallStatus.Executing:
                resultDisplay = call.liveOutput;
                ptyId = call.pid;
                progressMessage = call.progressMessage;
                progress = call.progress;
                progressTotal = call.progressTotal;
                break;
            case CoreToolCallStatus.Scheduled:
            case CoreToolCallStatus.Validating:
                break;
            default: {
                const exhaustiveCheck = call;
                debugLogger.warn(`Unhandled tool call status in mapper: ${exhaustiveCheck.status}`);
                break;
            }
        }
        return {
            ...baseDisplayProperties,
            status: call.status,
            resultDisplay,
            confirmationDetails,
            outputFile,
            ptyId,
            correlationId,
            progressMessage,
            progress,
            progressTotal,
            approvalMode: call.approvalMode,
            originalRequestName: call.request.originalRequestName,
        };
    });
    return {
        type: 'tool_group',
        tools: toolDisplays,
        borderTop,
        borderBottom,
        borderColor,
        borderDimColor,
    };
}
//# sourceMappingURL=toolMapping.js.map