/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import type { ConfirmationRequest } from '../types.js';
import type { UseHistoryManagerReturn } from '../hooks/useHistoryManager.js';
type AddItemFn = UseHistoryManagerReturn['addItem'];
export declare const VSCODE_SHIFT_ENTER_SEQUENCE = "\\\r\n";
export interface TerminalSetupResult {
    success: boolean;
    message: string;
    requiresRestart?: boolean;
}
type SupportedTerminal = 'vscode' | 'cursor' | 'windsurf' | 'antigravity';
export declare function getTerminalProgram(): SupportedTerminal | null;
/**
 * Determines whether it is useful to prompt the user to run /terminal-setup
 * in the current environment.
 *
 * Returns true when:
 * - Kitty/modifyOtherKeys keyboard protocol is not already enabled, and
 * - We're running inside a supported terminal (VS Code, Cursor, Windsurf, Antigravity), and
 * - The keybindings file either does not exist or does not already contain both
 *   of our Shift+Enter and Ctrl+Enter bindings.
 */
export declare function shouldPromptForTerminalSetup(): Promise<boolean>;
/**
 * Main terminal setup function that detects and configures the current terminal.
 *
 * This function:
 * 1. Detects the current terminal emulator
 * 2. Applies appropriate configuration for Shift+Enter and Ctrl+Enter support
 * 3. Creates backups of configuration files before modifying them
 *
 * @returns Promise<TerminalSetupResult> Result object with success status and message
 *
 * @example
 * const result = await terminalSetup();
 * if (result.success) {
 *   console.log(result.message);
 *   if (result.requiresRestart) {
 *     console.log('Please restart your terminal');
 *   }
 * }
 */
export declare function terminalSetup(): Promise<TerminalSetupResult>;
export declare const TERMINAL_SETUP_CONSENT_MESSAGE: string;
export declare function formatTerminalSetupResultMessage(result: TerminalSetupResult): string;
interface UseTerminalSetupPromptParams {
    addConfirmUpdateExtensionRequest: (request: ConfirmationRequest) => void;
    addItem: AddItemFn;
}
/**
 * Hook that shows a one-time prompt to run /terminal-setup when it would help.
 */
export declare function useTerminalSetupPrompt({ addConfirmUpdateExtensionRequest, addItem, }: UseTerminalSetupPromptParams): void;
export {};
