/**
 * @license
 * Copyright 2026 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect, vi, beforeEach, afterEach } from 'vitest';
import { BrowserAgentInvocation } from './browserAgentInvocation.js';
import { makeFakeConfig } from '../../test-utils/config.js';
// Mock dependencies before imports
vi.mock('../../utils/debugLogger.js', () => ({
    debugLogger: {
        log: vi.fn(),
        error: vi.fn(),
    },
}));
describe('BrowserAgentInvocation', () => {
    let mockConfig;
    let mockMessageBus;
    let mockParams;
    beforeEach(() => {
        vi.clearAllMocks();
        mockConfig = makeFakeConfig({
            agents: {
                overrides: {
                    browser_agent: {
                        enabled: true,
                    },
                },
                browser: {
                    headless: false,
                    sessionMode: 'isolated',
                },
            },
        });
        mockMessageBus = {
            publish: vi.fn().mockResolvedValue(undefined),
            subscribe: vi.fn(),
            unsubscribe: vi.fn(),
        };
        mockParams = {
            task: 'Navigate to example.com and click the button',
        };
    });
    afterEach(() => {
        vi.restoreAllMocks();
    });
    describe('constructor', () => {
        it('should create invocation with params', () => {
            const invocation = new BrowserAgentInvocation(mockConfig, mockParams, mockMessageBus);
            expect(invocation.params).toEqual(mockParams);
        });
        it('should use browser_agent as default tool name', () => {
            const invocation = new BrowserAgentInvocation(mockConfig, mockParams, mockMessageBus);
            expect(invocation['_toolName']).toBe('browser_agent');
        });
        it('should use custom tool name if provided', () => {
            const invocation = new BrowserAgentInvocation(mockConfig, mockParams, mockMessageBus, 'custom_name', 'Custom Display Name');
            expect(invocation['_toolName']).toBe('custom_name');
            expect(invocation['_toolDisplayName']).toBe('Custom Display Name');
        });
    });
    describe('getDescription', () => {
        it('should return description with input summary', () => {
            const invocation = new BrowserAgentInvocation(mockConfig, mockParams, mockMessageBus);
            const description = invocation.getDescription();
            expect(description).toContain('browser agent');
            expect(description).toContain('task');
        });
        it('should truncate long input values', () => {
            const longParams = {
                task: 'A'.repeat(100),
            };
            const invocation = new BrowserAgentInvocation(mockConfig, longParams, mockMessageBus);
            const description = invocation.getDescription();
            // Should be truncated to max length
            expect(description.length).toBeLessThanOrEqual(200);
        });
    });
    describe('toolLocations', () => {
        it('should return empty array by default', () => {
            const invocation = new BrowserAgentInvocation(mockConfig, mockParams, mockMessageBus);
            const locations = invocation.toolLocations();
            expect(locations).toEqual([]);
        });
    });
});
//# sourceMappingURL=browserAgentInvocation.test.js.map