import { renderRemoteNode } from '../node.mjs';
import { jsxs, Fragment } from 'react/jsx-runtime';

function usePropsForRemoteElement(element, options) {
  if (!element) return undefined;
  const {
    children,
    properties,
    attributes,
    eventListeners
  } = element;
  const reactChildren = [];
  const resolvedProperties = {
    ...properties,
    ...attributes,
    children: reactChildren
  };
  if (options.eventProps) {
    for (const [prop, eventDescription] of Object.entries(options.eventProps)) {
      const eventName = eventDescription?.event;
      if (eventName == null) continue;
      const listener = eventListeners[eventName];
      if (listener) {
        resolvedProperties[prop] = wrapEventListenerForCallback(listener);
      }
    }
  } else {
    // Assume all event listeners are allowed, and write each one as a conventional
    // Preact callback prop.
    for (const [eventName, listener] of Object.entries(eventListeners)) {
      resolvedProperties[`on${eventName[0].toUpperCase()}${eventName.slice(1)}`] = wrapEventListenerForCallback(listener);
    }
  }
  for (const child of children) {
    if (child.type === 1 && typeof child.attributes.slot === 'string') {
      const slot = child.attributes.slot;
      const rendered = renderRemoteNode(child, options);
      resolvedProperties[slot] = resolvedProperties[slot] ? /*#__PURE__*/jsxs(Fragment, {
        children: [resolvedProperties[slot], rendered]
      }) : rendered;
    } else {
      reactChildren.push(renderRemoteNode(child, options));
    }
  }
  return resolvedProperties;
}
function wrapEventListenerForCallback(listener) {
  return function eventListenerCallbackWrapper(...args) {
    if (args.length === 1 && args[0] instanceof Event) {
      const event = args[0];
      if (event.target !== event.currentTarget) return;
      return 'detail' in event ? listener(event.detail) : listener();
    }
    return listener(...args);
  };
}

export { usePropsForRemoteElement };
