import { AbortError } from './AbortError.mjs';

/**
 * Waits for the result of a function that returns a promise, and
 * returns its result. However, this promise is "raced" against the
 * `AbortSignal` you provide, and if it aborts before the promise resolves,
 * the promise returned by this function will be rejected.
 *
 * @param race A function that returns a promise. If this function resolves
 * before the `AbortSignal` aborts, the promise returned by `raceAgainstAbortSignal()`
 * will resolve with the same value.
 */
async function raceAgainstAbortSignal(race, {
  signal,
  onAbort
}) {
  const raceAbort = new AbortController();
  const result = await Promise.race([racer(), abortRacer()]);
  return result;
  async function racer() {
    try {
      const result = await race();
      return result;
    } finally {
      raceAbort.abort();
    }
  }
  async function abortRacer() {
    await new Promise((resolve, reject) => {
      signal.addEventListener('abort', async () => {
        try {
          if (onAbort) await onAbort();
          reject(new AbortError());
        } catch (error) {
          reject(error);
        }
      }, {
        signal: raceAbort.signal
      });
      raceAbort.signal.addEventListener('abort', () => {
        resolve();
      }, {
        signal
      });
    });
  }
}

export { raceAgainstAbortSignal };
