import { Thread, type ThreadOptions } from '../Thread.ts';
/**
 * Creates a thread from a `WebSocket` instance in the browser.
 *
 * @see https://developer.mozilla.org/en-US/docs/Web/API/WebSocket
 *
 * @example
 * import {ThreadBrowserWebSocket} from '@quilted/threads';
 *
 * const websocket = new WebSocket('ws://localhost:8080');
 * const thread = new ThreadBrowserWebSocket(websocket);
 * await thread.imports.sendMessage('Hello world!');
 */
export declare class ThreadBrowserWebSocket<Imports = Record<string, never>, Exports = Record<string, never>> extends Thread<Imports, Exports> {
    readonly socket: WebSocket;
    /**
     * Starts a thread wrapped around a `WebSocket` object, and returns the imports
     * of the thread.
     *
     * @example
     * ```ts
     * import {ThreadBrowserWebSocket} from '@quilted/threads';
     *
     * const websocket = new WebSocket('ws://localhost:8080');
     * const {getMessage} = ThreadBrowserWebSocket.import(websocket);
     * const message = await getMessage(); // 'Hello, world!'
     * ```
     */
    static import<Imports = Record<string, never>>(socket: WebSocket, options?: Omit<ThreadOptions<Imports, Record<string, never>>, 'imports'>): import("../Thread.ts").ThreadImports<Imports>;
    /**
     * Starts a thread wrapped around a `WebSocket` object, providing the second
     * argument as the exports of the thread.
     *
     * @example
     * ```ts
     * import {ThreadBrowserWebSocket} from '@quilted/threads';
     *
     * const websocket = new WebSocket('ws://localhost:8080');
     *
     * ThreadBrowserWebSocket.export(websocket, {
     *   async getMessage() {
     *     return 'Hello, world!';
     *   },
     * });
     * ```
     */
    static export<Exports = Record<string, never>>(socket: WebSocket, exports: Exports, options?: Omit<ThreadOptions<Record<string, never>, Exports>, 'exports'>): void;
    constructor(socket: WebSocket, { serialization, ...options }?: ThreadOptions<Imports, Exports>);
}
//# sourceMappingURL=ThreadBrowserWebSocket.d.ts.map